﻿<#
.SYNOPSIS
	Module synopsis ...
.DESCRIPTION
	Module description ...
.LINK
	http://www.cancom.de
#>
param(
	[hashtable]$InitialContext = $null
)

## throw Write-Error etc.
$ErrorActionPreference = "Stop"

[string]$DeploymentType = $null;
[string]$DeployMode = $null;
[switch]$AllowRebootPassThru = $false;
[switch]$TerminalServerMode = $false;
[switch]$DisableLogging = $false;
[string]$InstallMode = $null;
[string]$UserPartHandler = $null;
[string[]]$AdtExportVariables = @();
[string[]]$AdtExportFunctions = @();

$CurrentPackage = $null;
[string]$CurrentPackageDirectory = $null;
[string]$PackageDeploymentRegistryKeyName = "SOFTWARE\CANCOM\Package Deployment"; # HKLM / HKCU
[string]$InstalledAppsRegistryKeyName = "$($PackageDeploymentRegistryKeyName)\Installed Apps"; # HKLM / HKCU
[string]$SessionManagementRegistryKeyName = "$($PackageDeploymentRegistryKeyName)\Session Management"; # HKLM
[string]$PackageCacheDirectoryName = "CANCOM\PackageDeployment\PackageCache"; # %ProgramData% (Environment.SpecialFolder.CommonApplicationData)

## Get the directory of the current script
If (Test-Path -LiteralPath 'variable:HostInvocation') { $InvocationInfo = $HostInvocation } Else { $InvocationInfo = $MyInvocation }
[string]$scriptDirectory = Split-Path -Path $InvocationInfo.MyCommand.Definition -Parent

$script:validateFileName = { [regex]::Replace($args[0], "[$([regex]::Escape([string]::Join('',([IO.Path]::GetInvalidFileNameChars()))))]", "_") };
if ($InitialContext -ne $null)
{
	##*===============================================
	##* VARIABLE DECLARATION
	##*===============================================
	
	## Variables: Package
	$CurrentPackage = $InitialContext.Package
	if ($CurrentPackage -eq $null) { $CurrentPackage = @{ Name = ""; Vendor = ""; Version = ""; Architecture = ""; Language = ""; Revision = ""; ScriptVersion = ""; ScriptDate = ""; ScriptAuthor = ""; } }
	
	$CurrentPackageDirectory = $InitialContext.PackageDirectory
	if ([string]::IsNullOrEmpty($CurrentPackageDirectory)) { $CurrentPackageDirectory = $scriptDirectory; }

	## Variables: Application
	[string]$appVendor = $CurrentPackage.Vendor
	[string]$appName = $CurrentPackage.Name
	[string]$appVersion = $CurrentPackage.Version
	[string]$appArch = $CurrentPackage.Architecture
	[string]$appLang = $CurrentPackage.Language
	[string]$appRevision = $CurrentPackage.Revision
	[string]$appScriptVersion = $CurrentPackage.ScriptVersion
	[string]$appScriptDate = $CurrentPackage.ScriptDate
	[string]$appScriptAuthor = $CurrentPackage.ScriptAuthor
	##*===============================================
	## Variables: Install Titles (Only set here to override defaults set by the toolkit)
	[string]$installName = $InitialContext.installName
	[string]$installTitle = $InitialContext.installTitle
	
	## Variables: Script
	[string]$deployAppScriptFriendlyName = $InitialContext.deployAppScriptFriendlyName
	[version]$deployAppScriptVersion = [version]$InitialContext.deployAppScriptVersion
	[string]$deployAppScriptDate = $InitialContext.deployAppScriptDate
	[hashtable]$deployAppScriptParameters = $InitialContext.deployAppScriptParameters
	
	## Variables: Script parameters
	$DeploymentType = $InitialContext.DeploymentType
	$DeployMode = $InitialContext.DeployMode
	$AllowRebootPassThru = $InitialContext.AllowRebootPassThru
	$TerminalServerMode = $InitialContext.TerminalServerMode
	$DisableLogging = $InitialContext.DisableLogging
	$InstallMode = $InitialContext.InstallMode
	$UserPartHandler = $InitialContext.UserPartHandler
	
	## Variables: Log
	[string]$logName = (& $script:validateFileName $InitialContext.LogFileName)
	$AdtExportVariables = @($InitialContext.adtVariables | where { ![string]::IsNullOrEmpty($_) } | % { $_ });
	$AdtExportFunctions = @($InitialContext.adtFunctions | where { ![string]::IsNullOrEmpty($_) } | % { $_ });
}

# custom Installed Apps registry key ($InstalledAppsRegistryKeyName)
if ($CurrentPackage -ne $null)
{
	$key = "InstalledAppsRegistryKey";
	$value = ([string]$CurrentPackage.$key).Trim();
	if ([string]::IsNullOrEmpty($value)) { ([string]$CurrentPackage.Parameters.$key).Trim(); }
	if (![string]::IsNullOrEmpty($value)) { $script:InstalledAppsRegistryKeyName = $value; }
}

## Set ExecutionPolicy ByPass for the current process
try { Set-ExecutionPolicy -ExecutionPolicy ByPass -Scope Process -Force -ErrorAction Stop } catch {}

## Dot source the required App Deploy Toolkit Functions
##
## NOTE:
##  The value "True" for the setting "Toolkit_LogWriteToHost" in "AppDeployToolkitConfig.xml" may cause an error when "AppDeployToolkitMain.ps1"
##  runs without console UI. This situation occurs when the script / module is executed via [System.Management.Automation.PowerShell].
##  The "Packaging PowerBench" uses [System.Management.Automation.PowerShell] to load PowerShell modules and thus failw when loading "PackageDeployment.psd1" 
##
##  Symptoms:
##  Error at line 324 [v3.7.0]:
##    [Globalization.CultureInfo]$PrimaryWindowsUILanguage = [Globalization.CultureInfo]($HKULanguages[0])
##    {Der Wert "[06-27-2018 08:43:26.940] [Initialization] [Get-RegistryKey] :: Function Start" kann nicht in den Typ "System.Globalization.CultureInfo" konvertiert werden. ...}
##
##  Reason:
##  In Write-Log: If the -WriteHost switch parameter is set (default: $configToolkitLogWriteToHost -> "Toolkit_LogWriteToHost" is "True")
##  and console UI is not present (checked via $Host.UI.RawUI.ForegroundColor) then log messages are sent to the output stream (Write-Output -InputObject $lTextLogLine).
##  This output from Write-Log is merged with the regular output from the calling functions (e.g. Convert-RegistryPath) and may lead to unexpected behaviour and errors.
##    Here:
##    "[06-27-2018 08:43:26.940] [Initialization] [Get-RegistryKey] :: Function Start" is sent to the output stream by Write-Log, becomes part of the result of Convert-RegistryPath,
##    which is assigned to $HKULanguages and then misinterpreted as [Globalization.CultureInfo] which causes an exception.
##
##  Solution / Workaround:
##  Set "Toolkit_LogWriteToHost" to "False" when using with "Packaging PowerBench".
##
[string]$moduleAppDeployToolkitMain = "$scriptDirectory\AppDeployToolkit\AppDeployToolkitMain.ps1"
If (-not (Test-Path -LiteralPath $moduleAppDeployToolkitMain -PathType 'Leaf')) { Throw "Module does not exist at the specified location [$moduleAppDeployToolkitMain]." }
If ($DisableLogging) { . $moduleAppDeployToolkitMain -DisableLogging } Else { . $moduleAppDeployToolkitMain }
if (!$DisableLogging -and ![string]::IsNullOrEmpty($InitialContext.LogDirectory) -and ($InitialContext.LogDirectory.TrimEnd("\/") -ne $configToolkitLogDir.TrimEnd("\/")))
{
	Write-Log ">>>> Continue logging in $($InitialContext.LogDirectory)\$($logName)";
	$resumeLogging = "<<<< Resumed logging from $($configToolkitLogDir)\$($logName)";
	[string]$configToolkitLogDir = $InitialContext.LogDirectory;
	Write-Log $resumeLogging;
}

## Reversable NI-Commands
<#
	Reversable commands 
	When a package is uninstalled, the following commands are "reversed" automatically:

	+----------------------+------------------------+
	| Command              | Reverse                |
	+======================+========================+
	| Copy                 | Delete                 |
	+----------------------+------------------------+
	| InstallFile          | DeleteFileList         |
	+----------------------+------------------------+
	| InstallFileList      | DeleteFileList         |
	+----------------------+------------------------+
	| MakeDir              | RemoveDir              |
	+----------------------+------------------------+
	| CreateFolder         | RemoveFolder           |
	+----------------------+------------------------+
	| CreateIcon           | RemoveIcon             |
	+----------------------+------------------------+
	| CreateInternetLink   | RemoveLink             |
	+----------------------+------------------------+
	| CreateLink           | RemoveLink             |
	+----------------------+------------------------+
	| AddPrinterConnection | Deletes the connection |
	+----------------------+------------------------+
	| InstallService       | UninstallService       |
	+----------------------+------------------------+
	| MSIInstallProduct    | MSIUninstallProduct    |
	+----------------------+------------------------+
	| RegLoad              | Deletes the keys       |
	|                      | (only if specified)    |
	+----------------------+------------------------+

	The following command is created by NetInstall Spy and can be reversed only under specific conditions:
	+---------+-------------------------------------
	| Command | Action during uninstallation
	+=========+=====================================
	| AddINI  | CONDITIONALLY REVERSIBLE… 
	|         | …if the INI file resides in the Windows directory
	|         | The following rules apply:
	|         | - A section that already exists and has not been created by NetInstall will not be removed during uninstallation.
	|         | - A section that has been created by NetInstall will be removed as soon as the last NetInstall package is uninstalled (i.e. the last package that modified this section, for example by inserting the section, adding value names or changing values)
	|         | Background information: If the INI files resides in the Windows directory the Installer makes use of a reference counter for every section that AddINI modifies.
	|         +-------------------------------------
	|         | REVERSABLE… 
	|         | …if the INI file does not reside in the  Windows directory.
	|         | Please note:
	|         | - New sections that were created by AddIni will be removed (including all values therein)
	|         | - If the INI file is empty after removing specified sections, the complete file will be deleted
	+---------+-------------------------------------
	
#>

## Package deployment context

# $Export-ModuleMember -Variable InstallPhase, InstallName, AppDeployToolkitName, DeploymentType, LogName, AppName, DisableLogging
$script:PdContext = New-Object PSObject
$script:PdContext | Add-Member -NotePropertyMembers @{ IsUninstallScript = $false; DeleteAtEndOfScript = (New-Object PSObject); RebootRequests = 0; StartSystemShutdownParameters = $null; PackageControlsReboot = $false; RegistryCommandsContinueOnError = $true; Status = $null; StatusMessage = $null; ScriptPath = $null; ContinueScript = $null; WmiCache = @{}; }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name InstallMode -Value { $script:InstallMode } -SecondValue { $script:InstallMode = $args[0] }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name InstallPhase -Value { $script:InstallPhase } -SecondValue { $script:InstallPhase = $args[0] }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name InstallName -Value { $script:InstallName } -SecondValue { $script:InstallName = $args[0] }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name DeploymentType -Value { $script:DeploymentType } -SecondValue { $script:DeploymentType = $args[0] }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name DisableLogging -Value { $script:DisableLogging } -SecondValue { $script:DisableLogging = $args[0] }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name AppName -Value { $script:AppName } -SecondValue { $script:AppName = $args[0] }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name AppDeployToolkitName -Value { $script:AppDeployToolkitName }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name LogName -Value { $script:LogName } -SecondValue { $script:LogName = (& $script:validateFileName $args[0]) }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name Package -Value { $script:CurrentPackage }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name PackageDirectory -Value { $script:CurrentPackageDirectory }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name DeployMode -Value { $script:DeployMode }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name UserPartHandler -Value { $script:UserPartHandler }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name SessionZero -Value { $script:SessionZero } # $SessionZero from AppDeployToolkitMain.ps1
$script:PdContext | Add-Member -MemberType ScriptProperty -Name IsLocalSystemAccount -Value { $script:IsLocalSystemAccount } # $IsLocalSystemAccount from AppDeployToolkitMain.ps1
$script:PdContext | Add-Member -NotePropertyMembers @{ IncludeUserPart = $false; UserPartInstallMode = $script:InstallMode; UserIsAdmin = $IsAdmin; User = @{ Process = $ProcessNTAccount; LoggedOn = $CurrentLoggedOnUserSession.NTAccount; Console = $CurrentConsoleUserSession.NTAccount; RunAs = $RunAsActiveUser.NTAccount; } }
$script:PdContext | Add-Member -NotePropertyMembers @{ PSADT = @{ InstallationDeferExitCode = $script:configInstallationDeferExitCode; InstallationUIExitCode = $script:configInstallationUIExitCode; } }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name ShowBalloonNotifications -Value { $script:configShowBalloonNotifications } -SecondValue { $script:configShowBalloonNotifications = [bool]$args[0] }
$script:PdContext | Add-Member -MemberType ScriptProperty -Name InstalledAppsRegistryKey -Value { $script:InstalledAppsRegistryKeyName } -SecondValue { $script:InstalledAppsRegistryKeyName = [string]$args[0] }
$script:PdContext.DeleteAtEndOfScript | Add-Member -NotePropertyMembers @{ Files = @(); Directories = @(); NotEmptyDirectories = @(); }
$script:PdContext.DeleteAtEndOfScript | Add-Member -MemberType ScriptProperty -Name Count -Value { ($this.Files.Count + $this.Directories.Count + $this.NotEmptyDirectories.Count) }
$script:PdContext.DeleteAtEndOfScript | Add-Member -MemberType ScriptMethod -Name Clear -Value { $this.Files = @(); $this.Directories = @(); $this.NotEmptyDirectories = @(); }

function Get-PdContext()
{
	return $script:PdContext;
}

## Tools

$moduleName = "PackageDeployment";
$PsExecPath = [System.IO.Path]::Combine($dirSupportFiles, $(if ($Is64BitProcess) {"PsExec64.exe"} else  {"PsExec.exe"}));
$PsExecServiceName = [string]$script:PdContext.Package.PsExecServiceName;
$UsePaExecExe = $false;
$UsePaExecWow64 = $false;
$PaExecShare = $null;
if (![System.IO.File]::Exists($PsExecPath))
{
	$PaExec64Path = [System.IO.Path]::Combine($dirSupportFiles, "PaExec64.exe");
	$PaExecPath = $(if ($Is64BitProcess -and [System.IO.File]::Exists($PaExec64Path)) {$PaExec64Path} else {[System.IO.Path]::Combine($dirSupportFiles, "PaExec.exe")});
	if ([System.IO.File]::Exists($PaExecPath))
	{
		Write-Log -Message "Located PAEXEC, an open source equivalent to PSEXEC, as '$($PaExecPath)'." -Source $moduleName;
		$UsePaExecExe = $true;
		$PsExecPath = $PaExecPath;
		$UsePaExecWow64 = ($Is64BitProcess -and ($PaExecPath -ne $PaExec64Path));

		$value = [string]$script:PdContext.Package.PaExecShare;
		if (![string]::IsNullOrEmpty($value))
		{
			$name, $path, $computer = @($value.Split("|") | % { $_.Trim("\ ") });

			$localNames = @("localhost", "127.0.0.1", "::1");
			if ([string]::IsNullOrEmpty($computer)) { $computer = $localNames[0]; }

			if ($localNames -notcontains $computer)
			{
				Write-Log -Message "Package property PaExecShare: computer name '$($computer)' not allowed for PAEXEC calls - using '$($localNames[0])' instead." -Source $moduleName -Severity 2;
				$computer = $localNames[0];
			}

			if (![string]::IsNullOrEmpty($name) -and ![string]::IsNullOrEmpty($path))
			{
				$PaExecShare = @{ Name = $name; Path = $path; Computer = $computer; };
			}
			else
			{
				Write-Log -Message "Incomplete value of package property PaExecShare [$($value)] - ignoring for PAEXEC calls." -Source $moduleName -Severity 2;
			}
		}
	}
}
if (![System.IO.File]::Exists($PsExecPath)) { Write-Log -Message "Cannot locate PSEXEC as '$($PsExecPath)'. Some commands like Start-ProgramAs will fail." -Source $moduleName -Severity 2; }

## Helper functions

Add-Type -Path "$($scriptDirectory)\PackageDeployment.dll";
try
{
	[PSPD.CHK]::ValidateLicense();
}
catch
{
	$failed = "License check failed";
	Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source $moduleName;
	throw "$($failed): $($_.Exception.Message)";
}

function Test-RedirectWow64([bool]$Wow64 = $false)
{
	return ($Wow64 -and !([IntPtr]::Size -eq 4)); # provide WOW64 path for 64-Bit process
}

$script:SystemWow64Directory = [PSPD.API]::GetSystemWow64Directory().TrimEnd("\");
$script:System32Directory = [System.Environment]::SystemDirectory.TrimEnd("\");
$script:RedirectWow64Pattern = "^$([regex]::Escape($System32Directory).Replace('\\', '[\\/]+'))(?<path>[\\/]+.*)?`$";
$script:RedirectWow64ExcludesPattern = "^$([regex]::Escape($System32Directory).Replace('\\', '[\\/]+'))[\\/]+(catroot|catroot2|drivers[\\/]+etc|logfiles|spool)(`$|[\\/])";
$script:SystemNativeDirectory = [System.IO.Path]::Combine($env:windir, "Sysnative");
$script:SystemNativePattern = "^$([regex]::Escape($SystemNativeDirectory).Replace('\\', '[\\/]+'))(?<path>[\\/]+.*)?`$";
function Expand-Path([string]$Path, [switch]$Wow64 = $false)
{
	$pdc = Get-PdContext;
	if (![System.IO.Path]::IsPathRooted($Path)) { $Path = [System.IO.Path]::Combine([string]$pdc.PackageDirectory, $Path); }

	# $Path = $ExecutionContext.InvokeCommand.ExpandString($Path);
	# $Path = [Environment]::ExpandEnvironmentVariables($Path);
	$Path = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath($Path);

	if (Test-RedirectWow64 $Wow64)
	{
		if (($Path -match $RedirectWow64Pattern) -and ($Path -notmatch $RedirectWow64ExcludesPattern))
		{
			# %windir%\system32 -> %windir%\SysWOW64
			$wow64Path = "$($SystemWow64Directory)$($Path -replace $RedirectWow64Pattern, '${path}')";
			Write-Log -Message "Redirecting '$($Path)' to WOW64 '$($wow64Path)'." -Source ${CmdletName};
			return $wow64Path;
		}
		elseif ($Path -match $SystemNativePattern)
		{
			# %windir%\Sysnative -> %windir%\system32
			$nativePath = "$($System32Directory)$($Path -replace $SystemNativePattern, '${path}')";
			Write-Log -Message "Redirecting '$($Path)' to native '$($nativePath)'." -Source ${CmdletName};
			return $nativePath;
		}
	}
	
	return $Path;
}

function Test-InUse([string]$path)
{
	if (![System.IO.File]::Exists($path)) { return $false; }
	
	$fs = $null;
	try
	{
		$fs = [System.IO.File]::Open($path, [System.IO.FileMode]::Open, [System.IO.FileAccess]::ReadWrite, [System.IO.FileShare]::None);
	}
	catch
	{
		# $_.Exception.HResult ...
		# -2147024864 (0x80070020) = Der Prozess kann nicht auf die Datei zugreifen, da sie von einem anderen Prozess verwendet wird. (Ausnahme von HRESULT: 0x80070020)
		# ==>      32 (0x00000020) = The process cannot access the file because it is being used by another process. (ERROR_SHARING_VIOLATION)
		# -2147024864 (0x80070021) = Der Prozess kann nicht auf die Datei zugreifen, da ein anderer Prozess einen Teil der Datei gesperrt hat. (Ausnahme von HRESULT: 0x80070021)
		# ==>      33 (0x00000021) = The process cannot access the file because another process has locked a portion of the file. (ERROR_LOCK_VIOLATION)
		# ...
		# e.g. Read-Only Flag:
		# -2147024891 (0x80070005) = Zugriff verweigert (Ausnahme von HRESULT: 0x80070005 (E_ACCESSDENIED))
		# ==>       5 (0x00000005) = Access is denied. (ERROR_ACCESS_DENIED)

		return $true; # assume ...
	}
	finally
	{
		if ($fs -ne $null) { $fs.Close(); }
	}
	
	return $false;
}

function ConvertTo-Version([string]$version = $null, [int]$major = 0, [int]$minor = 0, [int]$build = 0, [int]$revision = 0)
{
	try
	{
		if (![string]::IsNullOrEmpty($version))
		{
			return New-Object System.Version $version;
		}
		else
		{
			return New-Object System.Version $major, $minor, $build, $revision;
		}
	}
	catch
	{
		return $null;
	}
}

function Test-Overwrite([string]$source, [string]$target, [string]$replace)
{
	$result = $false;
	$reason = "?";
	
	switch ($replace)
	{
		"Never" {
			$result = $false;
			$reason = "never replace";
		} # Never
		"Always" {
			$result = $true;
			$reason = "always replace";
		} # Always
		"Confirm" {
			$result = ((Show-InstallationPrompt -Title "Overwrite File" -Message "Overwrite '$($targetPath)'?" -ButtonLeftText "Yes" -ButtonRightText "No") -eq "Yes");
			$reason = "user confirmed: $($result)";
		} # Confirm
		"Older" {
			$sourceInfo = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($source);
			$targetInfo = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($target);
			if (![string]::IsNullOrEmpty($sourceInfo.FileVersion) -and ![string]::IsNullOrEmpty($targetInfo.FileVersion))
			{
				# compare file versions
				$sourceVersion = ConvertTo-Version -version $sourceInfo.FileVersion;
				$targetVersion = ConvertTo-Version -version $targetInfo.FileVersion;
				if (($sourceVersion -eq $null) -or ($targetVersion -eq $null))
				{
					$sourceVersion = ConvertTo-Version -major $sourceInfo.FileMajorPart -minor $sourceInfo.FileMinorPart -build $sourceInfo.FileBuildPart -revision $sourceInfo.FilePrivatePart;
					$targetVersion = ConvertTo-Version -major $targetInfo.FileMajorPart -minor $targetInfo.FileMinorPart -build $targetInfo.FileBuildPart -revision $targetInfo.FilePrivatePart;
				}
				$result = ($sourceVersion -gt $targetVersion);		
				$reason = "replace version $($targetVersion) by $($sourceVersion): $($result)";
			}
			else
			{
				# compare modification dates
				$sourceModified = [System.IO.Directory]::GetLastWriteTimeUtc($source);
				$targetModified = [System.IO.Directory]::GetLastWriteTimeUtc($target);
				$result = ($sourceModified -gt $targetModified);
				$reason = "replace modified '$($targetModified.ToString('s'))' by '$($sourceModified.ToString('s'))': $($result)";
			}
		} # Older
	}

	return $result, $reason;
}

function Request-Reboot()
{
	$pdc = Get-PdContext;
	$pdc.RebootRequests++;
	Write-Log -Message "Requesting reboot. Current requests: $($pdc.RebootRequests)." -Source ${CmdletName};
}

function Clear-DeleteAtEndOfScript()
{
	$pdc = Get-PdContext;
	Write-Log -Message "Clearing DeleteAtEndOfScript lists [$($pdc.DeleteAtEndOfScript.Count)]." -Source ${CmdletName}; # todo: ...
	$pdc.DeleteAtEndOfScript.Clear();
}

function Add-DeleteAtEndOfScript([string[]]$path = @(), [switch]$force = $false)
{
	$files = @($path | where { ![string]::IsNullOrEmpty($_) -and [System.IO.File]::Exists($_) });
	$directories = @($path | where { ![string]::IsNullOrEmpty($_) -and [System.IO.Directory]::Exists($_) });
	$pdc = Get-PdContext;
	$pdc.DeleteAtEndOfScript.Files += $files;
	if ($force)
	{
		$pdc.DeleteAtEndOfScript.NotEmptyDirectories += $directories;
	}
	else
	{
		$pdc.DeleteAtEndOfScript.Directories += $directories;
	}
	Write-Log -Message "Appended to DeleteAtEndOfScript lists [force: $($force)]: $($files.Count + $directories.Count) [total: $($pdc.DeleteAtEndOfScript.Count)]." -Source ${CmdletName};
}

function Invoke-DeleteAtEndOfScript()
{
	$pdc = $null;
	try
	{
		$pdc = Get-PdContext;
		Write-Log -Message "Invoking DeleteAtEndOfScript lists: $($pdc.DeleteAtEndOfScript.Count)." -Source $MyInvocation.MyCommand.Name;
		foreach ($path in $pdc.DeleteAtEndOfScript.Files)
		{
			Uninstall-SingleFile -Path $path -Delete -DeleteInUse;
		}
		foreach ($path in @($pdc.DeleteAtEndOfScript.Directories | sort -Descending -Unique)) # sorted by depth, no duplicates
		{
			Uninstall-SingleDirectory -Path $path;
		}
		foreach ($path in @($pdc.DeleteAtEndOfScript.NotEmptyDirectories | sort -Descending -Unique)) # sorted by depth, no duplicates
		{
			Uninstall-SingleDirectory -Path $path -DeleteNotEmpty;
		}
	}
	catch
	{
		$failed = "$($MyInvocation.MyCommand.Name) failed";
		Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source $MyInvocation.MyCommand.Name;
		throw "$($failed): $($_.Exception.Message)";
	}
	finally
	{
		if ($pdc -ne $null)
		{
			$pdc.DeleteAtEndOfScript.Clear();
		}
	}
}

function Test-Wildcards([string]$text)
{
	return ($text -match '[\*\?]');
}

function Get-TargetReplacement([string]$fileNamePattern)
{
	$match = [regex]::Match($fileNamePattern, '^(?<p>[^\*\?\.]+)?(?<w>(\*+|\?+))?(?<e>\.(\*+|\?+|[^\*\?]+)?)?$');
	if ($match.Success)
	{
		$pattern = "";
		$replacement = "";

		$p = $match.Groups['p'];
		if ($p.Success)
		{
			$pattern += "(.{0,$($p.Length)})";
			$replacement += "$($p.Value)";
		}

		$w = $match.Groups['w'];
		if ($w.Success)
		{
			if ($w.Value[0] -eq '*')
			{
				$pattern += "(?<w>.*)";
				$replacement += '${w}';
			}
			else
			{
				$pattern += "(?<w>.{0,$($w.Length)})(.*)";
				$replacement += '${w}';
			}
		}
		else
		{
			$pattern += "(.*)";
		}

		$e = $match.Groups['e'];
		if ($e.Success -and ($e.Length -gt 1))
		{
			if ($e.Value[1] -eq '*')
			{
				$pattern += "(?<e>\..*)";
				$replacement += '${e}';
			}
			elseif ($e.Value[1] -eq '?')
			{
				$pattern += "(?<e>\..{0,$($e.Length - 1)})(.*)";
				$replacement += '${e}';
			}
			else
			{
				$pattern += "(\..*)";
				$replacement += "$($e.Value)";
			}
		}
		else
		{
			$pattern += "(\..*)";
		}
		
		$pattern = "^$($pattern)`$";
		
		return ([regex]$pattern), $replacement;
	}
	else
	{
		throw "Complex target wildcard pattern '$($fileNamePattern)' is currently not supported.";
	}
}

function Resolve-FileRoute([string]$Path, [string]$Destination, [switch]$DestinationIsDirectory = $false, [switch]$Recurse = $false, [switch]$Wow64 = $false)
{
	function newRoute([string]$source, [string]$target) { return New-Object PSObject -Property @{ Source = $source; Target = $target; }; }
	
	$Path = Expand-Path $Path -Wow64:$Wow64;
	$sourceDir = [System.IO.Path]::GetDirectoryName($Path);
	$sourcePattern = [System.IO.Path]::GetFileName($Path);
	$sourceWildcards = (Test-Wildcards $sourcePattern);
	
	$Destination = Expand-Path $Destination -Wow64:$Wow64;
	$targetDir = [System.IO.Path]::GetDirectoryName($Destination);
	$targetPattern = [System.IO.Path]::GetFileName($Destination);
	$targetWildcards = (Test-Wildcards $targetPattern);
	
	if ($DestinationIsDirectory -or ($sourceWildcards -and !$targetWildcards) -or [System.IO.Directory]::Exists($Destination))
	{
		$targetDir = $Destination;
		$targetPattern = "";
		$targetWildcards = $false;
	}
	elseif ($targetWildcards -and (($targetPattern -eq "*.*") -or ($targetPattern -eq $sourcePattern)))
	{
		$targetPattern = "";
		$targetWildcards = $false;
	}
	
	$targetRegex = $null;
	$targetReplacement = $null;
	if ($targetWildcards) { $targetRegex, $targetReplacement = Get-TargetReplacement $targetPattern; }
	
	if (!$sourceWildcards -and !$Recurse)
	{
		if ($targetRegex -ne $null) { $targetPattern = $targetRegex.Replace($sourcePattern, $targetReplacement); }
		elseif ([string]::IsNullOrEmpty($targetPattern)) { $targetPattern = $sourcePattern; }
		
		return newRoute -source $Path -target ([System.IO.Path]::Combine($targetDir, $targetPattern));
	}
	
	$singleNamesRecurse = ($Recurse -and !$sourceWildcards -and !$targetWildcards -and ![string]::IsNullOrEmpty($targetPattern));
	if ($singleNamesRecurse)
	{
		# assume: -Path (file) -> -Destination (directory) due to -Recurse
		$targetDir = $Destination;
		$targetPattern = "";
	}
	
	$traverse = {param($source, $target)
	
		$routes = @();
		foreach ($file in [System.IO.Directory]::GetFiles($source, $sourcePattern))
		{
			$name = [System.IO.Path]::GetFileName($file);
			if ($targetRegex -ne $null) { $name = $targetRegex.Replace($name, $targetReplacement); }
			
			$routes += newRoute -source $file -target ([System.IO.Path]::Combine($target, $name));
		}
		
		if ($Recurse)
		{
			foreach ($directory in [System.IO.Directory]::GetDirectories($source))
			{
				$name = [System.IO.Path]::GetFileName($directory);
				$routes += @(& $traverse -source $directory -target ([System.IO.Path]::Combine($target, $name)));
			}
		}
		
		return $routes;
	}
	
	$result = @(. $traverse -source $sourceDir -target $targetDir);
	
	if ($singleNamesRecurse -and ($result.Count -eq 1) -and ($result[0].Source -eq $Path))
	{
		# conclusion: -Path (file) -> -Destination (file) despite of -Recurse
		return newRoute -source $Path -target $Destination;
	}
	
	return $result;
}

function Resolve-FilePath([string]$Path, [switch]$Recurse = $false, [switch]$Wow64 = $false, [switch]$MustExist = $false)
{
	$Path = Expand-Path $Path -Wow64:$Wow64;
	$sourceDir = [System.IO.Path]::GetDirectoryName($Path);
	$sourcePattern = [System.IO.Path]::GetFileName($Path);
	$sourceWildcards = (Test-Wildcards $sourcePattern);
	
	if (![System.IO.Directory]::Exists($sourceDir))
	{
		$message = "Directory does not exist: '$($sourceDir)'.";
		if ($MustExist) { throw $message; }
		else { Write-Log -Message $message -Source ${CmdletName}; }

		return @();
	}
	
	if (!$sourceWildcards -and !$Recurse)
	{
		return $Path;
	}
	
	$traverse = {param($source)
	
		$list = @();
		foreach ($file in [System.IO.Directory]::GetFiles($source, $sourcePattern))
		{
			$list += $file;
		}
		
		if ($Recurse)
		{
			foreach ($directory in [System.IO.Directory]::GetDirectories($source))
			{
				$list += @(& $traverse -source $directory);
			}
		}
		
		return $list;
	}
	
	return @(. $traverse -source $sourceDir);
}

$script:Framework64PathPattern = '^(?<a>.*\\Microsoft.NET\\Framework)64(?<b>\\.*)$';
function Invoke-InstallUtil([string]$assemblyPath, [string]$installerClassParameterList, [switch]$uninstall = $false, [switch]$Wow64 = $false)
{
	$installUtil = [System.IO.Path]::Combine([System.Runtime.InteropServices.RuntimeEnvironment]::GetRuntimeDirectory(), "InstallUtil.exe");
	if ((Test-RedirectWow64 $Wow64) -and ($installUtil -match $Framework64PathPattern))
	{
		$assemblyPath = Expand-Path $assemblyPath -Wow64:$Wow64;
		$wow64InstallUtil = ($installUtil -replace $Framework64PathPattern, '${a}${b}'); # ...\Framework64\... -> ...\Framework\...
		Write-Log -Message "Calling 32-Bit '$([System.IO.Path]::GetFileName($installUtil))' at '$($wow64InstallUtil)' (WOW64)." -Source ${CmdletName};
		$installUtil = $wow64InstallUtil;
	}
	if (![System.IO.File]::Exists($installUtil)) { throw "InstallUtil.exe not found at '$($installUtil)'." }

	$argumentList = "/ShowCallStack /LogToConsole=false";
	$logDir = $(If ($configToolkitCompressLogs) { $logTempFolder } Else { $configToolkitLogDir });
	if (![string]::IsNullOrEmpty($logDir)) { $argumentList += " /InstallStateDir=`"$($logDir)`" /Logfile=`"$([System.IO.Path]::Combine($logDir, 'InstallUtil'))_$([System.IO.Path]::GetFileNameWithoutExtension($assemblyPath)).log`""; }
	if (![string]::IsNullOrEmpty($installerClassParameterList)) { $argumentList += " $($installerClassParameterList)"; }
	if ($uninstall) { $argumentList += " /u"; }
	$argumentList += " `"$($assemblyPath)`"";
	
	Write-Log -Message "Using installutil: $($installUtil)" -Source ${CmdletName};
	Write-Log -Message "Using arguments: $($argumentList)" -Source ${CmdletName};
	$process = Start-Process -FilePath $installUtil -ArgumentList $argumentList -NoNewWindow -PassThru -Wait;
	Write-Log -Message "Exit code: $($process.ExitCode)" -Source ${CmdletName};
	
	return $process.ExitCode;
}

function Invoke-RegAsm([string]$assemblyPath, [switch]$unregister = $false, [switch]$Wow64 = $false)
{
	$regAsm = [System.IO.Path]::Combine([System.Runtime.InteropServices.RuntimeEnvironment]::GetRuntimeDirectory(), "regasm.exe");
	if ((Test-RedirectWow64 $Wow64) -and ($regAsm -match $Framework64PathPattern))
	{
		$assemblyPath = Expand-Path $assemblyPath -Wow64:$Wow64;
		$wow64RegAsm = ($regAsm -replace $Framework64PathPattern, '${a}${b}'); # ...\Framework64\... -> ...\Framework\...
		Write-Log -Message "Calling 32-Bit '$([System.IO.Path]::GetFileName($regAsm))' at '$($wow64RegAsm)' (WOW64)." -Source ${CmdletName};
		$regAsm = $wow64RegAsm;
	}
	if (![System.IO.File]::Exists($regAsm)) { throw "regasm.exe not found at '$($regAsm)'." }

	$argumentList = "";
	$argumentList += " `"$($assemblyPath)`"";
	$argumentList += " /nologo /silent";
	if ($unregister) { $argumentList += " /unregister"; }
	
	Write-Log -Message "Using regasm: $($regAsm)" -Source ${CmdletName};
	Write-Log -Message "Using arguments: $($argumentList)" -Source ${CmdletName};
	$process = Start-Process -FilePath $regAsm -ArgumentList $argumentList -NoNewWindow -PassThru -Wait;
	Write-Log -Message "Exit code: $($process.ExitCode)" -Source ${CmdletName};
	
	return $process.ExitCode;
}

function Invoke-AssemblyInstaller([string]$assemblyPath, [string]$installerClassParameterList, [switch]$uninstall = $false)
{
	$installUtil = [System.IO.Path]::Combine([System.Runtime.InteropServices.RuntimeEnvironment]::GetRuntimeDirectory(), "InstallUtil.exe");
	if (![System.IO.File]::Exists($installUtil)) { throw "InstallUtil.exe not found at '$($installUtil)'." }
	
	$argumentList = @("/ShowCallStack", "/LogToConsole=false");
	$logDir = $(If ($configToolkitCompressLogs) { $logTempFolder } Else { $configToolkitLogDir });
	if (![string]::IsNullOrEmpty($logDir)) { $argumentList += @("/InstallStateDir=$($logDir)", "/Logfile=$([System.IO.Path]::Combine($logDir, 'InstallUtil'))_$([System.IO.Path]::GetFileNameWithoutExtension($assemblyPath)).log"); }
	if (![string]::IsNullOrEmpty($installerClassParameterList)) { $argumentList += [PSPD.API]::ConvertToArgs($installerClassParameterList); }
	
	Write-Log -Message "Assembly-installer mode: $(if ($uninstall) {'uninstall'} else {'install'})" -Source ${CmdletName};
	Write-Log -Message "Using arguments: '$([string]::Join(''', ''', $argumentList))'" -Source ${CmdletName};
	if ($uninstall) { [PSPD.API]::UninstallAssembly($assemblyPath, $argumentList); }
	else { [PSPD.API]::InstallAssembly($assemblyPath, $argumentList); }
}

function Invoke-ShutdownExe([int]$Timeout = 0, [string]$Message = $null, [switch]$ForceLogoff = $null, [switch]$Restart = $false, [switch]$Cancel = $false)
{
	$arguments = @();
	if ($Cancel)
	{
		$arguments += "/a";
	}
	else
	{
		$arguments += $(if ($Restart) { "/r" } else { "/s" });
		
		if ($Timeout -gt 0) { $arguments += "/t $($Timeout)"; }
		
		if (![string]::IsNullOrEmpty($Message))
		{
			if ($Message.Length -gt 512) { $Message = $Message.Substring(0, 512); }
			$Message = $Message.Replace('"', '\"');
			$arguments += " /c `"$($Message)`"";
		}
		
		if ($ForceLogoff) { $arguments += "/f"; }

		$arguments += "/d p:4:2"; # Anwendung: Installiert (geplant)
	}
	
	$si = New-Object System.Diagnostics.ProcessStartInfo;
	$si.FileName = "shutdown.exe";
	$si.Arguments = [string]::Join(" ", $arguments);
	$si.CreateNoWindow = $true;
	$si.LoadUserProfile = $false;
	$si.UseShellExecute = $false;
	$si.WindowStyle = "Hidden";
	$si.RedirectStandardOutput = $true;
	$si.RedirectStandardError = $true;

	Write-Log -Message "Calling '$($si.FileName)' with arguments '$($si.Arguments)'." -Source ${CmdletName};
	$p = [System.Diagnostics.Process]::Start($si);
	$p.WaitForExit();
	
	$stderr = ([string]$p.StandardError.ReadToEnd()).Trim()
	$stdout = ([string]$p.StandardOutput.ReadToEnd()).Trim()
	[int]$exitCode = $p.ExitCode;
	if ($exitCode -ne 0)
	{
		if ([string]::IsNullOrEmpty($stderr)) { $stderr = $stdout; }
		throw "'$($si.FileName)' failed with exit code $($p.ExitCode)$(if (![string]::IsNullOrEmpty($stderr)) { ': ' + $stderr }).";
	}
}

function Set-MsCreateDirFlag([string]$directory, [switch]$clear)
{
	try
	{
		$directory = Expand-Path $directory;
		$path = [System.IO.Path]::Combine($directory, "MSCREATE.DIR");
		$exists = [System.IO.File]::Exists($path);
		if ($exists -and $clear)
		{
			[System.IO.File]::Delete($path);
			Write-Log -Message "Deleted '$($path)'." -Source ${CmdletName};
			return $true;
		}
		elseif (!$exists -and !$clear)
		{
			[System.IO.File]::WriteAllBytes($path, @());
			[System.IO.File]::SetAttributes($path, [System.IO.FileAttributes]"Hidden, Archive");
			Write-Log -Message "Created '$($path)'." -Source ${CmdletName};
			return $true;
		}
	}
	catch
	{
		$failed = "Failed to set MsCreateDirFlag at [$directory]";
		Write-Log -Message "$($failed): $($_.Exception.Message)" -Severity 2 -Source ${CmdletName};
	}
	
	return $false;
}

function Test-MsCreateDirFlag([string]$directory)
{
	try
	{
		$directory = Expand-Path $directory;
		$path = [System.IO.Path]::Combine($directory, "MSCREATE.DIR");
		$exists = [System.IO.File]::Exists($path);
		Write-Log -Message "Found '$($path)': $($exists)" -Source ${CmdletName};
		return $exists;
	}
	catch
	{
		$failed = "Failed to test MsCreateDirFlag at [$directory]";
		Write-Log -Message "$($failed): $($_.Exception.Message)" -Severity 2 -Source ${CmdletName};
	}
}

function Test-SkipCommand([string]$Context = $null, [switch]$SupportReverse = $false, [switch]$PreventUninstall = $false)
{
	$pdc = Get-PdContext;
	
	$reverseMode = (($pdc.DeploymentType -eq "Uninstall") -and !$pdc.IsUninstallScript);
	if ($reverseMode -and !$SupportReverse)
	{
		Write-Log -Message "Skipping command [DeploymentType: $($pdc.DeploymentType), IsUninstallScript: $($pdc.IsUninstallScript)]: Not reversible." -Source ${CmdletName};
		return $true;
	}
	elseif ($reverseMode -and $PreventUninstall)
	{
		Write-Log -Message "Skipping command [DeploymentType: $($pdc.DeploymentType), IsUninstallScript: $($pdc.IsUninstallScript)]: -PreventUninstall is set." -Source ${CmdletName};
		return $true;
	}
	
	$skip = $false;
	if ([string]::IsNullOrEmpty($Context) -or ($Context -eq "Any")) { $skip = $false; }
	elseif ($pdc.InstallMode -eq "InstallUserPart") { $skip = (($Context -eq "Computer") -or ($Context -eq "ComputerPerService")); }
	else { $skip = (($Context -eq "User") -or ($Context -eq "UserPerService")); }

	if ($skip -and $pdc.IncludeUserPart -and (($Context -eq "User") -or ($Context -eq "UserPerService")))
	{
		Write-Log -Message "Executing '$($Context)' command in install mode '$($pdc.InstallMode)' due to deploy mode '$($pdc.DeployMode)'." -Source ${CmdletName};
		$skip = $false;
	}

	if ($skip) { Write-Log -Message "Skipping command [InstallMode: $($pdc.InstallMode), CommandPart: $($Context)]: Not in '$($Context)' part." -Source ${CmdletName}; }
	return $skip;
}

function Test-ReverseMode()
{
	$pdc = Get-PdContext;
	$reverseMode = (($pdc.DeploymentType -eq "Uninstall") -and !$pdc.IsUninstallScript);
	if ($reverseMode) { Write-Log -Message "Reverse installation mode detected [DeploymentType: $($pdc.DeploymentType), IsUninstallScript: $($pdc.IsUninstallScript)]." -Source ${CmdletName}; }
	return $reverseMode;
}

function Set-BeginUninstallScript()
{
	$pdc = Get-PdContext;
	if ($pdc.DeploymentType -eq "Uninstall")
	{
		$pdc.IsUninstallScript = $true;
		Write-Log -Message "Uninstallation script part starting [DeploymentType: $($pdc.DeploymentType), InstallMode: $($pdc.InstallMode), IsUninstallScript: $($pdc.IsUninstallScript)]." -Source ${CmdletName};
		return $true;
	}
	else
	{
		$pdc.IsUninstallScript = $false;
		Write-Log -Message "Skipping uninstallation script part [DeploymentType: $($pdc.DeploymentType), InstallMode: $($pdc.InstallMode), IsUninstallScript: $($pdc.IsUninstallScript)]: Installation mode is '$($pdc.InstallMode)'." -Source ${CmdletName};
		return $false;
	}
}

function Install-SingleDirectory {
<#
	.SYNOPSIS
		Installs a Directory
	.DESCRIPTION
		This command installs a directory. Internal use only
	.PARAMETER Path
		Specifies the directory to be installed
	.PARAMETER Recurse
		Subdirectories are considered
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$Path = Expand-Path $Path -Wow64:$Wow64;
			if ([System.IO.Directory]::Exists($Path))
			{
				Write-Log -Message "Skipping to create directory '$($Path)': already exists." -Source ${CmdletName};
				return;
			}
			
			Write-Log -Message "Creating directory '$($Path)'." -Source ${CmdletName};
			$newDirectories = @();
			$directory = $Path;
			do
			{
				if ([System.IO.Directory]::Exists($directory)) { break; }
				$newDirectories += $directory;
				$directory = [System.IO.Path]::GetDirectoryName($directory);

			} while ($Recurse -and ![string]::IsNullOrEmpty($directory));
			
			if (!$Recurse -and ($newDirectories.Count -gt 1))
			{
				throw "Parent directory does not exist and -Recurse not specified.";
			}

			$void = [System.IO.Directory]::CreateDirectory($Path); # recurse
			$newDirectories | % { $void = Set-MsCreateDirFlag $_; }
		}
		catch
		{
			$failed = "Failed to create directory [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Uninstall-SingleDirectory {
	<#
	.SYNOPSIS
		Uninstalls a Directory
	.DESCRIPTION
		This command uninstalls a directory. Internal use only
	.PARAMETER Path
		Specifies the directory to be uninstalled
	.PARAMETER Recurse
		Subdirectories are considered
	.PARAMETER DeleteNotEmpty
		Delete directory even if it is not empty
	.PARAMETER DeleteAtEndOfScript
		Delete only at the end of the script
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][switch]$DeleteNotEmpty = $false,
		[Parameter(Mandatory=$false)][switch]$DeleteAtEndOfScript = $false,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		function testIsEmpty([string]$directory) { return ![System.IO.Directory]::EnumerateFileSystemEntries($directory).GetEnumerator().MoveNext(); }
		
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$Path = Expand-Path $Path -Wow64:$Wow64;
			if (![System.IO.Directory]::Exists($Path))
			{
				Write-Log -Message "Skipping to delete directory '$($Path)': does not exist." -Source ${CmdletName};
				return;
			}
			
			if ($DeleteNotEmpty)
			{
				if ($DeleteAtEndOfScript)
				{
					Write-Log -Message "Delay deletion of directory '$($Path)' until end of script." -Source ${CmdletName};
					Add-DeleteAtEndOfScript -path $Path -force;
				}
				else
				{
					try
					{
						Write-Log -Message "Delete directory '$($Path)' including files and sub-directories." -Source ${CmdletName};
						[System.IO.Directory]::Delete($Path, $true);
					}
					catch
					{
						Write-Log -Message "Failed to delete directory [$Path] now: $($_.Exception.Message)" -Severity 2 -Source ${CmdletName};
						Write-Log -Message "... Prepare to delete in-use/non-empty directory '$($Path)' after reboot." -Source ${CmdletName};
						[PSPD.API]::DeleteFileAfterReboot($Path);
						Request-Reboot;
					}
				}
			}
			elseif ($Recurse)
			{
				Write-Log -Message "Delete empty directory '$($Path)' including empty sub-directories." -Source ${CmdletName};
				$reverseMode = Test-ReverseMode;
				$traverse = {param($directory)
				
					foreach ($subDirectory in [System.IO.Directory]::GetDirectories($directory))
					{
						if (!(& $traverse -directory $subDirectory))
						{
							return $false;
						}
					}
					
					if ($DeleteAtEndOfScript)
					{
						Write-Log -Message "Delay deletion of directory '$($directory)' until end of script." -Source ${CmdletName};
						Add-DeleteAtEndOfScript -path $directory;
						return $true;
					}
					else
					{
						$hadFlag = Set-MsCreateDirFlag $directory -clear;
						if (testIsEmpty $directory)
						{
							Write-Log -Message "Delete empty directory '$($directory)'." -Source ${CmdletName};
							[System.IO.Directory]::Delete($directory);
							return $true;
						}
						elseif ($hadFlag -and $reverseMode)
						{
							Write-Log -Message "Found MsCreateDirFlag in non-empty directory '$($directory)'." -Source ${CmdletName};
							Write-Log -Message "Prepare to delete non-empty directory '$($directory)' after reboot." -Source ${CmdletName};
							[PSPD.API]::DeleteFileAfterReboot($directory);
							Request-Reboot;
						}
						else
						{
							Write-Log -Message "Skipping to delete directory '$($directory)': not empty." -Source ${CmdletName};
							return $false;
						}
					}
				}
				
				if (!(. $traverse -directory $Path))
				{
					Write-Log -Message "Could not delete '$($Path)' all sub-directories: not empty." -Severity 2 -Source ${CmdletName};
				}
			}
			elseif ($DeleteAtEndOfScript)
			{
				Write-Log -Message "Delay deletion of directory '$($Path)' until end of script." -Source ${CmdletName};
				Add-DeleteAtEndOfScript -path $Path;
			}
			else # !($Recurse)
			{
				$hadFlag = Set-MsCreateDirFlag $Path -clear;
				if (testIsEmpty $Path)
				{
					Write-Log -Message "Delete empty directory '$($Path)'." -Source ${CmdletName};
					[System.IO.Directory]::Delete($Path);
				}
				elseif ($hadFlag -and (Test-ReverseMode))
				{
					Write-Log -Message "Found MsCreateDirFlag in non-empty directory '$($Path)'." -Source ${CmdletName};
					Write-Log -Message "Prepare to delete non-empty directory '$($Path)' after reboot." -Source ${CmdletName};
					[PSPD.API]::DeleteFileAfterReboot($Path);
					Request-Reboot;
				}
				else
				{
					Write-Log -Message "Skipping to delete directory '$($Path)': not empty." -Source ${CmdletName};
				}
			}
		}
		catch
		{
			$failed = "Failed to delete directory [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Install-SingleFile {
	<#
	.SYNOPSIS
		Installs a File, DLL or Assembly
	.DESCRIPTION
		This command installs one or more files from a directory. Internal use only
	.PARAMETER Path
		Specifies the files to be installed including the path
	.PARAMETER CopyTo
		Destination path to which the source files are to be copied
	.PARAMETER Replace
		Replacement options
		- Always: Specifies that existing files with the same name are always overwritten.
		- Older: This is the default option and specifies that existing files with the same name are only overwritten if they are older than the file to be copied.
		- Never: Specifies that existing files with the same name are never overwritten.
		- Confirm: Existing files with the same name are only overwritten if the user confirms this.
	.PARAMETER AbortIfLocked
		Abort script if files are locked
	.PARAMETER BreakLock
		Break file lock
	.PARAMETER CreateBackup
		Create backup if files already exist
	.PARAMETER CreateTargetDirectory
		Create target directory if not exists
	.PARAMETER RemoveAllVersions
		Removes all previous versions
	.PARAMETER RegSvr32
		Installs DLL and OCX files
	.PARAMETER RegAsm
		Installs COM Files
	.PARAMETER InstallUtil
		Use InstallUtil
	.PARAMETER InstallUtilInProcess
		Use AssemblyInstaller
	.PARAMETER InstallUtilParameterList
		Parameter list for AssemblyInstaller and InstallUtil
	.PARAMETER GacUtil
		Installs a .Net Library
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	#>
	[CmdletBinding(DefaultParameterSetName="CopyTo")]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false, ParameterSetName="CopyTo")][string]$CopyTo = $null,
		[Parameter(Mandatory=$false, ParameterSetName="CopyTo")][ValidateSet("Always", "Older", "Never", "Confirm")][string]$Replace = "Always",
		[Parameter(Mandatory=$false, ParameterSetName="CopyTo")][switch]$AbortIfLocked = $false,
		[Parameter(Mandatory=$false, ParameterSetName="CopyTo")][switch]$BreakLock = $false,
		[Parameter(Mandatory=$false, ParameterSetName="CopyTo")][switch]$CreateBackup = $false,
		[Parameter(Mandatory=$false, ParameterSetName="CopyTo")][switch]$CreateTargetDirectory = $false,
		[Parameter(Mandatory=$true, ParameterSetName="GAC")][switch]$GacUtil = $false,
		[Parameter(Mandatory=$false, ParameterSetName="GAC")][switch]$RemoveAllVersions = $false,
		[Parameter(Mandatory=$false)][switch]$RegSvr32 = $false,
		[Parameter(Mandatory=$false)][switch]$RegAsm = $false,
		[Parameter(Mandatory=$false)][switch]$InstallUtil = $false,
		[Parameter(Mandatory=$false)][switch]$InstallUtilInProcess = $false,
		[Parameter(Mandatory=$false)][string]$InstallUtilParameterList = "",
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if ($InstallUtilInProcess -and !$InstallUtil) { Write-Log -Message "SET: -InstallUtilInProcess; NOT SET: -InstallUtil." -Source ${CmdletName} -DebugMessage; }
			
			Write-Log -Message "Installing '$($Path)'." -Source ${CmdletName};
			
			$redirectWow64 = Test-RedirectWow64 $Wow64;
			$sourcePath = Expand-Path $Path -Wow64:$Wow64;
			if (![System.IO.File]::Exists($sourcePath))
			{
				throw "Cannot install '$($sourcePath)': file not found.";
			}
			
			$inUse = $false;
			if ($GacUtil)
			{
				Write-Log -Message "Install file '$($sourcePath)' to the global assembly cache." -Source ${CmdletName};
				
				if ($RemoveAllVersions)
				{
					Write-Log -Message "Removing all current versions of assembly '$($sourcePath)' from the global assembly cache." -Source ${CmdletName};
					$result = [PSPD.API]::UninstallCacheAssemblyByPath($sourcePath, $true);
					Write-Log -Message "Removed assembly '$($sourcePath)' from the global assembly cache: $($result)." -Source ${CmdletName};
				}
				
				Write-Log -Message "Copying the assembly '$($sourcePath)' to the global assembly cache." -Source ${CmdletName};
				[PSPD.API]::InstallCacheAssembly($sourcePath);
				$assembly = [System.Reflection.Assembly]::LoadFrom($sourcePath);
				if ($assembly.GlobalAssemblyCache)
				{
					$targetPath = $assembly.Location;
					Write-Log -Message "Location in the global assembly cache: '$($targetPath)'." -Source ${CmdletName};
				}
				else
				{
					throw "Assembly '$($sourcePath)' is not in the global assembly cache [$($assembly.Location)].";
				}
			}
			elseif (![string]::IsNullOrEmpty($CopyTo))
			{
				$targetPath = Expand-Path $CopyTo -Wow64:$Wow64;
				Write-Log -Message "Install file '$($sourcePath)' to '$($targetPath)'." -Source ${CmdletName};
				
				if ($CreateTargetDirectory)
				{
					$directoryName = [System.IO.Path]::GetDirectoryName($targetPath);
					if (![System.IO.Directory]::Exists($directoryName))
					{
						Install-SingleDirectory -Path $directoryName -Recurse -Wow64:$Wow64;
					}
				}
				
				if ([System.IO.File]::Exists($targetPath))
				{
					$overwrite, $reason = Test-Overwrite -source $sourcePath -target $targetPath -replace $Replace;
					if ($overwrite)
					{
						Write-Log -Message "Overwriting target file '$($targetPath)' [$($reason)]." -Source ${CmdletName};
						
						$readOnly = ([System.IO.FileInfo]$targetPath).IsReadOnly;
						if ($readOnly)
						{
							Write-Log -Message "Clear read-only attribute on '$($targetPath)'." -Source ${CmdletName};
							([System.IO.FileInfo]$targetPath).IsReadOnly = $false;
						}
						
						$inUse = (Test-InUse -path $targetPath);
						if ($inUse -and $AbortIfLocked)
						{
							if ($readOnly)
							{
								Write-Log -Message "Restore read-only attribute on '$($targetPath)'." -Source ${CmdletName};
								([System.IO.FileInfo]$targetPath).IsReadOnly = $true;
							}
							
							$message = "Target file '$($targetPath)' is in use and -AbortIfLocked is specified.";
							Write-Log -Message "$($message) - Raising exception now ..." -Severity 2 -Source ${CmdletName};
							throw $message;
						}
						
						if ($CreateBackup)
						{
							$backupTargetPath = "$($targetPath).backup";
							$backupIndex = 0; while ([System.IO.File]::Exists($backupTargetPath)) { $backupIndex++; $backupTargetPath = "$($targetPath).backup$($backupIndex)"; }
							Write-Log -Message "Backup target file '$($targetPath)' to '$($backupTargetPath)'." -Source ${CmdletName};
							[System.IO.File]::Copy($targetPath, $backupTargetPath);
							
							if ($readOnly)
							{
								Write-Log -Message "Set read-only attribute on '$($backupTargetPath)'." -Source ${CmdletName};
								([System.IO.FileInfo]$backupTargetPath).IsReadOnly = $true;
							}
						}
						
						if ($inUse)
						{
							Write-Log -Message "Target file '$($targetPath)' is in use." -Source ${CmdletName};
							
							$tempTargetPath = "$($targetPath).$([System.IO.Path]::GetRandomFileName())";
							Write-Log -Message "Copy '$($sourcePath)' to new '$($tempTargetPath)'." -Source ${CmdletName};
							[System.IO.File]::Copy($sourcePath, $tempTargetPath);
							
							if ($readOnly)
							{
								Write-Log -Message "Set read-only attribute on '$($tempTargetPath)'." -Source ${CmdletName};
								([System.IO.FileInfo]$tempTargetPath).IsReadOnly = $true;
							}
							
							if ($BreakLock)
							{
								Write-Log -Message "Ignoring unsupported parameter -BreakLock (breaking file locks of the Server service)." -Severity 2 -Source ${CmdletName};
							}
							
							Write-Log -Message "Prepare to rename '$($tempTargetPath)' to in-use '$($targetPath)' after reboot." -Source ${CmdletName};
							[PSPD.API]::MoveFileAfterReboot($tempTargetPath, $targetPath);
							Request-Reboot;

						} # ($inUse)
						else
						{
							Write-Log -Message "Copy '$($sourcePath)' to existing '$($targetPath)'." -Source ${CmdletName};
							[System.IO.File]::Copy($sourcePath, $targetPath, $true);
							
							if ($readOnly)
							{
								Write-Log -Message "Restore read-only attribute on '$($targetPath)'." -Source ${CmdletName};
								([System.IO.FileInfo]$targetPath).IsReadOnly = $true;
							}
						} # !($inUse)
					} # ($overwrite)
					else
					{
						Write-Log -Message "Keeping target file '$($targetPath)' [$($reason)]." -Source ${CmdletName};
					}
				}
				else # !([System.IO.File]::Exists($targetPath))
				{
					Write-Log -Message "Copy '$($sourcePath)' to new '$($targetPath)'." -Source ${CmdletName};
					[System.IO.File]::Copy($sourcePath, $targetPath);
				}
				
			}
			else # !($GacUtil) -and (![string]::IsNullOrEmpty($CopyTo))
			{
				$targetPath = $sourcePath;
			}

			if ($RegSvr32)
			{
				$extension = [System.IO.Path]::GetExtension($targetPath);
				if (($extension -eq ".dll") -or ($extension -eq ".ocx"))
				{
					$isComServer = $false;
					try
					{
						$isComServer = [PSPD.API]::IsComServer($targetPath);
					}
					catch
					{
						$failed = "Failed to detect COM server API of [$targetPath]";
						Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 2 -Source ${CmdletName};
						$isComServer = $false; # todo: check 32-/64-Bit?
					}
					
					if ($isComServer)
					{
						Write-Log -Message "Target file '$($targetPath)' has COM server API." -Source ${CmdletName};
						if ($inUse) { throw "Cannot register COM server '$($targetPath)' while target is in use."; } # todo: ...
						
						Write-Log -Message "Registering COM server '$($targetPath)'." -Source ${CmdletName};
						if ($redirectWow64)
						{
							Invoke-RegSvr32 -Operation Register -FileName $targetPath -Wow64:$Wow64;
						}
						else
						{
							[PSPD.API]::RegisterComServer($targetPath)
						}
					}
				}
				else # !(.dll -or .ocx)
				{
					Write-Log -Message "Skipping COM server API check for '$($targetPath)'." -Source ${CmdletName};
				}
			} # ($RegSvr32)

			if ($RegAsm)
			{
				$hasComTypes = $false;
				try
				{
					$hasComTypes = [PSPD.API]::HasComTypes($targetPath);
				}
				catch
				{
					$failed = "Failed to detect COM types in [$targetPath]";
					Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 2 -Source ${CmdletName};
					$hasComTypes = $false; # todo: check 32-/64-Bit?
				}
				
				if ($hasComTypes)
				{
					Write-Log -Message "Assembly '$($targetPath)' has COM types." -Source ${CmdletName};
					if ($inUse) { throw "Cannot register COM types of '$($targetPath)' while target is in use."; } # todo: ...
					
					Write-Log -Message "Registering COM types of '$($targetPath)'." -Source ${CmdletName};
					if ($redirectWow64)
					{
						Invoke-RegAsm -assemblyPath $targetPath -Wow64:$Wow64;
					}
					else
					{
						[PSPD.API]::RegisterComTypes($targetPath)
					}
				}
			} # ($RegAsm)
			
			if ($InstallUtil)
			{
				if ([PSPD.API]::HasInstaller($targetPath))
				{
					Write-Log -Message "Assembly '$($targetPath)' contains installer." -Source ${CmdletName};
					if ($inUse) { throw "Cannot install assembly '$($targetPath)' while target is in use."; } # todo: ...
					
					Write-Log -Message "Installing assembly '$($targetPath)'." -Source ${CmdletName};
					if ($InstallUtilInProcess -and !$redirectWow64)
					{
						Invoke-AssemblyInstaller -assemblyPath $targetPath -installerClassParameterList $InstallUtilParameterList;
					}
					else
					{
						Invoke-InstallUtil -assemblyPath $targetPath -installerClassParameterList $InstallUtilParameterList -Wow64:$Wow64;
					}
				}
				else
				{
					Write-Log -Message "Assembly '$($targetPath)' contains no installer." -Severity 2 -Source ${CmdletName};
				}
			} # ($InstallUtil)
		}
		catch
		{
			$failed = "Failed to install [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Uninstall-SingleFile {
	<#
	.SYNOPSIS
		Uninstalls a File, DLL or Assembly
	.DESCRIPTION
		This command uninstalls one or more files from a directory. Internal use only
	.PARAMETER Path
		Specifies the files to be deleted including the path
	.PARAMETER RegSvr32
		Uninstalls DLL and OCX files
	.PARAMETER RegAsm
		Uninstalls COM Files
	.PARAMETER InstallUtil
		Use InstallUtil
	.PARAMETER InstallUtilInProcess
		Use AssemblyInstaller
	.PARAMETER InstallUtilParameterList
		Parameter list for AssemblyInstaller and InstallUtil
	.PARAMETER GacUtil
		Uninstalls a .Net Library
	.PARAMETER GacUtilAllVersions
		Uninstall all Versions
	.PARAMETER Delete
		Delete file after uninstalling
	.PARAMETER DeleteInUse
		Delete file also when in use (break lock)
	.PARAMETER DeleteAtEndOfScript	
		Deletes the files at the end of the script
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$RegSvr32 = $false,
		[Parameter(Mandatory=$false)][switch]$RegAsm = $false,
		[Parameter(Mandatory=$false)][switch]$InstallUtil = $false,
		[Parameter(Mandatory=$false)][switch]$InstallUtilInProcess = $false,
		[Parameter(Mandatory=$false)][string]$InstallUtilParameterList = "",
		[Parameter(Mandatory=$false)][switch]$GacUtil = $false,
		[Parameter(Mandatory=$false)][switch]$GacUtilAllVersions = $false,
		[Parameter(Mandatory=$false)][switch]$Delete = $false,
		[Parameter(Mandatory=$false)][switch]$DeleteInUse = $false,
		[Parameter(Mandatory=$false)][switch]$DeleteAtEndOfScript = $false,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if ($InstallUtilInProcess -and !$InstallUtil) { Write-Log -Message "SET: -InstallUtilInProcess; NOT SET: -InstallUtil." -Source ${CmdletName} -DebugMessage; }
			if ($GacUtilAllVersions -and !$GacUtil) { Write-Log -Message "SET: -GacUtilAllVersions; NOT SET: -GacUtil." -Source ${CmdletName} -DebugMessage; }
			if (($DeleteInUse -or $DeleteAtEndOfScript) -and !$Delete) { Write-Log -Message "SET: -DeleteInUse or -DeleteAtEndOfScript; NOT SET: -Delete." -Source ${CmdletName} -DebugMessage; }

			Write-Log -Message "Uninstalling '$($Path)'." -Source ${CmdletName};
			
			$redirectWow64 = Test-RedirectWow64 $Wow64;
			$targetPath = Expand-Path $Path -Wow64:$Wow64;
			if (![System.IO.File]::Exists($targetPath))
			{
				if ($GacUtil)
				{
					try
					{
						$assembly = [System.Reflection.Assembly]::LoadWithPartialName($Path);
						$targetPath = $assembly.Location;
						Write-Log -Message "Resolved assembly '$($Path)' to '$($targetPath)'." -Source ${CmdletName};
					}
					catch
					{
						Write-Log -Message "Cannot uninstall '$($Path)': assembly not found." -Source ${CmdletName};
						return;
					}
				}
				else
				{
					Write-Log -Message "Cannot uninstall '$($targetPath)': file not found." -Source ${CmdletName};
					return;
				}
			}
			
			if ($Delete) # check/prepare ...
			{
				$readOnly = ([System.IO.FileInfo]$targetPath).IsReadOnly;
				if ($readOnly)
				{
					Write-Log -Message "Clear read-only attribute on '$($targetPath)'." -Source ${CmdletName};
					([System.IO.FileInfo]$targetPath).IsReadOnly = $false;
				}
				
				$inUse = $false;
				if (!$GacUtil)
				{
					$inUse = (Test-InUse -path $targetPath);
					if ($inUse -and !$DeleteInUse)
					{
						if ($readOnly)
						{
							Write-Log -Message "Restore read-only attribute on '$($targetPath)'." -Source ${CmdletName};
							([System.IO.FileInfo]$targetPath).IsReadOnly = $true;
						}
						
						$message = "File '$($targetPath)' is in use and -DeleteInUse is not specified.";
						Write-Log -Message "$($message) - Raising exception now ..." -Severity 2 -Source ${CmdletName};
						throw $message;
					}
				}
			} # ($Delete)

			if ($RegSvr32)
			{
				$extension = [System.IO.Path]::GetExtension($targetPath);
				if (($extension -eq ".dll") -or ($extension -eq ".ocx"))
				{
					$isComServer = $false;
					try
					{
						$isComServer = [PSPD.API]::IsComServer($targetPath);
					}
					catch
					{
						$failed = "Failed to detect COM server API of [$targetPath]";
						Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 2 -Source ${CmdletName};
						$isComServer = $false; # todo: check 32-/64-Bit?
					}
					
					if ($isComServer)
					{
						Write-Log -Message "Target file '$($targetPath)' has COM server API. Unregistering COM server." -Source ${CmdletName};
						if ($redirectWow64)
						{
							Invoke-RegSvr32 -Operation Unregister -FileName $targetPath -Wow64:$Wow64;
						}
						else
						{
							[PSPD.API]::UnregisterComServer($targetPath);
						}
					}
				}
				else # !(.dll -or .ocx)
				{
					Write-Log -Message "Skipping COM server API check for '$($targetPath)'." -Source ${CmdletName};
				}
			} # ($RegSvr32)

			if ($RegAsm)
			{
				$hasComTypes = $false;
				try
				{
					$hasComTypes = [PSPD.API]::HasComTypes($targetPath);
				}
				catch
				{
					$failed = "Failed to detect COM types in [$targetPath]";
					Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 2 -Source ${CmdletName};
					$hasComTypes = $false; # todo: check 32-/64-Bit?
				}
				
				if ($hasComTypes)
				{
					Write-Log -Message "Assembly '$($targetPath)' has COM types. Unregistering COM types." -Source ${CmdletName};
					if ($redirectWow64)
					{
						Invoke-RegAsm -assemblyPath $targetPath -unregister -Wow64:$Wow64;
					}
					else
					{
						[PSPD.API]::UnregisterComTypes($targetPath);
					}
				}
			} # ($RegAsm)
			
			if ($InstallUtil)
			{
				if ([PSPD.API]::HasInstaller($targetPath))
				{
					Write-Log -Message "Assembly '$($targetPath)' contains installer. Uninstalling assembly." -Source ${CmdletName};
					if ($InstallUtilInProcess -and !$redirectWow64)
					{
						Invoke-AssemblyInstaller -assemblyPath $targetPath -installerClassParameterList $InstallUtilParameterList -uninstall;
					}
					else
					{
						Invoke-InstallUtil -assemblyPath $targetPath -installerClassParameterList $InstallUtilParameterList -uninstall -Wow64:$Wow64;
					}
				}
				else
				{
					Write-Log -Message "Assembly '$($targetPath)' contains no installer." -Severity 2 -Source ${CmdletName};
				}
			} # ($InstallUtil)
			
			if ($GacUtil)
			{
				[bool]$allVersions = $GacUtilAllVersions;
				Write-Log -Message "Uninstall assembly '$($targetPath)' from the global assembly cache [all versions: $($allVersions)]." -Source ${CmdletName};
				$result = [PSPD.API]::UninstallCacheAssemblyByPath($targetPath, $allVersions);
				Write-Log -Message "Removed assembly '$($sourcePath)' from the global assembly cache: $($result)." -Source ${CmdletName};
			}
			
			if ($Delete) # delete ...
			{
				if ($inUse)
				{
					Write-Log -Message "Prepare to delete in-use file '$($targetPath)' after reboot." -Source ${CmdletName};
					[PSPD.API]::DeleteFileAfterReboot($targetPath);
					Request-Reboot;
				}
				elseif ($DeleteAtEndOfScript)
				{
					Write-Log -Message "Delay deletion of file '$($targetPath)' until end of script." -Source ${CmdletName};
					Add-DeleteAtEndOfScript -path $targetPath;
				}
				else
				{
					Write-Log -Message "Delete file '$($targetPath)'." -Source ${CmdletName};
					[System.IO.File]::Delete($targetPath);
				}
			} # ($Delete)
		}
		catch
		{
			$failed = "Failed to uninstall [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

# from MSDN "Delete method of the Win32_BaseService class"
$script:WmiReturnCodes = @{
	0 = "Success [The request was accepted]";
	1 = "Not Supported [The request is not supported]";
	2 = "Access Denied [The user did not have the necessary access]";
	3 = "Dependent Services Running [The service cannot be stopped because other services that are running are dependent on it]";
	4 = "Invalid Service Control [The requested control code is not valid, or it is unacceptable to the service]";
	5 = "Service Cannot Accept Control [The requested control code cannot be sent to the service because the state of the service ( Win32_BaseService State property) is equal to 0, 1, or 2]";
	6 = "Service Not Active [The service has not been started]";
	7 = "Service Request Timeout [The service did not respond to the start request in a timely fashion]";
	8 = "Unknown Failure [Interactive process]";
	9 = "Path Not Found [The directory path to the service executable file was not found]";
	10 = "Service Already Running [The service is already running]";
	11 = "Service Database Locked [The database to add a new service is locked]";
	12 = "Service Dependency Deleted [A dependency on which this service relies has been removed from the system]";
	13 = "Service Dependency Failure [The service failed to find the service needed from a dependent service]";
	14 = "Service Disabled [The service has been disabled from the system]";
	15 = "Service Logon Failure [The service does not have the correct authentication to run on the system]";
	16 = "Service Marked For Deletion [This service is being removed from the system]";
	17 = "Service No Thread [There is no execution thread for the service]";
	18 = "Status Circular Dependency [There are circular dependencies when starting the service]";
	19 = "Status Duplicate Name [There is a service running under the same name]";
	20 = "Status Invalid Name [There are invalid characters in the name of the service]";
	21 = "Status Invalid Parameter [Invalid parameters have been passed to the service]";
	22 = "Status Invalid Service Account [The account which this service is to run under is either invalid or lacks the permissions to run the service]";
	23 = "Status Service Exists [The service exists in the database of services available from the system]";
	24 = "Service Already Paused [The service is currently paused in the system]";
};
function Get-WmiStatusText([int]$StatusCode)
{
	if ($script:WmiReturnCodes.ContainsKey($StatusCode))
	{
		return $script:WmiReturnCodes[$StatusCode];
	}
	else
	{
		return "#$($StatusCode) [Unknown status code]";
	}
}

function Find-Win32Service {
	<#
	.SYNOPSIS
		Get Service Information
	.DESCRIPTION
		Get Service Information. Internal use only
	.PARAMETER Name
		Find-Win32Service -Name ersupext
	.EXAMPLE
		Find-Win32Service
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$escName = $Name.Replace("\", "\\").Replace("'", "\'");
			$found = @(Get-WmiObject -Class Win32_BaseService -Filter "(Name='$($escName)') or (DisplayName='$($escName)')");
			if ($found.Count -gt 1) { $found = @($found | where { $_.Name -eq $Name }); } # prefer Name
			if ($found.Count -gt 1) { throw "Ambiguous service name '$($Name)' ($($found.Count))." }
			
			return $(if ($found.Count -gt 0) { $found[0] } else { $null });
		}
		catch
		{
			$failed = "Failed to find service [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Uninstall-Win32Service_Internal([string]$Name)
{
	$win32Service = Find-Win32Service -Name $Name;
	if ($win32Service -eq $null) { throw "Service '$($Name)' not found."; }
	
	Write-Log -Message "Stopping service '$($Name)'..." -Source ${CmdletName};
	Stop-ServiceAndDependencies -Name $Name -SkipServiceExistsTest -ContinueOnError:$true;

	Write-Log -Message "Uninstall service '$($win32Service.DisplayName)' ($($win32Service.Name)) [$($Name)]." -Source ${CmdletName};
	$result = $win32Service.Delete();
	if ($result -eq $null)
	{
		Write-Log -Message "Result: NULL." -Source ${CmdletName} -Severity 2;
	}
	elseif ($result.ReturnValue -ne $null)
	{
		[int]$statusCode = $result.ReturnValue;
		$statusText = Get-WmiStatusText $statusCode;
		Write-Log -Message "Result: $($statusCode) = '$($statusText)'." -Source ${CmdletName};
		if ($statusCode -ne 0) { throw $statusCode; }
	}
	else
	{
		Write-Log -Message "Result [unexpected format [$($result.GetType().Name)]]: $([string]$result)." -Source ${CmdletName} -Severity 2;
	}
}

$RegistryRootLookup = @{
	HKEY_CURRENT_USER = "CurrentUser";
	HKEY_LOCAL_MACHINE = "LocalMachine";
	HKEY_CLASSES_ROOT = "ClassesRoot";
	HKEY_CURRENT_CONFIG = "CurrentConfig";
	HKEY_USERS = "Users";
	HKEY_PERFORMANCE_DATA = "PerformanceData";
	HKEY_DYN_DATA = "DynData";
	
	HKCR = "ClassesRoot";
	HKCU = "CurrentUser";
	HKLM = "LocalMachine";
	HKU = "Users";
	HKCC = "CurrentConfig";
}

$script:Wow6432NodeName = "Wow6432Node";
$script:RedirectWow6432NodePattern = '^((?<root>HKEY_CLASSES_ROOT)(?<base>)|(?<root>(HKEY_CURRENT_USER|HKEY_LOCAL_MACHINE)[\\]+)(?<base>Software[\\]+Classes))(?<path>[\\]+(?<first>(CLSID|DirectShow|Interface|Media Type|MediaFoundation))([\\]+.*)?)|(?<root>HKEY_LOCAL_MACHINE[\\]+)(?<base>SOFTWARE)(?<path>[\\]+(?<first>(?!Classes)[^\\]+)?([\\]+.*)?)?$';
function Get-PdRegistryKey([string]$Path, [switch]$Create = $false, [switch]$Writable = $false, [switch]$AcceptNull = $false, [switch]$Wow64 = $false)
{
	$match = [regex]::Match($Path,'^(Registry::)?(?<root>[^:\\]+)(:\\+|\\+|:|$)((?<key>.+)|)$', "IgnoreCase");
	if (!$match.Success) { throw "Cannot resolve registry path '$($Path)'."; }
	
	$rootName = $match.Groups["root"].Value;
	$root = ([Microsoft.Win32.Registry]::($RegistryRootLookup[$rootName]));
	if ($root -eq $null) { throw "Cannot resolve registry root '$($rootName)'."; }
	
	$keyName = $match.Groups["key"].Value;
	if ([string]::IsNullOrEmpty($keyName))
	{
		return $root;
	}
	
	if (Test-RedirectWow64 $Wow64)
	{
		$match = [regex]::Match("$($root.Name)\$($keyName)", $RedirectWow6432NodePattern, "IgnoreCase");
		if ($match.Success -and ($match.Groups["first"].Value -ne $Wow6432NodeName))
		{
			$wow64KeyName = ($match.Groups["base"].Value + "\$($Wow6432NodeName)" + $match.Groups["path"].Value).TrimStart("\");
			Write-Log -Message "Redirecting '$($root.Name)\$($keyName)' to WOW64 '$($root.Name)\$($wow64KeyName)'." -Source ${CmdletName};
			$keyName = $wow64KeyName;
		}
	}
	
	$key = $null;
	if ($Create)
	{
		# $key = $root.CreateSubKey($keyName, $Writable); # requires .NET 4.6
		$key = $root.CreateSubKey($keyName); # write access by default ...
		if ($key -eq $null) { throw "Cannot create registry key '$($Path)'."; }
	}
	else
	{
		$key = $root.OpenSubKey($keyName, $Writable);
		if (($key -eq $null) -and !$AcceptNull) { throw "Cannot open registry key '$($Path)'."; }
	}
	
	return $key;
}

function Invoke-RegExe([string]$Operation, [string]$KeyName, [string]$FileName, [string]$TargetKeyName = $null, [switch]$Recurse = $false, [switch]$Wow64 = $false)
{
	$arguments = "";
	switch ($Operation)
	{
		"EXPORT" { $arguments = "$($Operation) `"$($KeyName)`" `"$($FileName)`" /y"; break; } # KeyName FileName [/y] [/reg:32 | /reg:64]
		"IMPORT" { $arguments = "$($Operation) `"$($FileName)`""; break; } # FileName [/reg:32 | /reg:64]
		"SAVE" { $arguments = "$($Operation) `"$($KeyName)`" `"$($FileName)`" /y"; break; } # KeyName FileName [/y] [/reg:32 | /reg:64]
		"RESTORE" { $arguments = "$($Operation) `"$($FileName)`""; break; } # FileName [/reg:32 | /reg:64]
		"LOAD" { $arguments = "$($Operation) `"$($KeyName)`" `"$($FileName)`""; break; } # KeyName FileName [/reg:32 | /reg:64]
		"UNLOAD" { $arguments = "$($Operation) `"$($KeyName)`""; break; } # KeyName
		"COPY" { $arguments = "$($Operation) `"$($KeyName)`" `"$($TargetKeyName)`" $(if ($Recurse) {'/s'}) /f"; break; } # KeyName1 KeyName2 [/s] [/f] [/reg:32 | /reg:64]
		default { throw "Unsupported operation '$($Operation)'."; break; }
	}
	
	$si = New-Object System.Diagnostics.ProcessStartInfo;
	$si.FileName = "reg.exe";
	if (Test-RedirectWow64 $Wow64)
	{
		$wow64FileName = [System.IO.Path]::Combine($SystemWow64Directory, $si.FileName);
		Write-Log -Message "Calling 32-Bit '$($si.FileName)' at '$($wow64FileName)' (WOW64)." -Source ${CmdletName};
		$si.FileName = $wow64FileName;
	}
	$si.Arguments = $arguments;
	$si.CreateNoWindow = $true;
	$si.LoadUserProfile = $false;
	$si.UseShellExecute = $false;
	$si.WindowStyle = "Hidden";
	$si.RedirectStandardOutput = $true;
	$si.RedirectStandardError = $true;

	Write-Log -Message "Calling '$($si.FileName)' with arguments '$($si.Arguments)'." -Source ${CmdletName};
	$p = [System.Diagnostics.Process]::Start($si);
	$p.WaitForExit();
	
	$stderr = ([string]$p.StandardError.ReadToEnd()).Trim()
	$stdout = ([string]$p.StandardOutput.ReadToEnd()).Trim()
	[int]$exitCode = $p.ExitCode;
	if ($exitCode -ne 0)
	{
		if ([string]::IsNullOrEmpty($stderr)) { $stderr = $stdout; }
		throw "'$($si.FileName)' failed with exit code $($p.ExitCode)$(if (![string]::IsNullOrEmpty($stderr)) { ': ' + $stderr }).";
	}
}

function Invoke-PnpUtilExe([string]$Operation, [string]$InfName)
{
	$arguments = "$($Operation) `"$($InfName)`"";
	
	$si = New-Object System.Diagnostics.ProcessStartInfo;
	$si.FileName = "pnputil.exe";
	if (([IntPtr]::Size -eq 4) -and [PSPD.API]::IsWow64())
	{
		$nativeFileName = [System.IO.Path]::Combine($SystemNativeDirectory, $si.FileName);
		Write-Log -Message "Calling native '$($si.FileName)' at '$($nativeFileName)' (WOW64)." -Source ${CmdletName};
		$si.FileName = $nativeFileName;
	}
	$si.Arguments = $arguments;
	$si.CreateNoWindow = $true;
	$si.LoadUserProfile = $false;
	$si.UseShellExecute = $false;
	$si.WindowStyle = "Hidden";
	$si.RedirectStandardOutput = $true;
	$si.RedirectStandardError = $true;

	Write-Log -Message "Calling '$($si.FileName)' with arguments '$($si.Arguments)'." -Source ${CmdletName};
	$p = [System.Diagnostics.Process]::Start($si);
	$p.WaitForExit();
	
	$stderr = ([string]$p.StandardError.ReadToEnd()).Trim();
	$stdout = ([string]$p.StandardOutput.ReadToEnd()).Trim();
	[int]$exitCode = $p.ExitCode;
	if ($exitCode -eq 259) # 259 (0x103) ERROR_NO_MORE_ITEMS  
	{
		Write-Log -Message "'$($si.FileName)' returned exit code $($p.ExitCode) [no more data]." -Severity 2 -Source ${CmdletName};
	}
	elseif ($exitCode -ne 0)
	{
		if ([string]::IsNullOrEmpty($stderr)) { $stderr = $stdout; }
		throw "'$($si.FileName)' failed with exit code $($p.ExitCode)$(if (![string]::IsNullOrEmpty($stderr)) { ': ' + $stderr }).";
	}

	# if (![string]::IsNullOrEmpty($stdout)) { Write-Log -Message "[STDOUT] $($stdout)" -Source ${CmdletName} -DebugMessage; }
}

function Invoke-MsiExecExe([string]$Arguments, [switch]$SecureParameters = $false, [switch]$PassThru = $false, [switch]$ContinueOnError = $false, [switch]$Wow64 = $false)
{
	$si = New-Object System.Diagnostics.ProcessStartInfo;
	$si.FileName = "msiexec.exe";
	if (Test-RedirectWow64 $Wow64)
	{
		$wow64FileName = [System.IO.Path]::Combine($SystemWow64Directory, $si.FileName);
		Write-Log -Message "Calling 32-Bit '$($si.FileName)' at '$($wow64FileName)' (WOW64)." -Source ${CmdletName};
		$si.FileName = $wow64FileName;
	}
	$si.Arguments = $arguments;
	$si.CreateNoWindow = $true;
	$si.LoadUserProfile = $false;
	$si.UseShellExecute = $false;
	$si.WindowStyle = "Hidden";
	$si.RedirectStandardOutput = $true;
	$si.RedirectStandardError = $true;

	$displayArguments = $(if ($SecureParameters) {"-hidden-"} else {$si.Arguments});
	Write-Log -Message "Calling '$($si.FileName)' with arguments '$($displayArguments)'." -Source ${CmdletName};
	$p = [System.Diagnostics.Process]::Start($si);
	$p.WaitForExit();
	
	$stderr = ([string]$p.StandardError.ReadToEnd()).Trim();
	$stdout = ([string]$p.StandardOutput.ReadToEnd()).Trim();
	
	$status = New-Object PSObject -Property @{ ExitCode = [int]$p.ExitCode; Success = $true; Reboot = $false; StatusText = $null; }
	try { $status.StatusText = [PSPD.API]::GetMessageText($p.ExitCode); } catch {}
	
	if ($status.ExitCode -eq 1641) # 1641 = ERROR_SUCCESS_REBOOT_INITIATED
	{
		Write-Log -Message "'$($si.FileName)' returned exit code $($status.ExitCode): Reboot initiated [$($status.StatusText)]." -Source ${CmdletName};
		$status.Reboot = $true;
	}
	elseif ($status.ExitCode -eq 3010) # 3010 = ERROR_SUCCESS_REBOOT_REQUIRED
	{
		Write-Log -Message "'$($si.FileName)' returned exit code $($status.ExitCode): Reboot required [$($status.StatusText)]." -Source ${CmdletName};
		$status.Reboot = $true;
	}
	elseif ($status.ExitCode -ne 0)
	{
		$status.Success = $false;
		$message = "'$($si.FileName)' failed with exit code $($status.ExitCode) [$($status.StatusText)].";
		if ($ContinueOnError) { Write-Log -Message $message -Severity 3 -Source ${CmdletName}; } else { throw $message; }
	}

	if ($PassThru) { return $status;}
}

function Invoke-RegSvr32([ValidateSet("Register", "Unregister", "InstallOnly")][string]$Operation, [string]$FileName, [string]$InstallArgs = $null, [switch]$Wow64 = $false)
{
	$arguments = "";
	switch ($Operation)
	{
		"Register" { $arguments = ("/s" + $(if (![string]::IsNullOrEmpty($InstallArgs)) {" `"/i:$($InstallArgs)`""} else {" /i"}) + " `"$($FileName)`""); break; } # /s /i[:<args>] <dll>
		"Unregister" { $arguments = ("/s /u" + $(if (![string]::IsNullOrEmpty($InstallArgs)) {" `"/i:$($InstallArgs)`""} else {""}) + " `"$($FileName)`""); break; } # /s /u [/i:<args>] <dll>
		"InstallOnly" { $arguments = ("/s /n `"/i:$($InstallArgs)`" `"$($FileName)`""); break; } # /s /n /i:<args> <dll>
		default { throw "Unsupported operation '$($Operation)'."; break; }
	}
	
	$si = New-Object System.Diagnostics.ProcessStartInfo;
	$si.FileName = "regsvr32.exe";
	if (Test-RedirectWow64 $Wow64)
	{
		$wow64FileName = [System.IO.Path]::Combine($SystemWow64Directory, $si.FileName);
		Write-Log -Message "Calling 32-Bit '$($si.FileName)' at '$($wow64FileName)' (WOW64)." -Source ${CmdletName};
		$si.FileName = $wow64FileName;
	}
	$si.Arguments = $arguments;
	$si.CreateNoWindow = $true;
	$si.LoadUserProfile = $false;
	$si.UseShellExecute = $false;
	$si.WindowStyle = "Hidden";
	$si.RedirectStandardOutput = $true;
	$si.RedirectStandardError = $true;

	Write-Log -Message "Calling '$($si.FileName)' with arguments '$($si.Arguments)'." -Source ${CmdletName};
	$p = [System.Diagnostics.Process]::Start($si);
	$p.WaitForExit();
	
	$stderr = ([string]$p.StandardError.ReadToEnd()).Trim()
	$stdout = ([string]$p.StandardOutput.ReadToEnd()).Trim()
	[int]$exitCode = $p.ExitCode;
	if ($exitCode -ne 0)
	{
		if ([string]::IsNullOrEmpty($stderr)) { $stderr = $stdout; }
		throw "'$($si.FileName)' failed with exit code $($p.ExitCode)$(if (![string]::IsNullOrEmpty($stderr)) { ': ' + $stderr }).";
	}
}

function Invoke-DynWow64Script([string]$Path, [switch]$Wow64 = $false)
{
	$content = Get-Content -Raw -Path $Path;
	
	$pattern = '(?<a>^|\s+)-Wow64(?<b>\s+|$)';
	$replace = '${a}-Wow64:$Wow64${b}';
	$content = [regex]::Replace($content,$pattern, $replace, "IgnoreCase, Singleline");
	
	Write-Log -Message "Creating and invoking DynWow64Script from '$($Path)'." -Source ${CmdletName};
	$scriptBlock = [scriptblock]::Create($content);
	$scriptBlock.Invoke();
}

function Expand-PSString([string]$source)
{
	# note: ExpandString has special handling of quotation mark characters ...
	if ($envPSVersionMajor -lt 3) { $source = $source -replace '''', '`'''; } # ' -> `'
	else { $source = $source -replace '((?<e>(^|[^`])(``)*)`")', '${e}"'; } # -> '" -> "

	return $ExecutionContext.InvokeCommand.ExpandString($source);
}

function ConvertTo-PlainText([string]$obfuscated)
{
	if (($obfuscated.Length -lt 2) -or !$obfuscated.StartsWith("=") -or !$obfuscated.EndsWith("="))
	{
		return $obfuscated;
	}

	try
	{
		if (($obfuscated.Length -lt 3) -or !$obfuscated.StartsWith("==") -or !$obfuscated.EndsWith("="))
		{
			return [System.Text.Encoding]::UTF8.GetString([Convert]::FromBase64String($obfuscated.Substring(1, ($obfuscated.Length - 2))));
		}
		
		$obfuscated = $obfuscated.Substring(2, ($obfuscated.Length - 3));
		# return [System.Text.Encoding]::UTF8.GetString([Convert]::FromBase64String($obfuscated));
		$xor, $bytes = [Convert]::FromBase64String($obfuscated);
		$bytes = @($bytes | % { $_ -bxor $xor })
		return [System.Text.Encoding]::UTF8.GetString($bytes);
	}
	catch
	{
		return $obfuscated;
	}
}

function ConvertTo-RegexPattern([string]$simpleFind, [string]$simpleReplace = $null, [switch]$anySpace = $false)
{
	if ($anySpace) { $simpleFind = $simpleFind.Trim(); }
	$escapedFind = [regex]::Escape($simpleFind);
	$matches = [regex]::Matches($escapedFind, "\\[\*\?]");
	$lastIndex = 0;
	$findAnyStringCount = 0;
	$findAnyCharCount = 0;
	$findPattern = "^";
	if ($anySpace) { $findPattern += "(?<lb>\s*)"; }
	for ($i = 0; $i -lt $matches.Count; $i++)
	{
		$match = $matches[$i];
		$findPattern += $escapedFind.Substring($lastIndex, ($match.Index - $lastIndex));
		if ($match.Value -eq "\*")
		{
			$findPattern += "(?<s$($findAnyStringCount)>.*?)";
			$findAnyStringCount++;
		}
		elseif ($match.Value -eq "\?")
		{
			$findPattern += "(?<c$($findAnyCharCount)>.)";
			$findAnyCharCount++;
		}
		else
		{
			$findPattern += $match.Value;
		}
		
		$lastIndex = ($match.Index + $match.Length);
	}
	if ($lastIndex -lt $escapedFind.Length)
	{
		$findPattern += $escapedFind.Substring($lastIndex);
	}
	if ($anySpace) { $findPattern += "(?<tb>\s*)"; }
	$findPattern += "`$";

	if ([string]::IsNullOrEmpty($simpleReplace))
	{
		return $findPattern;
	}

	$escapedReplace = $simpleReplace.Replace("`$", "`$`$");
	$matches = [regex]::Matches($escapedReplace, "[\*\?]");
	$lastIndex = 0;
	$replaceAnyStringCount = 0;
	$replaceAnyCharCount = 0;
	$replacePattern = "";
	# if ($anySpace) { $replacePattern += "`${lb}"; }
	for ($i = 0; $i -lt $matches.Count; $i++)
	{
		$match = $matches[$i];
		$replacePattern += $escapedReplace.Substring($lastIndex, ($match.Index - $lastIndex));
		if (($match.Value -eq "*") -and ($replaceAnyStringCount -lt $findAnyStringCount))
		{
			$replacePattern += "`${s$($replaceAnyStringCount)}";
			$replaceAnyStringCount++;
		}
		elseif (($match.Value -eq "?") -and ($replaceAnyCharCount -lt $findAnyCharCount))
		{
			$replacePattern += "`${c$($replaceAnyCharCount)}";
			$replaceAnyCharCount++;
		}
		else
		{
			$replacePattern += $match.Value;
		}
		
		$lastIndex = ($match.Index + $match.Length);
	}
	if ($lastIndex -lt $escapedReplace.Length)
	{
		$replacePattern += $escapedReplace.Substring($lastIndex);
	}
	# if ($anySpace) { $replacePattern += "`${tb}"; }

	return $findPattern, $replacePattern;
}

function Get-TextWithEncoding([string]$path, [System.Text.Encoding]$defaultEncoding = [System.Text.Encoding]::Default)
{
	$encodings = @("UTF-8", "UTF-16", "UTF-32", "UTF-16BE", "UTF-32BE") | % { [System.Text.Encoding]::GetEncoding($_) };
	
	[byte[]]$bytes = [System.IO.File]::ReadAllBytes($path);
	
	$encoding = $defaultEncoding;
	$offset = 0;
	
	foreach ($item in $encodings)
	{
		[byte[]]$preamble = $item.GetPreamble();
		if (($preamble.Length -gt 0) -and ($bytes.Length -ge $preamble.Length) -and ([string]$bytes[0..($preamble.Length - 1)] -eq [string]$preamble))
		{
			$encoding = $item;
			$offset = $preamble.Length;
			break;
		}
	}

	$text = $encoding.GetString($bytes[($offset)..($bytes.Length - 1)]);

	return New-Object -TypeName PSObject -Property @{ Path = $path; Encoding = $encoding; Text = $text; };;
}

function Get-TextLines([string]$text, [string]$defaultNewLine = "`r`n")
{
	if ($text -eq $null) { $text = ""; }

	$match = [regex]::Match($text, '(\r\r\n|\r\n|\n|\r)');
	$newLine = $(if ($match.Success -and ![string]::IsNullOrEmpty($match.Value)) { $match.Value } else { $defaultNewLine });

	$lines = $text.Split(@("`r`r`n", "`r`n", "`n", "`r"), [System.StringSplitOptions]::None);

	return New-Object -TypeName PSObject -Property @{ NewLine = $newLine; Lines = $lines; };;
}

## NI-Commands

function Set-PdVar {
	<#
	.SYNOPSIS
		Sets Value of a Variable
	.DESCRIPTION
		Sets the value of a variable. Internal use only
	.PARAMETER Name
		Name of the variable
	.PARAMETER Value
		The value
	.PARAMETER DoNotLog
		Dont write variable name and value into the logfile
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-PdVar -Name Counter -Value 5
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-PdVar.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][object]$Value = $null,
		[Parameter(Mandatory=$false)][Alias("L")][switch]$DoNotLog = $false,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		if ($DoNotLog) { return; }
		
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			Set-Variable -Name $Name -Value $Value -Scope Global -Visibility Public
			if ($DoNotLog) { return; }
			Write-Log -Message "Set variable '$($Name)' to value '$($Value)'." -Source ${CmdletName};
		}
		catch
		{
			$failed = "Failed to set variable [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		if ($DoNotLog) { return; }
		
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-PdVar {
	<#
	.SYNOPSIS
		Read Value from Variable
	.DESCRIPTION
		Reads the value of a variable. Internal use only
	.PARAMETER Name
		Name of the variable
	.PARAMETER DoNotLog
		Dont write variable name and value into the logfile
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Get-PdVar -Name Counter -DoNotLog
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][switch]$DoNotLog = $false,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		if ($DoNotLog) { return; }
		
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$value = Get-Variable -Name $Name -Scope Global -ValueOnly;
			if (!$DoNotLog) { Write-Log -Message "Get variable '$($Name)', value: '$($value)'." -Source ${CmdletName}; }
			return $value;
		}
		catch
		{
			$failed = "Failed to get variable [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		if ($DoNotLog) { return; }
		
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Install-FileList {
	<#
	.SYNOPSIS
		Install a list of source files
	.DESCRIPTION
		Use this command to install a list of source files in the same target path. 
		As parameter, the source files and the common target path are entered.
	.PARAMETER FileList
		Specifies the file to be copied including the path.
	.PARAMETER TargetDir
		Specifies the target file including path. If you want to keep the file name, specify only the destination directory
	.PARAMETER Replace
		Replacement options
		- Always: Specifies that existing files with the same name are always overwritten.
		- Older: This is the default option and specifies that existing files with the same name are only overwritten if they are older than the file to be copied.
		- Never: Specifies that existing files with the same name are never overwritten.
		- Confirm: Existing files with the same name are only overwritten if the user confirms this.
	.PARAMETER AbortIfLocked
		Causes the execution of the script to stop if a file to be updated is locked. 
		If the option is not enabled, the script tries to update the locked file after a restart of the computer.
	.PARAMETER CreateBackup
		When files are replaced by the copy operation, a backup copy of the existing files is created.
	.PARAMETER BreakLock
		Break file lock, if file is locked by the server service
	.PARAMETER PreventUninstall	
		Prevents script from executing 'Uninstall-SingleFile" on 'uninstall mode'
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Install-FileList -TargetDir "${env:ProgramFiles}\Foxit Software" -Replace Always -BreakLock -FileList @(".\Files\pddomproxy.dll",'.\Files\config.ini') -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Install-FileList.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string[]]$FileList,
		[Parameter(Mandatory=$true)][string]$TargetDir,
		[Parameter(Mandatory=$false)][ValidateSet("Always", "Older", "Never", "Confirm")][string]$Replace = "Always",
		[Parameter(Mandatory=$false)][Alias("K")][switch]$AbortIfLocked = $false,
		[Parameter(Mandatory=$false)][Alias("F")][switch]$BreakLock = $false,
		[Parameter(Mandatory=$false)][Alias("B")][switch]$CreateBackup = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			$routes = @($FileList | % { Resolve-FileRoute -Path $_ -Destination $TargetDir -DestinationIsDirectory -Wow64:$Wow64 });
			Write-Log -Message "Resolved files: $($routes.Count)." -Source ${CmdletName};
			
			if (Test-ReverseMode)
			{
				if ($routes.Count -eq 0) { return; }
				
				foreach ($route in $routes)
				{
					Uninstall-SingleFile -Path $route.Target -Delete -DeleteInUse -DeleteAtEndOfScript -Wow64:$Wow64;
				}
				
				Uninstall-SingleDirectory -Path ([System.IO.Path]::GetDirectoryName($routes[0].Target)) -DeleteAtEndOfScript -Wow64:$Wow64;
				
				return; # exit from reverse mode
			}
			
			$params = @{
				Replace = $Replace;
				AbortIfLocked = $AbortIfLocked;
				BreakLock = $BreakLock;
				CreateBackup = $CreateBackup;
				ContinueOnError = $ContinueOnError;
				Wow64 = $Wow64;
			}

			$createTargetDirectory = $true;
			foreach ($route in $routes)
			{
				Install-SingleFile -Path $route.Source -CopyTo $route.Target -CreateTargetDirectory:$createTargetDirectory @params;
				$createTargetDirectory = $false; # only first Install-SingleFile
			}
		}
		catch
		{
			$failed = "Failed to copy files [$FileList.Count] to destination [$TargetDir]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Install-File {
	<#
	.SYNOPSIS
		Copies a single source file to a destination directory
	.DESCRIPTION
		This command copies a single source file to a destination directory. The command works identically to the Copy-File command.
	.PARAMETER File
		Specifies the file to be copied including the path.
	.PARAMETER TargetDir
		Specifies the target file including path. If you want to keep the file name, specify only the destination directory
	.PARAMETER Replace
		Replacement options
		- Always: Specifies that existing files with the same name are always overwritten.
		- Older: This is the default option and specifies that existing files with the same name are only overwritten if they are older than the file to be copied.
		- Never: Specifies that existing files with the same name are never overwritten.
		- Confirm: Existing files with the same name are only overwritten if the user confirms this.
	.PARAMETER CreateBackup
		When files are replaced by the copy operation, a backup copy of the existing files is created.
	.PARAMETER BreakLock
		Break file lock, if file is locked by the server service
	.PARAMETER PreventUninstall	
		Prevents script from executing 'Uninstall-SingleFile" on 'uninstall mode'
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Install-File -File '.\Files\config.ini' -TargetDir "${env:ProgramFiles}\Foxit Software\Foxit Reader" -Replace Older -CreateBackup -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Install-File.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$File,
		[Parameter(Mandatory=$true)][string]$TargetDir,
		[Parameter(Mandatory=$false)][ValidateSet("Always", "Older", "Never", "Confirm")][string]$Replace = "Always",
		[Parameter(Mandatory=$false)][Alias("F")][switch]$BreakLock = $false,
		[Parameter(Mandatory=$false)][Alias("B")][switch]$CreateBackup = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			$routes = @(Resolve-FileRoute -Path $File -Destination $TargetDir -Wow64:$Wow64);
			Write-Log -Message "Resolved files: $($routes.Count)." -Source ${CmdletName};
			
			if (Test-ReverseMode)
			{
				if ($routes.Count -eq 0) { return; }
				
				foreach ($route in $routes)
				{
					Uninstall-SingleFile -Path $route.Target -Delete -DeleteInUse -DeleteAtEndOfScript -Wow64:$Wow64;
				}
				
				Uninstall-SingleDirectory -Path ([System.IO.Path]::GetDirectoryName($routes[0].Target)) -DeleteAtEndOfScript -Wow64:$Wow64;
				
				return; # exit from reverse mode
			}
			
			$params = @{
				Replace = $Replace;
				AbortIfLocked = $false;
				BreakLock = $BreakLock;
				CreateBackup = $CreateBackup;
				ContinueOnError = $ContinueOnError;
				Wow64 = $Wow64;
			}

			$createTargetDirectory = $true;
			foreach ($route in $routes)
			{
				Install-SingleFile -Path $route.Source -CopyTo $route.Target -CreateTargetDirectory:$createTargetDirectory @params;
				$createTargetDirectory = $false; # only first Install-SingleFile
			}
		}
		catch
		{
			$failed = "Failed to copy [$File] to destination [$TargetDir]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Copy-File {
	<#
	.SYNOPSIS
		Copy a file
	.DESCRIPTION
		This command copies one or more files from a source directory to the destination directory. 
	.PARAMETER File
		Specifies the files to be copied including the path
	.PARAMETER TargetDir
		Specifies the target file(s) including path. 
		If you want to keep the file name(s), specify only the destination directory
	.PARAMETER Replace
		Replacement options
		- Always: Specifies that existing files with the same name are always overwritten.
		- Older: This is the default option and specifies that existing files with the same name are only overwritten if they are older than the file to be copied.
		- Never: Specifies that existing files with the same name are never overwritten.
		- Confirm: Existing files with the same name are only overwritten if the user confirms this.
	.PARAMETER CreateBackup
		When files are replaced by the copy operation, a backup copy of the existing files is created.
	.PARAMETER Recurse
		Determines whether files in subdirectories should also be copied. This is particularly relevant when using wildcards.
	.PARAMETER PreventUninstall	
		Prevents script from executing 'Uninstall-SingleDirectory" on 'uninstall mode'
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Copy-File -File '.\Files\config.ini' -TargetDir "${env:ProgramFiles}\Foxit Software\Foxit Reader" -Replace Older -CreateBackup -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Copy-File.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$File,
		[Parameter(Mandatory=$true)][string]$TargetDir,
		[Parameter(Mandatory=$false)][ValidateSet("Always", "Older", "Never", "Confirm")][string]$Replace = "Always",
		[Parameter(Mandatory=$false)][Alias("B")][switch]$CreateBackup = $false,
		[Parameter(Mandatory=$false)][Alias("S")][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }

			$path = Expand-Path $File -Wow64:$Wow64;
			if ([System.IO.Directory]::Exists($path))
			{
				$File = [System.IO.Path]::Combine($File, "*");
				Write-Log -Message "The -File parameter value is a directory ($($path)'). Using '$($File)' to copy all contained files." -Source ${CmdletName};
			}
			
			$routes = @(Resolve-FileRoute -Path $File -Destination $TargetDir -Recurse:$Recurse -Wow64:$Wow64);
			Write-Log -Message "Resolved files: $($routes.Count) [recurse: $($Recurse)]." -Source ${CmdletName};
			
			if (Test-ReverseMode)
			{
				$targetDirLookup = @{};
				foreach ($route in $routes)
				{
					$targetDir = [System.IO.Path]::GetDirectoryName($route.Target);
					if (!$targetDirLookup.ContainsKey($targetDir))
					{
						$targetDirLookup[$targetDir] = (Test-MsCreateDirFlag $targetDir);
					}
					
					Uninstall-SingleFile -Path $route.Target -Delete -DeleteInUse -DeleteAtEndOfScript -Wow64:$Wow64;
				}
				
				foreach ($targetDir in @($targetDirLookup.Keys | sort -Descending | % { $_ })) # sorted by depth
				{
					if ($targetDirLookup[$targetDir])
					{
						Uninstall-SingleDirectory -Path $targetDir -DeleteAtEndOfScript -Wow64:$Wow64;
					}
				}
				
				return; # exit from reverse mode
			}
			
			$params = @{
				Replace = $Replace;
				AbortIfLocked = $false;
				BreakLock = $false;
				CreateBackup = $CreateBackup;
				ContinueOnError = $ContinueOnError;
				Wow64 = $Wow64;
			}
			
			$targetDirLookup = @{};
			foreach ($route in $routes)
			{
				$targetDir = [System.IO.Path]::GetDirectoryName($route.Target);
				$createTargetDirectory = !$targetDirLookup.ContainsKey($targetDir); # only first Install-SingleFile per $targetDir
				if ($createTargetDirectory) { $targetDirLookup[$targetDir] = $false; }
				
				Install-SingleFile -Path $route.Source -CopyTo $route.Target -CreateTargetDirectory:$createTargetDirectory @params;
			}
		}
		catch
		{
			$failed = "Failed to copy [$File] to destination [$TargetDir]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-FileList {
	<#
	.SYNOPSIS
		Delete Files
	.DESCRIPTION
		Deletes a list of files from any directory. 
		This command works identically to the Remove-File command, but offers more options 
		when specifying the files to be deleted, since entire lists of files from 
		different directories can be deleted. Variables and placeholders are used in 
		the same way. Subdirectories are not included in Remove-FileList, 
		use the Remove-File command instead.
	.PARAMETER FileList
		String Array with Paths to the file(s) to be deleted. 
	.PARAMETER IncludeInUse
		This option ensures that files that are 
		locked by Windows are removed the next time the system is restarted.
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-FileList -FileList @("${env:SystemDrive}\temp\export.xlsx","${env:USERPROFILE}\Documents\docs\license-fr.rtf") -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-FileList.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][Alias("PathList")][string[]]$FileList,
		[Parameter(Mandatory=$false)][Alias("F")][switch]$IncludeInUse = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }
			
			$filePaths = @($FileList | % { Resolve-FilePath -Path $_ -Wow64:$Wow64 });
			Write-Log -Message "Resolved files: $($filePaths.Count)." -Source ${CmdletName};
			
			$params = @{
				Delete = $true;
				DeleteInUse = $IncludeInUse;
				Wow64 = $Wow64;
			}
			
			foreach ($path in $filePaths)
			{
				Uninstall-SingleFile -Path $path @params;
			}
		}
		catch
		{
			$failed = "Failed to delete files [$FileList.Count]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-File {
	<#
	.SYNOPSIS
		Delete Files
	.DESCRIPTION
		This command can be used to delete files. The operation and syntax are the same 
		as the Windows shell command DEL (using wildcards and environment variables)
	.PARAMETER File
		Path to the file(s) to be deleted.
	.PARAMETER Recurse
		Specifies that files that conform to the 
		schema should also be deleted in subdirectories of the specified path
	.PARAMETER IncludeInUse
		This option ensures that files that are 
		locked by Windows are removed the next time the system is restarted.
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-File -File "${env:windir}\*.log" -IncludeInUse -Recurse
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-File.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][Alias("Path")][string]$File,
		[Parameter(Mandatory=$false)][Alias("S")][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][Alias("F")][switch]$IncludeInUse = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }
			
			$filePaths = @(Resolve-FilePath -Path $File -Recurse:$Recurse -Wow64:$Wow64);
			Write-Log -Message "Resolved files: $($filePaths.Count)." -Source ${CmdletName};
			
			$params = @{
				Delete = $true;
				DeleteInUse = $IncludeInUse;
				Wow64 = $Wow64;
			}
			
			foreach ($path in $filePaths)
			{
				Uninstall-SingleFile -Path $path @params;
			}
			
			if ($Recurse)
			{
				$dirLookup = @{}; $filePaths | % { $dirLookup[[System.IO.Path]::GetDirectoryName($_)] = $_ }
				$baseDir = [System.IO.Path]::GetDirectoryName((Expand-Path $File));
				if ($dirLookup.ContainsKey($baseDir)) { $dirLookup.Remove($baseDir) }
				
				$dirPaths = @($dirLookup.Keys | sort -Descending); # sorted by depth

				Write-Log -Message "Deleting sub-directories: $($dirPaths.Count)." -Source ${CmdletName};
				foreach ($path in $dirPaths)
				{
					Uninstall-SingleDirectory -Path $path -Wow64:$Wow64;
				}
			}
		}
		catch
		{
			$failed = "Failed to delete [$File]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Install-Assembly {
	<#
	.SYNOPSIS
		Install an assembly
	.DESCRIPTION
		Installs a .NET assembly in a target directory or in the Global Assembly Cache (GAC)
	.PARAMETER SourceFile
		Specifies the file to be installed including the path
	.PARAMETER TargetFile
		Specifies the target file(s) including path.
	.PARAMETER InstallIntoGAC
		Enabling this option causes the selected .NET assembly to be installed in the GAC
	.PARAMETER UninstallCurrentVersions
		If the assembly is to be installed in the GAC, 
		this additional option is available. When enabled, 
		existing versions of the selected .NET assembly are uninstalled from the GAC. 
		Only the newly installed version is then available.
	.PARAMETER ExecuteInstallerClass
	.PARAMETER InstallerClassParameterList
		Execute installation with following parameters
		If you want to install the .NET assembly with certain parameters, they must be specified here.
	.PARAMETER Replace
		Overwrite Options
		- Always: This is the default option and specifies that existing files with the same name are always overwritten.
		- Older: This option specifies that existing files with the same name are only overwritten if they are older than the file to be copied.
		- Never: Specifies that existing files with the same name are never overwritten.
		- Confirm: Existing files with the same name are only overwritten if the user confirms this.
	.PARAMETER BreakLock
		Break file lock, if file is locked by server service
		If the existing assembly is currently in use and therefore locked by the server 
		service, existing handles are broken when this option is enabled. 
		This can cause running applications that use the assembly to stop working.
	.PARAMETER CreateBackup
		When files are replaced by the installation, a backup copy of the existing files is created.
	.PARAMETER PreventUninstall
		Prevents the script from executing "Uninstall-Assembly" on 'uninstall mode'
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Install-Assembly -SourceFile '.\Files\Example.dll' -InstallIntoGAC -UninstallCurrentVersions
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Install-Assembly.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$SourceFile,
		[Parameter(Mandatory=$false)][string]$TargetFile, # set CopyTo
		[Parameter(Mandatory=$false)][switch]$InstallIntoGAC = $false, # set GAC
		[Parameter(Mandatory=$false)][switch]$UninstallCurrentVersions = $false, # set GAC
		[Parameter(Mandatory=$false)][switch]$ExecuteInstallerClass = $false,
		[Parameter(Mandatory=$false)][string]$InstallerClassParameterList = "",
		[Parameter(Mandatory=$false)][ValidateSet("Always", "Older", "Never", "Confirm")][string]$Replace = "Always", # set CopyTo
		[Parameter(Mandatory=$false)][Alias("F")][switch]$BreakLock = $false, # set CopyTo
		[Parameter(Mandatory=$false)][Alias("B")][switch]$CreateBackup = $false, # set CopyTo
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			if ($ExecuteInstallerClass -and ![string]::IsNullOrEmpty($InstallerClassParameterList))
			{
				$matches = [regex]::Matches($InstallerClassParameterList, '([/-](?<n>\w+)([:=]("(?<v>[^"]+)"|(?<v>\w+)))?)');
				$normalized = [string]::Join(" ", @($matches | sort {$_.Groups['n'].Value} | % { "/" + $_.Groups['n'].Value.ToLower() + $(if ($_.Groups['v'].Success) { '="' + $_.Groups['v'].Value + '"' }) }))
				if ($InstallerClassParameterList -cne $normalized)
				{
					Write-Log -Message "Normalized (DSM-like) installer parameter list: '$($normalized)'." -Source ${CmdletName};
					$InstallerClassParameterList = $normalized;
				}
			}
			
			$params = @{
				InstallUtil = $ExecuteInstallerClass;
				InstallUtilInProcess = $false;
				InstallUtilParameterList = $InstallerClassParameterList;
				ContinueOnError = $ContinueOnError;
			}
			
			if (Test-ReverseMode)
			{
				if ($InstallIntoGAC)
				{
					$params += @{
						Path = $SourceFile;
						GacUtil = $InstallIntoGAC;
						GacUtilAllVersions = $UninstallCurrentVersions;
					}
				}
				else
				{
					$params += @{
						Path = $TargetFile;
						Delete = $true;
						DeleteInUse = $true;
						DeleteAtEndOfScript = $true;
					}
				}

				Uninstall-SingleFile @params;
				
				if (!$InstallIntoGAC)
				{
					$TargetFile = Expand-Path $TargetFile;
					Uninstall-SingleDirectory -Path ([System.IO.Path]::GetDirectoryName($TargetFile)) -DeleteAtEndOfScript;
				}
				
				return; # exit from reverse mode
			}
			
			if ($InstallIntoGAC)
			{
				$params += @{
					GacUtil = $InstallIntoGAC;
					RemoveAllVersions= $UninstallCurrentVersions;
				}
			}
			else
			{
				$params += @{
					CopyTo = $TargetFile;
					Replace = $Replace;
					AbortIfLocked = $false;
					BreakLock = $BreakLock;
					CreateBackup = $CreateBackup;
					CreateTargetDirectory = $true;
				}
			}
			
			Install-SingleFile -Path $SourceFile @params;
		}
		catch
		{
			$failed = "Failed to install assembly [$SourceFile]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Uninstall-GacAssembly {
	<#
	.SYNOPSIS
		Delete an assembly
	.DESCRIPTION
		This command deletes an assembly from the Global Assembly Cache (GAC) of the computer on which it is executed.
	.PARAMETER Name
		Select the assembly to be deleted
	.PARAMETER UninstallVersions
		Uninstall options
		- All: All versions of the specified assembly installed in the Global Assembly Cache are uninstalled
		- Specific: Deletes only the specified version or language of the assembly selected above. 
	.PARAMETER Version
		Specifies a version number of the assembly to be deleted.
	.PARAMETER Culture
		Specifies the culture of the assembly to be deleted.
	.PARAMETER PublicKeyToken
		Unique key that identifies the assembly to be deleted.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Uninstall-GacAssembly -Name BouncyCastle.Crypto -UninstallVersions All
	.EXAMPLE
		Uninstall-GacAssembly -Name BouncyCastle.Crypto -UninstallVersions Specific -Version '1.8.1.0' -Culture 'neutral' -PublicKeyToken '0e99375e54769942'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Uninstall-GacAssembly.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][ValidateSet("All", "Specific")][string]$UninstallVersions = $null,
		[Parameter(Mandatory=$false)][string]$Version = "", # set "Specific"
		[Parameter(Mandatory=$false)][string]$Culture = "", # set "Specific"
		[Parameter(Mandatory=$false)][string]$PublicKeyToken = "", # set "Specific"
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }
			
			[bool]$allVersions = ($UninstallVersions -eq "All");
			if (!$allVersions -and ((Test-Wildcards $Version) -or (Test-Wildcards $Culture) -or (Test-Wildcards $PublicKeyToken)))
			{
				throw "Assembly name wildcard pattern 'Version=$($Version), Culture=$($Culture), PublicKeyToken=$($PublicKeyToken)' is currently not supported.";
			}
			
			Write-Log -Message "Uninstall assembly '$($Name)' from the global assembly cache [all versions: $($allVersions)]." -Source ${CmdletName};
			$result = [PSPD.API]::UninstallCacheAssembly($Name, $allVersions, $Version, $Culture, $PublicKeyToken);
			Write-Log -Message "Removed the assembly '$($Name)' from the global assembly cache: $($result)." -Source ${CmdletName};
		}
		catch
		{
			$failed = "Failed to uninstall assembly [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function New-Directory {
	<#
	.SYNOPSIS
		Create Directory
	.DESCRIPTION
		The command creates the specified directory or directory structure.
	.PARAMETER Path
		Path to the directory
	.PARAMETER PreventUninstall
		Prevents from calling Remove-Directory in 'uninstall mode'
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		New-Directory -Path "${env:temp}\demo"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/New-Directory.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			if (Test-ReverseMode)
			{
				Uninstall-SingleDirectory -Path $Path -Recurse -DeleteAtEndOfScript -Wow64:$Wow64;
				return; # exit from reverse mode
			}
			
			Install-SingleDirectory -Path $Path -Recurse -Wow64:$Wow64;
		}
		catch
		{
			$failed = "Failed to create directory [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-Directory {
	<#
	.SYNOPSIS
		Remove Directory
	.DESCRIPTION
		Removes the specified directory if it does not contain any files. 
		If the directory is not empty, it is not deleted and no error message is displayed when the installation package is executed.
	.PARAMETER Path
		Path to the directory to be deleted
	.PARAMETER DeleteNotEmpty
		Also delete directory if its not empty
	.PARAMETER Recurse
		Subdirectories are also removed if they do not contain files.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-Directory -Path "${env:temp}\demo" -Recurse -DeleteNotEmpty
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-Directory.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][Alias("S")][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][switch]$DeleteNotEmpty = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if ($DeleteNotEmpty -and $Recurse) { Write-Log -Message "SET: -DeleteNotEmpty; AND SET: -Recurse." -Source ${CmdletName} -DebugMessage; }
			
			$params = @{
				Path = $Path;
				Recurse = $Recurse;
				DeleteNotEmpty = $DeleteNotEmpty;
				Wow64 = $Wow64;
			}

			Uninstall-SingleDirectory @params;
		}
		catch
		{
			$failed = "Failed to delete directory [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Install-Win32Service {
	<#
	.SYNOPSIS
		Install Service
	.DESCRIPTION
		Use this command to install a service or device for Windows.
	.PARAMETER Name
		The name of the service in the service database
	.PARAMETER DisplayName
		The descriptive name of the service. This name is displayed in the Services dialog in the Control Panel, for example.
	.PARAMETER PathName
		Full path and file name of the executable file associated with the service.
	.PARAMETER StartName
		Allows you to assign a specific user account to a service. The specification is made according to the syntax Domäne\UserID. 
	.PARAMETER StartPassword
		The password for the user account must be entered twice. Note that passwords are case-sensitive.
	.PARAMETER LoadOrderGroupAndDependencies
		Declares dependencies and groups
	.PARAMETER ServiceTypeAndErrorControl
		A bit pattern that specifies the type of service. Possible values are:
		1: SERVICE_KERNEL_DRIVER
		2: SERVICE_FILE_SYSTEM_DRIVER
		4: SERVICE_ADAPTER
		8: SERVICE_RECOGNIZER_DRIVER
		16: SERVICE_WIN32_OWN_PROCESS
		32:	SERVICE_WIN32_SHARE_PROCESS
		If you specify either 16 or 32, and the service runs under the Local System account, you can specify the following additional value by adding it: 256 (SERVICE_INTERACTIVE_PROCESS). Example: 16+256=272
	.PARAMETER StartMode
		To change the way the service starts, select one of the following options:
		- Automatic: Starts the service automatically at system startup after the services of the startup type "Boot" and "System".
		- System: Starts the service on system startup after the services of the "Boot" startup type.
		- Manual: Allows a user or a dependent service to start the service.
		- Disabled: Prevents a user from starting the service. However, it can be started by the system
		- Boot: Starts the service when the computer is turned on.
		- AutomaticDelayed: Automatically starts the service at system startup 120 seconds after the last service with startup type "Automatic" was started.
	.PARAMETER DesktopInteract
		Allow interactive relation to the desktop
	.PARAMETER StartService
		This will start the service immediately after installation. Using the Start-Win32Service command is then superfluous.
	.PARAMETER PreventUninstall
		Prevents uninstalling service in uninstall mode.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Install-Win32Service -Name RasMan -DisplayName '@%Systemroot%\system32\rasmans.dll,-200' -PathName 'C:\Windows\System32\svchost.exe -k netsvcs' -LoadOrderGroupAndDependencies '||SstpSvc;DnsCache' -ServiceTypeAndErrorControl "32,1" -StartMode Automatic -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Install-Win32Service.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][string]$DisplayName = $null,
		[Parameter(Mandatory=$false)][string]$PathName = $null,
		[Parameter(Mandatory=$false)][string]$LoadOrderGroupAndDependencies = $null,
		[Parameter(Mandatory=$false)][string]$StartName = $null,
		[Parameter(Mandatory=$false)][string]$StartPassword = $null,
		[Parameter(Mandatory=$false)][string]$ServiceTypeAndErrorControl = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Boot", "System", "Automatic", "Manual", "Disabled", "AutomaticDelayed")][string]$StartMode = "Automatic",
		[Parameter(Mandatory=$false)][switch]$DesktopInteract = $false,
		[Parameter(Mandatory=$false)][switch]$StartService = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			if (Test-ReverseMode)
			{
				Uninstall-Win32Service_Internal -Name $Name;
				return; # exit from reverse mode
			}

			$win32Service = Find-Win32Service -Name $Name;
			if ($win32Service -ne $null) { throw "Service '$($Name)' already exists: '$($win32Service.DisplayName)' [$($win32Service.PathName)]."; }

			if ([string]::IsNullOrEmpty($DisplayName)) { $DisplayName = $Name }
			if ([string]::IsNullOrEmpty($PathName)) { $PathName = $Name }
			
			# format -LoadOrderGroupAndDependencies: "<LoadOrderGroup>|<LoadOrderGroupDependencies>|<ServiceDependencies>"
			# <LoadOrderGroupDependencies> = <LoadOrderGroupDependency>[;<LoadOrderGroupDependency>]...
			# <ServiceDependencies> = <ServiceDependency>[;<ServiceDependency>]...
			[string]$LoadOrderGroup = $null;
			[string[]]$LoadOrderGroupDependencies = $null;
			[string[]]$ServiceDependencies = $null;
			if (![string]::IsNullOrEmpty($LoadOrderGroupAndDependencies))
			{
				$parts = $LoadOrderGroupAndDependencies.Split("|");
				if ($parts.Count -gt 0) { $LoadOrderGroup = $parts[0].Trim(); }
				if ($parts.Count -gt 1) { $LoadOrderGroupDependencies = @($parts[1].Split(";", [System.StringSplitOptions]::RemoveEmptyEntries) | % { $_.Trim() }); }
				if ($parts.Count -gt 2) { $ServiceDependencies = @($parts[2].Split(";", [System.StringSplitOptions]::RemoveEmptyEntries) | % { $_.Trim() }); }
			}
			
			# format -ServiceTypeAndErrorControl: "<ServiceType>, <ErrorControl>"
			#
			#  <ServiceType>:
			#     1 (0x1) Kernel Driver
			#     2 (0x2) File System Driver
			#     4 (0x4) Adapter
			#     8 (0x8) Recognizer Driver
			#    16 (0x10) Own Process
			#    32 (0x20) Share Process
			#   256 (0x100) Interactive Process
			#
			#  <ErrorControl>:
			#   0 User is not notified.
			#   1 User is notified.
			#   2 System is restarted with the last-known-good configuration.
			#   3 System attempts to start with a good configuration.
			[int]$ServiceType = 16;
			[int]$ErrorControl = 1;
			if (![string]::IsNullOrEmpty($ServiceTypeAndErrorControl))
			{
				$match = [regex]::Match($ServiceTypeAndErrorControl, '^\s*(?<ServiceType>\d+)(\s*,\s*(?<ErrorControl>(\d+|)))?\s*$');
				if (!$match.Success) { throw "Invalid argument for parameter -ServiceTypeAndErrorControl: '$($ServiceTypeAndErrorControl)'."; }
				$ServiceType = [int]$match.Groups["ServiceType"].Value;
				$ErrorControl = [int]$match.Groups["ErrorControl"].Value;
			}
			
			[bool]$DelayedAutoStart = $false;
			if ($StartMode -eq "AutomaticDelayed")
			{
				$DelayedAutoStart = $true;
				$StartMode = "Automatic";
				Write-Log -Message "Creating the '$($Name)' service with StartMode '$($StartMode)' - setting DelayedAutoStart later." -Source ${CmdletName};
			}

			Write-Log -Message "Install service '$($Name)' - Parameters:" -Source ${CmdletName};
			$parameters = @{
				Name = [string]$Name;
				DisplayName = [string]$DisplayName;
				PathName = [string]$PathName;
				ServiceType = [byte]$ServiceType;
				ErrorControl = [byte]$ErrorControl;
				StartMode = [string]$StartMode;
				DesktopInteract = [bool]$DesktopInteract;
				StartName = $(if ([string]::IsNullOrEmpty($StartName)) { $null } else { $StartName });
				StartPassword = $(if ([string]::IsNullOrEmpty($StartPassword)) { $null } else { (ConvertTo-PlainText $StartPassword) });
				LoadOrderGroup = $(if ([string]::IsNullOrEmpty($LoadOrderGroup)) { $null } else { $LoadOrderGroup });
				LoadOrderGroupDependencies = [string[]]$(if (($LoadOrderGroupDependencies -eq $null) -or ($LoadOrderGroupDependencies.Count -eq 0)) { $null } else { $LoadOrderGroupDependencies });
				ServiceDependencies = [string[]]$(if (($ServiceDependencies -eq $null) -or ($ServiceDependencies.Count -eq 0)) { $null } else { $ServiceDependencies });
			}
			foreach ($kvp in $parameters.GetEnumerator())
			{
				Write-Log -Message "- $($kvp.Key): '$([string]$kvp.Value)'" -Source ${CmdletName};
			}
			
			$result = Invoke-CimMethod -ClassName Win32_Service -MethodName Create -Arguments $parameters;
			if ($result -eq $null)
			{
				Write-Log -Message "Result: NULL." -Source ${CmdletName} -Severity 2;
			}
			elseif ($result.ReturnValue -ne $null)
			{
				[int]$statusCode = $result.ReturnValue;
				$statusText = Get-WmiStatusText $statusCode;
				Write-Log -Message "Result: $($statusCode) = '$($statusText)'." -Source ${CmdletName};
				if ($statusCode -ne 0) { throw $statusCode; }
			}
			else
			{
				Write-Log -Message "Result [unexpected format [$($result.GetType().Name)]]: $([string]$result)." -Source ${CmdletName} -Severity 2;
			}

			if ($DelayedAutoStart)
			{
				Write-Log -Message "Setting DelayedAutoStart for service '$($Name)' ..." -Source ${CmdletName};
				Set-ServiceStartMode -Name $Name -StartMode 'Automatic (Delayed Start)';
			}

			if ($StartService)
			{
				Write-Log -Message "Starting new service '$($Name)'..." -Source ${CmdletName};
				Start-ServiceAndDependencies -Name $Name -SkipServiceExistsTest -ContinueOnError:$ContinueOnError;
			}
		}
		catch
		{
			$failed = "Failed to install service [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Start-Win32Service {
	<#
	.SYNOPSIS
		Start Service
	.DESCRIPTION
		Use this command to start a service or a device.
	.PARAMETER Name
		The name of the service in the service database
	.PARAMETER Wait
		Specifies that script execution does not continue until the service is started.
	.PARAMETER Recurse
		Start also dependent services
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Start-Win32Service -Name ersupext
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Start-Win32Service.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][switch]$Wait = $false,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$win32Service = Find-Win32Service -Name $Name;
			if ($win32Service -eq $null) { throw "Service '$($Name)' not found."; }

			[bool]$SkipDependentServices = !$Recurse;
			if (!$Wait) { Write-Log -Message "The -Wait parameter is not set. However, Start-Win32Service waits by default." -Source ${CmdletName}; }
			
			Start-ServiceAndDependencies -Name $win32Service.Name -SkipServiceExistsTest -SkipDependentServices:$SkipDependentServices -ContinueOnError:$ContinueOnError;
		}
		catch
		{
			$failed = "Failed to start service [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Stop-Win32Service {
	<#
	.SYNOPSIS
		Stop Service
	.DESCRIPTION
		Use this command to stop a service or a device.
	.PARAMETER Name
		The name of the service in the service database
	.PARAMETER Wait
		Specifies that script execution does not continue until the service is stopped.
	.PARAMETER Recurse
		Stop also dependent services
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Stop-Win32Service -Name ersupext -Wait 30 -Recurse
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Stop-Win32Service.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][switch]$Wait = $false,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$win32Service = Find-Win32Service -Name $Name;
			if ($win32Service -eq $null) { throw "Service '$($Name)' not found."; }

			[bool]$SkipDependentServices = !$Recurse;
			if (!$Wait) { Write-Log -Message "The -Wait parameter is not set. However, Stop-Win32Service waits by default." -Source ${CmdletName}; }
			
			Stop-ServiceAndDependencies -Name $win32Service.Name -SkipServiceExistsTest -SkipDependentServices:$SkipDependentServices -ContinueOnError:$ContinueOnError;
		}
		catch
		{
			$failed = "Failed to stop service [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Uninstall-Win32Service {
	<#
	.SYNOPSIS
		Uninstall Service
	.DESCRIPTION
		Use this command to uninstall a service or a device for Windows. If required, the service will be stopped before it is uninstalled.
	.PARAMETER Name
		The name of the service in the service database
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Uninstall-Win32Service -Name ersupext
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Uninstall-Win32Service.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Uninstall-Win32Service_Internal -Name $Name;
		}
		catch
		{
			$failed = "Failed to uninstall service [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}
function Start-Program {
	<#
	.SYNOPSIS
		Run an application
	.DESCRIPTION
		Runs an application with the current account
	.PARAMETER Command
		Specifies the file to be executed including the path
	.PARAMETER WorkingDirectory
		Path to the alternative working directory. 
	.PARAMETER ExitCodeVariable
		Optional - Name of a variable that holds the returncode of the called command line.
	.PARAMETER Wait
		If this option is enabled, the script waits for the called process to 
		complete before continuing to execute the script.
	.PARAMETER MaxWaitMinutes
		Time in minutes to wait for the process to be completed. If the process is not terminated within this time, 
		package execution continues anyway and the variable for the returncode gets the value TimedOut.
	.PARAMETER SecureParameters
		The command or variable is not written to the log file. 
		This option prevents sensitive information contained in the command line, 
		such as license keys or passwords, from being read out via the log file.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Start-Program -Command '".\Files\setup.exe" /S' -ExitCodeVariable _returncode -MaxWaitMinutes 10 -Wait
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Start-Program.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Command,
		[Parameter(Mandatory=$false)][string]$WorkingDirectory = $null,
		[Parameter(Mandatory=$false)][string]$ExitCodeVariable = $null,
		[Parameter(Mandatory=$false)][switch]$Wait = $false,
		[Parameter(Mandatory=$false)][string]$MaxWaitMinutes = $null,
		[Parameter(Mandatory=$false)][switch]$SecureParameters = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		[hashtable]$displayParameters = $PSBoundParameters;
		if ($SecureParameters) { @("Command") | where { $displayParameters.ContainsKey($_) } | % { $displayParameters[$_] = "-hidden-" } }
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $displayParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$separator = "{(})";
			if ($Command.IndexOf($separator) -ge 0)
			{
				$index = $Command.IndexOf($separator);
				$WorkingDirectory = $Command.Substring($index + $separator.Length);
				$Command = $Command.Substring(0, $index);
				Write-Log -Message "Command contains working directory path '$($WorkingDirectory)' - resulting command: '$($Command)'." -Source ${CmdletName};
			}

			Write-Log -Message "Extracting file path from command [$($displayParameters['Command'])]" -Source ${CmdletName};
			$FilePath, $Arguments = [PSPD.API]::SearchProgram($Command);
			
			$params = @{
				WorkingDirectory = $WorkingDirectory;
				ExitCodeVariable = $ExitCodeVariable;
				Wait = $Wait;
				MaxWaitMinutes = $MaxWaitMinutes;
				SecureParameters = $SecureParameters;
				ContinueOnError = $false;
				Wow64 = $Wow64;
			}
			
			Start-ProgramAs -FilePath $FilePath -Arguments $Arguments -RunAs CurrentUser @params;
		}
		catch
		{
			$failed = "Failed to execute command [$($displayParameters['Command'])]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Start-ProgramAs {
	<#
	.SYNOPSIS
		Run an application as another user
	.DESCRIPTION
		Runs an application under a different user account than the current account
	.PARAMETER FilePath
		Specifies the file to be executed including the path
	.PARAMETER Arguments
		Specify the parameter you want to pass to the executable file call.
	.PARAMETER WorkingDirectory
		Path to the alternative working directory. 
	.PARAMETER ExitCodeVariable
		Optional - Name of a variable that holds the returncode of the called command line.
	.PARAMETER SignedExitCode
		If this option is activated, the return value is interpreted as signed double word.
	.PARAMETER Wait
		If this option is enabled, the script waits for the called process to 
		complete before continuing to execute the script.
	.PARAMETER MaxWaitMinutes
		Time in minutes to wait for the process to be completed. If the process is not terminated within this time, 
		package execution continues anyway and the variable for the returncode gets the value TimedOut.
	.PARAMETER SecureParameters
		The command or variable is not written to the log file. 
		This option prevents sensitive information contained in the command line, 
		such as license keys or passwords, from being read out via the log file.
	.PARAMETER WindowStyle
		How should the application window of the called process be displayed.
		- Normal: The application window is shown normally.
		- Minimized: The application window is minimized and only displayed as an icon in the taskba
		- Hidden: The application window is hidden and not visible to the user.
	.PARAMETER RunAs
		Specifies the credentials under which the command line is executed.
		- CurrentUser: The application runs under the account of the executing user.
		- UserName: This option can be used to explicitly specify the credentials with which the command line is executed.
		- DsmAccount: This option is available for compatibility reasons for imported DSM Packages and cannot be used in the Packaging PowerBench context.
		- LocalSystem: The application runs under the local system account.
		- LoggedOnUser: The application runs under the account of the interactively logged in user.
	.PARAMETER UserName
		The user account under which the application is to run. This is entered as USERNAME for local accounts or DOMAIN\USERNAME for domain accounts.
	.PARAMETER Password
		The password of the specified account. (Cleartext or Base64 encoded)
	.PARAMETER LeastPrivilege
		If User Account Control (UAC) is enabled, 
		Start-ProgramAs always executes the specified commandline as 
		administrator by default, unless this option is activated,
	.PARAMETER Logon
		If you want the script to run under a defined user account, this option determines how the account profile is handled.
		- NoProfile: The user profile is not loaded. No changes are written to the user profile after the script is executed
		- LoadProfile: The user profile is loaded; changes are written to the user profile
		- NetworkOnly: The script itself is executed in the context of the current user account. The specified user account is only used for network access
	.PARAMETER OnError
		Defines the status with which the package is terminated.
		- Done: 	This option causes the status information for successful installation to be written to the registry.
		- Undone: 	This option causes status information about package execution to be written to the registry, 
					but the IsInstalled value is set to 0 and the Status value is set to Undone to document that the package is not considered installed or executed.
		- UndoneContinueParentScript: Some as undone, but proceed executing parent script.
		- Failed: 	This option causes status information about package execution to be written to the registry, but the IsInstalled value is set to 0 and the Status value is set 
					to Failed to document that the package is considered failed and not executed. For packages with a user part, the entries for the Active Setup are not created. 
	.PARAMETER PassThru
		PassThrough process
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Start-ProgramAs -FilePath '.\Files\installer.exe' -Arguments '/VERYSILENT' -RunAs UserName -UserName '.\Administrator' -Password '=IVN0YXJ0MDE==' -MaxWaitMinutes 10 -ExitCodeVariable _returncode -WindowStyle Hidden -Wait -Logon LoadProfile -OnError Failed -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Start-ProgramAs.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FilePath,
		[Parameter(Mandatory=$false)][string]$Arguments = "",
		[Parameter(Mandatory=$false)][string]$WorkingDirectory = $null,
		[Parameter(Mandatory=$false)][string]$ExitCodeVariable = $null,
		[Parameter(Mandatory=$false)][switch]$SignedExitCode = $false,
		[Parameter(Mandatory=$false)][switch]$Wait = $false,
		[Parameter(Mandatory=$false)][string]$MaxWaitMinutes = $null,
		[Parameter(Mandatory=$false)][switch]$SecureParameters = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Normal", "Minimized", "Hidden")][string]$WindowStyle = "Normal",
		[Parameter(Mandatory=$false)][ValidateSet("UserName", "DsmAccount", "LocalSystem", "CurrentUser", "LoggedOnUser")][string]$RunAs = "UserName",
		[Parameter(Mandatory=$false)][string]$UserName = $null,
		[Parameter(Mandatory=$false)][string]$Password = $null,
		[Parameter(Mandatory=$false)][switch]$LeastPrivilege = $false,
		[Parameter(Mandatory=$false)][ValidateSet("NoProfile", "LoadProfile", "NetworkOnly")][string]$Logon = "NoProfile",
		[Parameter(Mandatory=$false)][ValidateSet("Done", "Undone", "UndoneContinueParentScript", "Failed")][string]$OnError = $null,
		[Parameter(Mandatory=$false)][switch]$PassThru = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		[hashtable]$displayParameters = $PSBoundParameters;
		if ($SecureParameters) { @("Arguments", "Password") | where { $displayParameters.ContainsKey($_) } | % { $displayParameters[$_] = "-hidden-" } }
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $displayParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$separator = "|";
			if ($WorkingDirectory -eq $separator)
			{
				$index = $FilePath.IndexOf($separator);
				$WorkingDirectory = $FilePath.Substring($index + $separator.Length);
				$FilePath = $FilePath.Substring(0, $index);
				Write-Log -Message "File path contains working directory path '$($WorkingDirectory)' - resulting file path: '$($FilePath)'." -Source ${CmdletName};
			}

			if ($FilePath.StartsWith("`""))
			{
				Write-Log -Message "The executable path is quoted, removing quotes from [$($FilePath)]." -Source ${CmdletName};
				$FilePath = $FilePath.Trim("`"");
			}

			if (![string]::IsNullOrEmpty($WorkingDirectory) -and $WorkingDirectory.StartsWith("`""))
			{
				Write-Log -Message "The working directory path is quoted, removing quotes from [$($WorkingDirectory)]." -Source ${CmdletName};
				$WorkingDirectory = $WorkingDirectory.Trim("`"");
			}

			$FilePath = Expand-Path $FilePath -Wow64:$Wow64;
			Write-Log -Message "Starting program [$($FilePath)] with arguments [$($displayParameters['Arguments'])]" -Source ${CmdletName};

			$si = New-Object System.Diagnostics.ProcessStartInfo;
			$si.FileName = $FilePath;
			$si.Arguments = $Arguments;
			$si.CreateNoWindow = $true;
			$si.LoadUserProfile = ($Logon -eq "LoadProfile");
			$si.UseShellExecute = $false;
			$si.WindowStyle = $WindowStyle;
			$si.RedirectStandardOutput = $true;
			$si.RedirectStandardError = $true;
			
			if (![string]::IsNullOrEmpty($WorkingDirectory))
			{
				$si.WorkingDirectory = $WorkingDirectory;
				Write-Log -Message "Setting working directory to: '$($si.WorkingDirectory)'." -Source ${CmdletName};
			}
			
			$runAsLocalSystem = (($RunAs -eq "LocalSystem") -or ($RunAs -eq "DsmAccount"));
			$runAsLoggedOnUser = ($RunAs -eq "LoggedOnUser");
			$usePsExec = ($runAsLocalSystem -or ($IsAdmin -eq [bool]$LeastPrivilege));
			$credential = $null;
			if (($RunAs -eq "UserName") -and ![string]::IsNullOrEmpty($UserName))
			{
				Write-Log -Message "Starting program as user '$($UserName)'." -Source ${CmdletName};
				$credential = New-Object PSCredential $UserName, (ConvertTo-SecureString -String (ConvertTo-PlainText $Password) -AsPlainText -Force);
				$si.UserName = $credential.GetNetworkCredential().UserName;
				$si.Domain = $credential.GetNetworkCredential().Domain;
				$si.Password = $credential.Password;
			}
			elseif ($runAsLocalSystem)
			{
				Write-Log -Message "Starting program as local system." -Source ${CmdletName};
			}
			elseif ($runAsLoggedOnUser)
			{
				$action = "Handling -RunAs $($RunAs)";
				$theLoggedOnUser = $null;

				$pdc = Get-PdContext;
				if ($pdc.User.Process -eq $pdc.User.RunAs) { $theLoggedOnUser = "the active logged on user ($($pdc.User.RunAs))"; }
				elseif ($pdc.User.Process -eq $pdc.User.Console) { $theLoggedOnUser = "the console user ($($pdc.User.Console))"; }
				elseif ($pdc.User.Process -eq $pdc.User.LoggedOn) { $theLoggedOnUser = "the logged on user ($($pdc.User.LoggedOn))"; }

				if (![string]::IsNullOrEmpty($theLoggedOnUser))
				{
					Write-Log -Message "$($action): The executing (current) user is $($theLoggedOnUser) -> starting program as current user." -Source ${CmdletName};
					$runAsLoggedOnUser = $false;
				}
				else
				{
					Write-Log -Message "$($action): Starting program as interactively logged on user." -Source ${CmdletName};
				}
			}
			else
			{
				Write-Log -Message "Starting program as current user." -Source ${CmdletName};
			}
			
			if ($usePsExec)
			{
				if (![System.IO.File]::Exists($PsExecPath)) { throw "Missing tool '$($PsExecPath)'."; }
				
				$psExecOptions = "-accepteula"; # -accepteula = no license dialog

				$usingService = $runAsLocalSystem;
				if ($usingService -and ![string]::IsNullOrEmpty($PsExecServiceName))
				{
					$arg = $(if ($PsExecServiceName.IndexOf(" ") -gt 0) { "`"$($PsExecServiceName)`"" } else { $PsExecServiceName });
					if ($UsePaExecExe) { $psExecOptions += " -sname $($arg)"; } # -sname = remote service name
					else { $psExecOptions += " -r $($arg)"; } # -r = name of the remote service to create or interact with.
				}
				
				if ($usingService -and $UsePaExecExe -and ($PaExecShare -ne $null))
				{
					$psExecOptions = "\\$($PaExecShare.Computer) $($psExecOptions)"; # \\computer = run the application on the remote computer (here: localhost)

					$arg = $(if ($PaExecShare.Name.IndexOf(" ") -gt 0) { "`"$($PaExecShare.Name)`"" } else { $PaExecShare.Name });
					$psExecOptions += " -share $($arg)"; # -share = remote share name

					$arg = $(if ($PaExecShare.Path.IndexOf(" ") -gt 0) { "`"$($PaExecShare.Path)`"" } else { $PaExecShare.Path });
					$psExecOptions += " -sharepath $($arg)"; # -sharepath = real path of specified remote share
				}

				if ($runAsLocalSystem) { $psExecOptions += " -s"; } # -s = system account
				if ($IsAdmin -and $LeastPrivilege) { $psExecOptions += " -l"; } # -l = limited user
				elseif (!$IsAdmin -and !$LeastPrivilege) { $psExecOptions += " -h"; } # -h = elevated token (if available)
				if ($WindowStyle -eq "Normal") { $psExecOptions += " -i"; } # -i = visible
				if (!$Wait) { $psExecOptions += " -d"; } # -d = do not wait
				if ($UsePaExecWow64) { $psExecOptions += " -dfr"; } # -dfr = Disable WOW64 File Redirection for the new process

				Write-Log -Message "Using tool '$($PsExecPath)' with options '$($psExecOptions)'." -Source ${CmdletName};
				$si.FileName = $PsExecPath;
				$si.Arguments = "$($psExecOptions) `"$($FilePath)`" $($Arguments)";

				Write-Log -Message "Starting tool [$($si.FileName)] with arguments [$($psExecOptions) `"$($FilePath)`" $($displayParameters['Arguments'])]" -Source ${CmdletName};
			}
			
			$status = $null;

			$process = $null;
			$pdc = Get-PdContext;
			if ($pdc.IsLocalSystemAccount -and ($credential -ne $null))
			{
				Write-Log -Message "Executing account is LocalSystem ($($pdc.User.Process)) - calling PSPD.API.StartProcessAsUser." -Source ${CmdletName};
				$process = [PSPD.API]::StartProcessAsUser($si);
			}
			elseif ($runAsLoggedOnUser)
			{
				Write-Log -Message "Starting program as interactively logged on user via PSPD.API.StartInteractive." -Source ${CmdletName};
				$process = [PSPD.API]::StartInteractive($si, "~");
			}
			else
			{
				$process = [System.Diagnostics.Process]::Start($si);
			}

			if ($Wait)
			{
				$timeout = [int]::MaxValue;
				$timeoutDisplay = "infinite";
				if ([double]$MaxWaitMinutes -gt 0)
				{
					$ts = [TimeSpan]::FromMinutes($MaxWaitMinutes);
					$timeout = [int]$ts.TotalMilliseconds;
					$timeoutDisplay = "$($ts.ToString()) ($($timeout)ms)";
				}
				
				Write-Log -Message "Started process with ID $($process.Id) - waiting for exit: $($timeoutDisplay)." -Source ${CmdletName};
				$exited = $process.WaitForExit($timeout);
				if ($process.HasExited)
				{
					$stderr = $null;
					$signedStatus = [int]$process.ExitCode;
					if ($signedStatus -ne 0)
					{
						$stderr = $(if ($process.StandardError -ne $null) { ([string]$process.StandardError.ReadToEnd()).Trim() } else { $null });
						$stdout = $(if ($process.StandardOutput -ne $null) { ([string]$process.StandardOutput.ReadToEnd()).Trim() } else { $null });
						if ([string]::IsNullOrEmpty($stderr)) { $stderr = $stdout; }
					}
					
					$unsignedStatus = [UInt32]"0x$($signedStatus.ToString('X'))";
					$status = $(if ($SignedExitCode) {$signedStatus} else {$unsignedStatus});
					Write-Log -Message "Process has exited with exit code $($status) (0x$($status.ToString('X8')))$(if (![string]::IsNullOrEmpty($stderr)) { ': [' + $stderr + ']' })." -Source ${CmdletName};
					
					if ($usePsExec -and $UsePaExecExe -and $pdc.Package.PaExecThrowOnWellKnownErrorCodes -and (@(-1 .. -11) -contains $signedStatus))
					{
						$knownErrors = @{ -1 = "internal error"; -2 = "command line error"; -3 = "failed to launch app (locally)"; -4 = "failed to copy PAExec to remote (connection to ADMIN$ might have failed)"; -5 = "connection to server taking too long (timeout)"; -6 = "PAExec service could not be installed/started on remote server"; -7 = "could not communicate with remote PAExec service"; -8 = "failed to copy app to remote server"; -9 = "failed to launch app (remotely)"; -10 = "app was terminated after timeout expired"; -11 = "forcibly stopped with Ctrl-C / Ctrl-Break" };
						$exitInfo = $(if ($knownErrors.ContainsKey([int]$signedStatus)) { "$($signedStatus) [$($knownErrors[[int]$signedStatus])]" } else { $signedStatus });
						Write-Log -Message "PaExec.exe returned exit code $($exitInfo) - throwing exception (PaExecThrowOnWellKnownErrorCodes)." -Source ${CmdletName};
						throw "PaExec.exe failed with known exit code $($exitInfo)";
					}
				}
				else
				{
					$status = "TimeOut";
					Write-Log -Message "Reached maximum wait time: continue." -Source ${CmdletName};
				}
			}
			else
			{
				Write-Log -Message "Started process with ID $($process.Id): continue." -Source ${CmdletName};
				$status = "NoWait";
			}
			
			if (![string]::IsNullOrEmpty($ExitCodeVariable))
			{
				Set-PdVar -Name $ExitCodeVariable -Value $status;
			}
			
			if ($PassThru) { return $process; }
		}
		catch
		{
			$failed = "Failed to execute program [$($FilePath)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			
			if ($OnError -eq "Done")
			{
				Write-Log -Message "Value of -OnError is '$($OnError)': continue." -Source ${CmdletName};
				$ContinueOnError = $true;
			}
			elseif (($OnError -eq "Undone") -or ($OnError -eq "UndoneContinueParentScript")) #"Undone", "UndoneContinueParentScript", "Failed"
			{
				Write-Log -Message "Value of -OnError is '$($OnError)': terminate with status code '$($OnError)'." -Source ${CmdletName};
				Exit-Package -Status $OnError -Message $failed;
			}
			elseif (($OnError -eq "Failed") -and $ContinueOnError)
			{
				Write-Log -Message "Value of -OnError is '$($OnError)': continue due to -ContinueOnError switch." -Source ${CmdletName};
			}
			elseif ($OnError -eq "Failed")
			{
				Write-Log -Message "Value of -OnError is '$($OnError)': fail." -Source ${CmdletName};
			}
			
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }

			if (![string]::IsNullOrEmpty($ExitCodeVariable))
			{
				if ([string]::IsNullOrEmpty($status)) { $status = "Error"; }
				Set-PdVar -Name $ExitCodeVariable -Value $status;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Copy-RegistryKey {
	<#
	.SYNOPSIS
		Copy a registry key.
	.DESCRIPTION
		Copies everything below the specified registry subtree to another subtree. 
		If the key to be copied to does not exist, it is created.
	.PARAMETER SourceKeyPath
		Select the key whose values or subordinate keys are to be copied.
	.PARAMETER TargetKeyPath
		Select the key under which the values or subordinate keys are to be copied. 
	.PARAMETER Recurse
		If this is set, all subordinate keys are copied as well.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Copy-RegistryKey -SourceKeyPath 'HKEY_CURRENT_USER\SOFTWARE\CANCOM\Package Deployment' -TargetKeyPath 'HKEY_CURRENT_USER\SOFTWARE\CANCOM\Package Deployment Backup' -Recurse -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Copy-RegistryKey.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$SourceKeyPath,
		[Parameter(Mandatory=$true)][string]$TargetKeyPath,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$sourceKey = Get-PdRegistryKey -Path $SourceKeyPath -Wow64:$Wow64;
			$targetKey = Get-PdRegistryKey -Path $TargetKeyPath -Create -Writable -Wow64:$Wow64;
			
			Write-Log -Message "Copy registry '$($sourceKey.Name)' to '$($targetKey.Name)' - recurse: $($Recurse)." -Source ${CmdletName};
			Invoke-RegExe -Operation COPY -KeyName $sourceKey.Name -TargetKeyName $targetKey.Name -Recurse:$Recurse; # -Wow64:$Wow64
		}
		catch
		{
			$failed = "Failed to copy registry key [$SourceKeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-RegistryKey {
	<#
	.SYNOPSIS
		Delete a registry key.
	.DESCRIPTION
		Deletes a key or value from the registry.
	.PARAMETER KeyPath
		Select the key from which a value or the entire key is to be deleted.
	.PARAMETER ValueName
		Optionally the whole key or the name of a value.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-RegistryKey -KeyPath 'HKEY_CURRENT_USER\SOFTWARE\CANCOM\Package Deployment' -ValueName '' -Context User
	.EXAMPLE
		Remove-RegistryKey -KeyPath 'HKEY_CURRENT_USER\SOFTWARE\CANCOM\Package Deployment\Installed Apps\{2444B599-7E1B-408E-9172-1869542448B4}' -ValueName Description -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-RegistryKey.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][Alias("Key")][string]$KeyPath,
		[Parameter(Mandatory=$false)][Alias("Name")][string]$ValueName = $null,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false, [Parameter(Mandatory=$false)][string]$SID = $null, # see AppDeployToolkitMain.ps1
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		$key = $null;
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$key = Get-PdRegistryKey -Path $KeyPath -AcceptNull -Writable -Wow64:$Wow64;
			if ($key -eq $null)
			{
				Write-Log -Message "Registry key '$($KeyPath)' does not exist." -Source ${CmdletName};
				return;
			}
			
			if ([string]::IsNullOrEmpty($ValueName))
			{
				Write-Log -Message "Deleting registry key '$($KeyPath)' and any subkeys recursively." -Source ${CmdletName};
				$key.DeleteSubKeyTree(""); # "" -> $KeyPath
			}
			elseif ([System.Management.Automation.WildcardPattern]::ContainsWildcardCharacters($ValueName))
			{
				$pattern = $ValueName;
				Write-Log -Message "Value name contains wildcards: '$($pattern)'" -Source ${CmdletName};
				
				$ReducedWildcardCharacters = $true; # restricted to * and ? - no ` as escape character
				if ($ReducedWildcardCharacters)
				{
					# $pattern = $pattern.Replace('`','``').Replace('[','`[').Replace(']','`]');
					$pattern = [System.Management.Automation.WildcardPattern]::Escape($pattern.Replace('`','``')).Replace('`*','*').Replace('`?','?');
					Write-Log -Message "Restricting wildcards to * and ? - new pattern: '$($pattern)'" -Source ${CmdletName};
				}
				
				foreach ($name in $key.GetValueNames())
				{
					if ($name -notlike $pattern) { continue; }

					Write-Log -Message "Value name '$($name)' is matching '$($pattern)' - delete in registry key '$($KeyPath)'." -Source ${CmdletName};
					$key.DeleteValue($name);
				}
			}
			else
			{
				Write-Log -Message "Deleting value '$($ValueName)' in registry key '$($KeyPath)'." -Source ${CmdletName};
				$key.DeleteValue($ValueName); # "" -> $KeyPath
			}
		}
		catch
		{
			$failed = "Failed to delete registry key [$KeyPath][$ValueName]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($key -ne $null)
			{
				$key.Close();
				$key = $null;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Mount-Registry {
	<#
	.SYNOPSIS
		Mounts a registry key.
	.DESCRIPTION
		Creates a subkey under HKEY_USERS or HKEY_LOCAL_MACHINE and includes the registry information from a specified file in that subkey
	.PARAMETER FilePath
		Specifies the file to be mounted including the path.
	.PARAMETER KeyPath
		Select the key to which the information from the file specified in Filename should be loaded. Variables are allowed.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Mount-Registry -KeyPath 'HKEY_LOCAL_MACHINE\Mount -FilePath 'C:\Users\Default\NTUSER.DAT'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Mount-Registry.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FilePath,
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }
			
			$FilePath = Expand-Path $FilePath;
			
			$mountName = $null;
			$sep = $KeyPath.LastIndexOf("\");
			if ($sep -ge 0)
			{
				$rootPath = $KeyPath.Substring(0, $sep);
				$keyName = $KeyPath.Substring($sep);
				$root = Get-PdRegistryKey -Path $rootPath;
				$mountName = ($root.Name + $keyName);
				$root.Close();
			}
			else
			{
				$key = Get-PdRegistryKey -Path $KeyPath;
				$mountName = $key.Name;
				$key.Close();
			}
			
			Write-Log -Message "Mount file '$($FilePath)' on registry key '$($mountName)'." -Source ${CmdletName};
			Invoke-RegExe -Operation LOAD -KeyName $mountName -FileName $FilePath;
		}
		catch
		{
			$failed = "Failed to mount registry [$FilePath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Dismount-Registry {
	<#
	.SYNOPSIS
		Unloads a registry key.
	.DESCRIPTION
		Unloads a registry key which was previously loaded via the Mount-Registry command.
	.PARAMETER KeyPath
		The key to be unloaded
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Dismount-Registry -KeyPath 'HKEY_LOCAL_MACHINE\Mount
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Dismount-Registry.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$mountName = $null;
			$key = Get-PdRegistryKey -Path $KeyPath -AcceptNull;
			if ($key -eq $null)
			{
				Write-Log -Message "Registry key '$($KeyPath)' is not mounted." -Source ${CmdletName};
				return;
			}
			
			$mountName = $key.Name;
			$key.Close();
			
			Write-Log -Message "Dismount registry key '$($mountName)'." -Source ${CmdletName};
			Invoke-RegExe -Operation UNLOAD -KeyName $mountName;
		}
		catch
		{
			$failed = "Failed to dismount registry [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Import-RegXml {
	<#
	.SYNOPSIS
		Import from file
	.DESCRIPTION
		Extends the registry with a reg- or regx-file. Existing entries in the registry are updated.
	.PARAMETER FilePath
		Specifies the path included in the .reg- or .regx-file.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER PreventUninstall
		Prevents script from undoing configuration if running on 'uninstall mode'
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Import-RegXml -FilePath 'D:\Keys.regx'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Import-RegXml.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FilePath,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			$isReverseMode = Test-ReverseMode;

			$FilePath = Expand-Path $FilePath -Wow64:$Wow64;
			Write-Log -Message "Import content of registry information file '$($FilePath)'." -Source ${CmdletName};
			
			$xml = New-Object System.Xml.XmlDocument;
			$xml.Load($FilePath);
			$root = $xml.DocumentElement;
			
			$pdc = Get-PdContext;
			$installMode = $pdc.InstallMode;
			$isUserPart = (($Context -eq "User") -or ($Context -eq "UserPerService") -or ($pdc.InstallMode -eq "InstallUserPart"));
			
			if ([string]::IsNullOrEmpty($installMode))
			{
				$installMode = "Install";
				Write-Log -Message "No installation mode set - assume: '$($installMode)'." -Source ${CmdletName};
			}
			elseif ($isUserPart -and ![string]::IsNullOrEmpty($pdc.UserPartInstallMode))
			{
				$installMode = $pdc.UserPartInstallMode;
				Write-Log -Message "Installation mode for user part: '$($installMode)'." -Source ${CmdletName};
			}
			
			$isRepairMode = ($installMode -eq "Repair");
			$isUninstallMode = ($installMode -eq "Uninstall");
			Write-Log -Message "Install mode: $($installMode), is user part: $($isUserPart), is repair mode: $($isRepairMode), is uninstall mode: $($isUninstallMode)." -Source ${CmdletName};
			
			function isTrue([string]$v) { $r = 0; return $(if ([int]::TryParse($v, [ref]$r) -or [bool]::TryParse($v, [ref]$r)) {[bool]$r} else {[bool]$v}); }

			$toRemoveKeyNames = @();
			
			$keyNodes = $root.SelectNodes("Key");
			foreach ($keyNode in $keyNodes)
			{
				$keyName = Expand-PSString $keyNode.GetAttribute("Name");
				$isUserPartKey = ($keyName -match '^(Registry::)?(HKCU|HKEY_CURRENT_USER)([:\\/]|$)');
				$isRepairModeKey = (isTrue $keyNode.GetAttribute("Repair"));
				$isUninstallModeKey = (isTrue $keyNode.GetAttribute("Uninstall"));
				
				Write-Log -Message "Processing key '$($keyName)', user-part-key: $($isUserPartKey), repair-key: $($isRepairModeKey), uninstall-key: $($isUninstallModeKey)." -Source ${CmdletName};
				
				$skip = $null;
				if ($isUserPart -ne $isUserPartKey) { $skip = "Wrong install-part"; }
				elseif ($isRepairMode -and !$isRepairModeKey) { $skip = "No Repair-Key"; }
				elseif ($isUninstallMode -and !$isUninstallModeKey) { $skip = "No uninstall-key"; }
				if (![string]::IsNullOrEmpty($skip))
				{
					Write-Log -Message "Skipping key: $($skip)." -Source ${CmdletName};
					continue;
				}

				$valueNodes = $keyNode.SelectNodes("Value");
				Write-Log -Message "Values found: $($valueNodes.Count)." -Source ${CmdletName};

				# uninstall mode
				if ($isUninstallMode)
				{
					$key = Get-PdRegistryKey -Path $keyName -Wow64:$Wow64 -AcceptNull -Writable;
					if ($key -eq $null)
					{
						$skip = "Not found";
						Write-Log -Message "Skipping key: $($skip)." -Source ${CmdletName};
						continue; # -> next $keyName
					}
					
					foreach ($valueNode in $valueNodes)
					{
						$valueName = Expand-PSString $valueNode.GetAttribute("Name");
						Write-Log -Message "Removing value '$($valueName)' at key '$($keyName)'." -Source ${CmdletName};
						
						$data = $key.GetValue($valueName);
						if ($data -eq $null)
						{
							$skip = "Not found";
							Write-Log -Message "Skipping value: $($skip)." -Source ${CmdletName};
							continue;
						}
						
						$key.DeleteValue($valueName);
					}
					
					if (($key.SubKeyCount -eq 0) -and ($key.ValueCount -eq 0))
					{
						Write-Log -Message "Removing empty key '$($keyName)'." -Source ${CmdletName};
						$key.DeleteSubKey(""); # "" -> $keyName
					}
					else
					{
						$toRemoveKeyNames += $keyName; # suspend
					}
					
					$key.Close();
					continue; # -> next $keyName

				} # ($isUninstallMode)
				
				if ($valueNodes.Count -eq 0) # create empty key
				{
					$key = Get-PdRegistryKey -Path $keyName -Wow64:$Wow64 -AcceptNull;
					if ($key -eq $null)
					{
						Write-Log -Message "Creating empty key '$($keyName)'." -Source ${CmdletName};
						$key = Get-PdRegistryKey -Path $keyName -Wow64:$Wow64 -Create;
					}
					$key.Close();
					continue; # -> next $keyName
				}
				
				foreach ($valueNode in $valueNodes)
				{
					$valueName = Expand-PSString $valueNode.GetAttribute("Name");
					$type = $valueNode.GetAttribute("Type");
					
					$action = $valueNode.GetAttribute("Action");
					if ([string]::IsNullOrEmpty($action)) { $action = "Set"; }
					
					$position = $valueNode.GetAttribute("Position");
					if ([string]::IsNullOrEmpty($position)) { $position = "0"; }
					
					$allowDuplicates = (isTrue $valueNode.GetAttribute("AllowDuplicates"));

					$data = @($valueNode.SelectNodes("Data") | % { Expand-PSString $_.InnerText });
					if ($data.Count -eq 0) { $data = Expand-PSString $valueNode.InnerText; }
					
					Write-Log -Message "Processing value '$($valueName)', type: $($type), action: $($action)." -Source ${CmdletName};
					
					$params = @{ KeyPath = $keyName; ValueName = $valueName; Value = $data; Action = $action; Wow64 = $Wow64; Context = "Any"; ContinueOnError = $ContinueOnError; };

					switch ($type)
					{
						"MultiString"  { Write-RegistryMultiString @params -Position $position; break; }
						"DWord"        { Write-RegistryDWord @params; break; }
						"QWord"        { Write-RegistryQWord @params; break; }
						"String"       { Write-RegistryValue @params -ValueKind String -AllowDuplicates:$allowDuplicates; break; }
						"ExpandString" { Write-RegistryValue @params -ValueKind ExpandString -AllowDuplicates:$allowDuplicates; break; }
						default        { Write-RegistryValue @params -ValueKind Binary; break; }
					}

				} # foreach ($valueNode in $valueNodes)
			} # foreach ($keyNode in $keyNodes)
			
			# finishing uninstall mode
			if ($isUninstallMode -and ($toRemoveKeyNames.Count -gt 0))
			{
				Write-Log -Message "Removing remaining uninstall-keys ($($toRemoveKeyNames.Count)):" -Source ${CmdletName};
				$toRemoveKeyNames = @($toRemoveKeyNames | sort -Descending { $_.Length }); # should be subkeys first
				foreach ($keyName in $toRemoveKeyNames)
				{
					$key = Get-PdRegistryKey -Path $keyName -Wow64:$Wow64 -AcceptNull -Writable;
					
					$skip = $null;
					if ($key -eq $null) { $skip = "Not found"; }
					elseif (($key.SubKeyCount -gt 0) -or ($key.ValueCount -gt 0))
					{
						$keyInfo = $(if ($key.SubKeyCount -gt 0) {"$($key.SubKeyCount) ('$([string]::Join(''', ''', $key.GetSubKeyNames()))')"} else {"0"});
						$valueInfo = $(if ($key.ValueCount -gt 0) {"$($key.ValueCount) ('$([string]::Join(''', ''', $key.GetValueNames()))')"} else {"0"});
						$skip = "Not empty [keys: $($keyInfo), values: $($valueInfo)]";
					}
					
					if ([string]::IsNullOrEmpty($skip))
					{
						Write-Log -Message "Removing key '$($keyName)'." -Source ${CmdletName};
						$key.DeleteSubKey(""); # "" -> $keyName
						$key.Close();
					}
					else
					{
						Write-Log -Message "Skipping key '$($keyName)': $($skip)." -Source ${CmdletName};
						if ($key -ne $null) { $key.Close(); }
					}
				}
			} # ($isUninstallMode -and ($toRemoveKeyNames.Count -gt 0))
		}
		catch
		{
			$failed = "Failed to import registry [$FilePath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Import-Registry {
	<#
	.SYNOPSIS
		Import from file
	.DESCRIPTION
		Extends the registry with a reg- or regx-file. Existing entries in the registry are updated.
	.PARAMETER FilePath
		Specifies the path included in the .reg- or .regx-file.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER PreventUninstall
		Prevents script from undoing configuration if running on 'uninstall mode'
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Import-Registry -FilePath 'D:\Keys.regx'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Import-Registry.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FilePath,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			$isReverseMode = Test-ReverseMode;

			$FilePath = Expand-Path $FilePath -Wow64:$Wow64;
			Write-Log -Message "Import content of registry information file '$($FilePath)'." -Source ${CmdletName};
			
			$extension = [System.IO.Path]::GetExtension($FilePath);
			
			$RegXmlExtension = ".regx";
			$RegXmlCommonParams = @{ Wow64 = $Wow64; ContinueOnError = $ContinueOnError; Context = $(if ([string]::IsNullOrEmpty($Context)) { "Any" } else { $Context }); PreventUninstall = $PreventUninstall; };
			
			if ($extension -eq $RegXmlExtension)
			{
				Import-RegXml -FilePath $FilePath @RegXmlCommonParams;
				return;
			}
			elseif ($extension -eq ".nir")
			{
				$extension = $RegXmlExtension;
				Write-Log -Message "Prefer file extension '$($extension)'." -Source ${CmdletName};
				$regFilePath = [System.IO.Path]::ChangeExtension($FilePath, $extension);
				if ([System.IO.File]::Exists($regFilePath))
				{
					Import-RegXml -FilePath:$regFilePath @RegXmlCommonParams;
					return;
				}
				else
				{
					$extension = ".reg";
					Write-Log -Message "File '$($regFilePath)' not found. Check file extension '$($extension)'." -Source ${CmdletName};
				}
			}
			
			$pdc = Get-PdContext;
			$installMode = $pdc.InstallMode;
			
			if ([string]::IsNullOrEmpty($installMode))
			{
				$installMode = "Install";
				Write-Log -Message "No installation mode set - assume: '$($installMode)'." -Source ${CmdletName};
			}
			elseif (($installMode -eq "InstallUserPart") -and ![string]::IsNullOrEmpty($pdc.UserPartInstallMode))
			{
				$installMode = $pdc.UserPartInstallMode;
				Write-Log -Message "Installation mode for user part: '$($installMode)'." -Source ${CmdletName};
			}
			
			$checkInstall = (($installMode -ne "Repair") -or ($installMode -eq "Uninstall"));
			$processed = 0;
			
			$extension = ".reg";
			$regFilePath = [System.IO.Path]::ChangeExtension($FilePath, ".$($installMode)$($extension)");
			$found = [System.IO.File]::Exists($regFilePath);
			Write-Log -Message "Found '$($regFilePath)': $($found)." -Source ${CmdletName};
			
			if (!$found -and $checkInstall -and ($installMode -ne "Install"))
			{
				$regFilePath = [System.IO.Path]::ChangeExtension($FilePath, ".Install$($extension)");
				$found = [System.IO.File]::Exists($regFilePath);
				Write-Log -Message "Found '$($regFilePath)': $($found)." -Source ${CmdletName};
			}
			
			if (!$isReverseMode -and !$found)
			{
				$regFilePath = [System.IO.Path]::ChangeExtension($FilePath, $extension);
				$found = [System.IO.File]::Exists($regFilePath);
				Write-Log -Message "Found '$($regFilePath)': $($found)." -Source ${CmdletName};
			}
			
			if ($found)
			{
				Write-Log -Message "Processing '$($regFilePath)' ...." -Source ${CmdletName};
				Invoke-RegExe -Operation IMPORT -FileName $regFilePath -Wow64:$Wow64;
				$processed++;
			}

			$extension = ".ps1";
			$ps1FilePath = [System.IO.Path]::ChangeExtension($FilePath, ".$($installMode)$($extension)");
			$found = [System.IO.File]::Exists($ps1FilePath);
			Write-Log -Message "Found '$($ps1FilePath)': $($found)." -Source ${CmdletName};
			
			if (!$found -and $checkInstall -and ($installMode -ne "Install"))
			{
				$ps1FilePath = [System.IO.Path]::ChangeExtension($FilePath, ".Install$($extension)");
				$found = [System.IO.File]::Exists($ps1FilePath);
				Write-Log -Message "Found '$($ps1FilePath)': $($found)." -Source ${CmdletName};
			}
			
			if (!$found)
			{
				$ps1FilePath = [System.IO.Path]::ChangeExtension($FilePath, $extension);
				$found = [System.IO.File]::Exists($ps1FilePath);
				Write-Log -Message "Found '$($ps1FilePath)': $($found)." -Source ${CmdletName};
			}
			
			if ($found)
			{
				Write-Log -Message "Processing '$($ps1FilePath)' ...." -Source ${CmdletName};
				. $ps1FilePath; # Invoke-DynWow64Script -Path $ps1FilePath -Wow64:$Wow64;
				$processed++;
			}
			
			if ($processed -eq 0)
			{
				Write-Log -Message "No registry or script files found to process based on '$($FilePath)'." -Severity 2 -Source ${CmdletName};
				if (!$isReverseMode -and ($FilePath -ne $regFilePath) -and [System.IO.File]::Exists($FilePath))
				{
					Write-Log -Message "Processing '$($FilePath)' ...." -Source ${CmdletName};
					Invoke-RegExe -Operation IMPORT -FileName $FilePath -Wow64:$Wow64;
					$processed++;
				}
			}
		}
		catch
		{
			$failed = "Failed to import registry [$FilePath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Save-Registry {
	<#
	.SYNOPSIS
		Save a registry key
	.DESCRIPTION
		Use this command to save a registry keys including all subkeys and values in a file.
	.PARAMETER KeyPath
		Select the key from which the entire branch of the registry is to be saved
	.PARAMETER FilePath
		Binary file that holds the registry information.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Save-Registry -KeyPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\.NETFramework' -FilePath 'D:\Keys'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Save-Registry.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$true)][string]$FilePath,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$keyName = $null;
			$key = Get-PdRegistryKey -Path $KeyPath -AcceptNull:$Wow64; # -Wow64:$Wow64
			if ($key -ne $null) { $keyName = $key.Name; }
			else { $keyName = (Get-PdRegistryKey -Path $KeyPath -Wow64:$Wow64).Name.Replace("\$($Wow6432NodeName)",""); } # $KeyPath only in WOW64
			$FilePath = Expand-Path $FilePath; # -Wow64:$Wow64
			
			Write-Log -Message "Save registry key '$($keyName)' to file '$($FilePath)'." -Source ${CmdletName};
			Invoke-RegExe -Operation SAVE -KeyName $keyName -FileName $FilePath -Wow64:$Wow64;
		}
		catch
		{
			$failed = "Failed to export registry [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-RegistryValue {
	<#
	.SYNOPSIS
		Read information from the registry
	.DESCRIPTION
		This command reads information from the registry. 
	.PARAMETER KeyPath
		The Path to the Registry Key.
	.PARAMETER ValueName
		The name of the value
	.PARAMETER ValueVariable
		Name of the variable in which the read value is stored. 
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-RegistryValue -KeyPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\.NETFramework' -ValueName InstallRoot -ValueVariable _installRoot -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-RegistryValue.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][string]$ValueName = $null,
		[Parameter(Mandatory=$true)][string]$ValueVariable = $null,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		$key = $null;
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }
			
			$key = Get-PdRegistryKey -Path $KeyPath -Wow64:$Wow64;
			Write-Log -Message "Read value '$($ValueName)' of registry key '$($key.Name)' into variable '$($ValueVariable)'." -Source ${CmdletName};
			$value = $key.GetValue($ValueName, $null, [Microsoft.Win32.RegistryValueOptions]::DoNotExpandEnvironmentNames);
			if (![string]::IsNullOrEmpty($ValueVariable))
			{
				Set-PdVar -Name $ValueVariable -Value $value;
			}
		}
		catch
		{
			$failed = "Failed to read registry value '$ValueName' [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($key -ne $null)
			{
				$key.Close();
				$key = $null;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Write-RegistryValue {
	<#
	.SYNOPSIS
		Changes or adds a string value in the registry
	.DESCRIPTION
		Changes or adds a string value in the registry, existing entries in the registry are updated.
	.PARAMETER KeyPath
		The Path to the Registry Key.
	.PARAMETER ValueName
		The name of the value
	.PARAMETER Value
		Optionally the default value of the key or the name of another value.
	.PARAMETER ValueKind
		The Kind of String:
		- String: Default
		- ExpandString: Save as REG_EXPAND_SZ
		- Binary: Saves the value in binary format
	.PARAMETER Action
		Defines how to handle the contents of a possibly existing value
		- Set: The content is set to the specified value and any existing value is overwritten.
		- Append: The specified value is appended to any existing line.
		- Remove: If the value already exists in the existing string, it is removed.
		- Prepend: The specified value is inserted at the beginning of any existing line.
	.PARAMETER AllowDuplicates
		Determines whether the specified value is inserted even if it already occurs in the existing content 
		(only valid for the options "Add value to the end of existing line" and "Insert value at the beginning of existing line")
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Write-RegistryValue -KeyPath 'HKEY_CURRENT_USER\SOFTWARE\CANCOM' -ValueName DemoValue -Value 'Lorem Ipsum' -ValueKind String -Action Set -Context User
	.EXAMPLE
		Write-RegistryValue -KeyPath 'HKEY_CURRENT_USER\SOFTWARE\CANCOM' -ValueName DemoValue -Value 'Loren Ipsum' -ValueKind String -Action Prepend -AllowDuplicates -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Write-RegistryValue.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][string]$ValueName = $null,
		[Parameter(Mandatory=$false)][string]$Value = $null,
		[Parameter(Mandatory=$false)][ValidateSet("String", "ExpandString", "Binary")][Microsoft.Win32.RegistryValueKind]$ValueKind = "String",
		[Parameter(Mandatory=$false)][ValidateSet("Set", "Remove", "Append", "Prepend")][string]$Action = "Set",
		[Parameter(Mandatory=$false)][switch]$AllowDuplicates = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		$key = $null;
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$key = Get-PdRegistryKey -Path $KeyPath -Create -Wow64:$Wow64;
			
			Write-Log -Message "Registry key '$($key.Name)', value '$($ValueName)'." -Source ${CmdletName};
			Write-Log -Message "Action '$($Action)' - data: '$($Value)' [$($ValueKind)]." -Source ${CmdletName};
			
			$current = $null;
			$valueData = $null;
			[Microsoft.Win32.RegistryValueKind]$valueDataKind = $ValueKind;
			$update = $false;

			if ($ValueKind -eq "Binary")
			{
				if ($Action -ne "Set") { throw "Unsupported action '$($Action)' for value kind '$($ValueKind)'."; }
				
				if ($Value -eq $null) { $Value = ""; }
				
				$update = $true;

				$parts = $Value.Split(":", 2);
				if ($parts.Count -gt 1)
				{
					$valueDataKind = $parts[0];
					$valueData = $parts[1];
				}
				else
				{
					$valueData = $parts[0];
				}
				
				if ($valueData.IndexOf(",") -gt 0)
				{
					$valueData = [byte[]]($valueData.Split(", ", [System.StringSplitOptions]::RemoveEmptyEntries) | % { [byte]::Parse($_, "HexNumber") });
				}
				else
				{
					$valueData = [Convert]::FromBase64String($valueData);
				}
			}
			elseif (($ValueKind -eq "String") -or ($ValueKind -eq "ExpandString"))
			{
				[string]$current = $key.GetValue($ValueName, $null, [Microsoft.Win32.RegistryValueOptions]::DoNotExpandEnvironmentNames);
				Write-Log -Message "Current data: '$($current)'." -Source ${CmdletName};
				
				if ((Test-RedirectWow64 $Wow64) -and (($Action -eq "Set") -or ($Action -eq "Prepend")))
				{
					$match = [regex]::Match($Value, '^(?<a>%(ProgramFiles|commonprogramfiles))(?<b>%.*)$', "None"); # case sensitive match required
					if ($match.Success)
					{
						$wow64Value = "$($match.Groups['a'].Value)(x86)$($match.Groups['b'].Value)";
						Write-Log -Message "Changing value '$($Value)' to '$($wow64Value)' due to WOW64." -Source ${CmdletName};
						$Value = $wow64Value;
					}
				}

				if (($Action -eq "Set") -or [string]::IsNullOrEmpty($Action))
				{
					$update = $true;
					$valueData = $Value;
				}
				elseif ($Action -eq "Remove")
				{
					$valueData = [regex]::Replace($current, [regex]::Escape($Value), "", "IgnoreCase");
					$update = ($valueData -cne $current);
				}
				elseif ($Action -eq "Append")
				{
					$update = ($AllowDuplicates -or ($current.IndexOf($Value, [System.StringComparison]::OrdinalIgnoreCase) -lt 0));
					if ($update) { $valueData = ($current + $Value); }
				}
				elseif ($Action -eq "Prepend")
				{
					$update = ($AllowDuplicates -or ($current.IndexOf($Value, [System.StringComparison]::OrdinalIgnoreCase) -lt 0));
					if ($update) { $valueData = ($Value + $current); }
				}
				else { throw "Unsupported action '$($Action)'."; }
			}
			else { throw "Unsupported value kind '$($ValueKind)'."; }
			
			if ($update)
			{
				if (($valueData -is [byte[]]) -and ($valueDataKind -ne $ValueKind))
				{
					Write-Log -Message "Write data bytes ($($valueData.Count)) [$($valueDataKind)] to value '$($ValueName)' of registry key '$($key.Name)'." -Source ${CmdletName};
					[PSPD.API]::WriteRegistryValue($key, $ValueName, [int]$valueDataKind, $valueData);
				}
				else
				{
					Write-Log -Message "Write data '$($valueData)' [$($ValueKind)] to value '$($ValueName)' of registry key '$($key.Name)'." -Source ${CmdletName};
					$key.SetValue($ValueName, $valueData, $ValueKind);
				}
			}
			else
			{
				Write-Log -Message "No update required of value '$($ValueName)' of registry key '$($key.Name)'." -Source ${CmdletName};
			}
		}
		catch
		{
			$failed = "Failed to write registry value '$ValueName' [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($key -ne $null)
			{
				$key.Close();
				$key = $null;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Write-RegistryDWord {
	<#
	.SYNOPSIS
		Changes or adds a value of type DWord
	.DESCRIPTION
		Changes or adds a value of type DWord (= double word, 4 bytes) in the registry, existing entries in the registry are updated.
	.PARAMETER KeyPath
		The Path to the Registry Key.
	.PARAMETER ValueName
		The name of the value
	.PARAMETER Value
		Optionally the default value of the key or the name of another value.
	.PARAMETER Action
		Defines how to handle the contents of a possibly existing value
		- Set: The content is set to the specified value and any existing value is overwritten.
		- Add: If a value already exists, the specified value is added to the existing content.
		- Subtract: If a value already exists, the specified value is subtracted from the existing content.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Write-RegistryDWord -KeyPath 'HKEY_CURRENT_USER\Control Panel\Desktop' -ValueName ScreenSaveUsePassword -Value 1 -Action Set -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Write-RegistryDWord.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][string]$ValueName = $null,
		[Parameter(Mandatory=$false)][string]$Value = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Set", "Add", "Subtract")][string]$Action = "Set",
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		$key = $null;
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$key = Get-PdRegistryKey -Path $KeyPath -Create -Wow64:$Wow64;

			$valueKind = [Microsoft.Win32.RegistryValueKind]::DWord;
			Write-Log -Message "Registry key '$($key.Name)', value '$($ValueName)'." -Source ${CmdletName};
			Write-Log -Message "Action '$($Action)' - data: '$($Value)' [$($valueKind)]." -Source ${CmdletName};
			
			[decimal]$data = $(if ($Value -match "^0x") {[UInt32]$Value} else {$Value});
			if (($data -lt 0) -or ($data -gt [UInt32]::MaxValue))
			{
				while ($data -lt 0) { $data += ([UInt32]::MaxValue + [decimal]1); }
				if ($data -gt [UInt32]::MaxValue) { $data = [UInt32]::MaxValue; }
				Write-Log -Message "Adjusted data: '$($data)'." -Source ${CmdletName};
			}
			
			[UInt32]$valueData = 0;
			if (($Action -eq "Set") -or [string]::IsNullOrEmpty($Action))
			{
				$valueData = [UInt32]$data;
			}
			else
			{
				[Int32]$int = $key.GetValue($ValueName, 0);
				[decimal]$current = [System.BitConverter]::ToUInt32([System.BitConverter]::GetBytes([Int32]$int), 0);
				Write-Log -Message "Current data: '$($current)'." -Source ${CmdletName};
				
				if ($Action -eq "Add")
				{
					[decimal]$result = ($current + $data);
					while ($result -gt [UInt32]::MaxValue) { $result -= ([UInt32]::MaxValue + [decimal]1); }
					$valueData = $result;
				}
				elseif ($Action -eq "Subtract")
				{
					[decimal]$result = ($current - $data);
					if ($result -lt 0) { $result = 0; }
					$valueData = $result;
				}
				else { throw "Unsupported action '$($Action)'."; }
			}
			
			Write-Log -Message "Write data '$($valueData)' [$($valueKind)] to value '$($ValueName)' of registry key '$($key.Name)'." -Source ${CmdletName};
			[Int32]$int = [System.BitConverter]::ToInt32([System.BitConverter]::GetBytes([UInt32]$valueData), 0);
			$key.SetValue($ValueName, $int, $valueKind); # [Microsoft.Win32.RegistryValueKind]::DWord: [Int32]::MinValue .. [Int32]::MaxValue
		}
		catch
		{
			$failed = "Failed to write registry value '$ValueName' [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($key -ne $null)
			{
				$key.Close();
				$key = $null;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Write-RegistryMultiString {
	<#
	.SYNOPSIS
		Changes Multi_SZ in the registry.
	.DESCRIPTION
		Changes or adds a value of type Multi_SZ in the registry. This command can be used to create, delete, and modify multi_SZ strings.
	.PARAMETER KeyPath
		Select the key in which a value is to be set.
	.PARAMETER ValueName
		Optionally the default value of the key or the name of another value.
	.PARAMETER Value
		The value or values to be entered in the MULTI_SZ value.
	.PARAMETER Action
		- Set: Replaces values
		- SetAt: Replaces values from position given in position parameter
		- Prepend: Insert values at the beginning
		- Append: Insert values at the end
		- InsertAscending: Insert values in alphabetical order
		- InsertDescending: Insert values in reverse alphabetical order
		- InsertAt: Insert values at position given in position parameter
		- Remove: Delete values from existing string
	.PARAMETER Position
		Defines the position when using 'InsertAt' or 'SetAt' Action
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Write-RegistryMultiString -KeyPath 'HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\NlaSvc' -ValueName DependOnService -Action Set -Value @('NTDS','DNS') -Context Computer
	.EXAMPLE
		Write-RegistryMultiString -KeyPath 'HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\NlaSvc' -ValueName DependOnService -Action Set -Value @('NTDS','DNS') -Context Computer
	.EXAMPLE
		Write-RegistryMultiString -KeyPath 'HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\NlaSvc' -ValueName DependOnService -Action InsertAt -Position 15 -Value @('NTDS','DNS') -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Write-RegistryMultiString.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][string]$ValueName = $null,
		[Parameter(Mandatory=$false)][string[]]$Value = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Set", "SetAt", "Prepend", "Append", "InsertAscending", "InsertDescending", "InsertAt", "Remove")][string]$Action = "Set",
		[Parameter(Mandatory=$false)][string]$Position = $null,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		function displayData([string[]]$data) { return $(if ($data.Count -gt 0) {"('$([string]::Join("', '", $data))')"} else {"()"}); }
		
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		$key = $null;
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$key = Get-PdRegistryKey -Path $KeyPath -Create -Wow64:$Wow64;
			
			$valueKind = [Microsoft.Win32.RegistryValueKind]::MultiString;
			Write-Log -Message "Registry key '$($key.Name)', value '$($ValueName)'." -Source ${CmdletName};
			$actionInfo = $(if (($Action -eq "SetAt") -or ($Action -eq "InsertAt")) { "$($Action)($($Position))" } else { $Action });
			Write-Log -Message "Action '$($actionInfo)' - data: $(displayData $Value) [$($valueKind)]." -Source ${CmdletName};
			
			[string[]]$valueData = @();
			if (($Action -eq "Set") -or [string]::IsNullOrEmpty($Action))
			{
				$valueData = $Value;
			}
			else
			{
				$current = $key.GetValue($ValueName, [string[]]@());
				[int]$index = [int]$Position;
				Write-Log -Message "Current data: $(displayData $current)." -Source ${CmdletName};
				
				if ($Action -eq "SetAt")
				{
					if (($index -lt 0) -or ($index -ge $current.Count)) { throw "The position '$($index)' is out of range for action '$($Action)' - accept: 0..$($current.Count)." }
					elseif (($index + $Value.Count) -gt $current.Count) { throw "Too many strings ($Value.Count) to replace in current strings ($current.Count) at position ($index) for action '$($Action)' - accept: 0..$($current.Count - $index)." }
					
					$valueData = $current;
					for ($i = 0; $i -lt $Value.Count; $i++)
					{
						$valueData[$i + $index] = $Value[$i];
					}
				}
				elseif ($Action -eq "Prepend")
				{
					$valueData = ($Value + $current);
				}
				elseif ($Action -eq "Append")
				{
					$valueData = ($current + $Value);
				}
				elseif ($Action -eq "InsertAscending")
				{
					# not: $valueData = @(($current + $Value) | sort -CaseSensitive)
					# ---> keep order of current items, insert sorted items ascending, sort by character ordinals
					[string[]]$insert = $Value;
					$comparer = [System.StringComparer]::Ordinal;
					[array]::Sort($insert, $comparer);
					$valueData = [string[]](@($current | % { $sz = $_; $append = @($insert | where { $comparer.Compare($_, $sz) -le 0 }); $insert = @($insert | where { $comparer.Compare($_, $sz) -gt 0 }); ($append + $sz); }) + $insert);
				}
				elseif ($Action -eq "InsertDescending")
				{
					# not: $valueData = @(($current + $Value) | sort -Descending -CaseSensitive)
					# ---> sort current items, insert sorted items ascending, sort by character ordinals
					$valueData = [string[]]($current + $Value);
					[array]::Sort($valueData, [System.StringComparer]::Ordinal);
					[array]::Reverse($valueData);
				}
				elseif ($Action -eq "InsertAt")
				{
					if (($index -lt 0) -or ($index -gt $current.Count)) { throw "The position '$($index)' is out of range for action '$($Action)' - accept: 0..$($current.Count)." }
					
					$before = [string[]]$(if ($index -gt 0) { $current[0..($index - 1)] } else { @() });
					$after = [string[]]$(if ($index -lt $current.Count) { $current[$index..($current.Count - 1)] } else { @() });
					$valueData = ($before + $Value + $after);
				}
				elseif ($Action -eq "Remove")
				{
					# $valueData = @($current | where { $_ -notin $Value });
					$valueData = @($current | where { $Value -notcontains $_ });
				}
				else { throw "Unsupported action '$($Action)'."; }
			}
			
			Write-Log -Message "Write data $(displayData $valueData) [$($valueKind)] to value '$($ValueName)' of registry key '$($key.Name)'." -Source ${CmdletName};
			$key.SetValue($ValueName, $valueData, $valueKind);
		}
		catch
		{
			$failed = "Failed to write registry value '$ValueName' [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($key -ne $null)
			{
				$key.Close();
				$key = $null;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Write-RegistryQWord {
	<#
	.SYNOPSIS
		Changes Registry QWord
	.DESCRIPTION
		Changes or adds a value of type QWord (= quad word, 8 bytes) in the registry, existing entries in the registry are updated.
	.PARAMETER KeyPath
		The Path to the Registry Key.
	.PARAMETER ValueName
		The name of the value
	.PARAMETER Value
		Optionally the default value of the key or the name of another value.
	.PARAMETER Action
		Defines how to handle the contents of a possibly existing value
		- Set: The content is set to the specified value and any existing value is overwritten.
		- Add: If a value already exists, the specified value is added to the existing content.
		- Subtract: If a value already exists, the specified value is subtracted from the existing content.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Write-RegistryQWord -KeyPath 'HKEY_CURRENT_USER\SOFTWARE\Microsoft\Siuf\Rules' -ValueName 'PeriodInNanoSeconds' -Value '8640000000' -Action Set -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Write-RegistryQWord.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][string]$ValueName = $null,
		[Parameter(Mandatory=$false)][string]$Value = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Set", "Add", "Subtract")][string]$Action = "Set",
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		$key = $null;
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$key = Get-PdRegistryKey -Path $KeyPath -Create -Wow64:$Wow64;

			$valueKind = [Microsoft.Win32.RegistryValueKind]::QWord;
			Write-Log -Message "Registry key '$($key.Name)', value '$($ValueName)'." -Source ${CmdletName};
			Write-Log -Message "Action '$($Action)' - data: '$($Value)' [$($valueKind)]." -Source ${CmdletName};

			[decimal]$data = $(if ($Value -match "^0x") {[UInt64]$Value} else {$Value});
			if (($data -lt 0) -or ($data -gt [UInt64]::MaxValue))
			{
				while ($data -lt 0) { $data += ([UInt64]::MaxValue + [decimal]1); }
				if ($data -gt [UInt64]::MaxValue) { $data = [UInt64]::MaxValue; }
				Write-Log -Message "Adjusted data: '$($data)'." -Source ${CmdletName};
			}
			
			[UInt64]$valueData = 0;
			if (($Action -eq "Set") -or [string]::IsNullOrEmpty($Action))
			{
				$valueData = [UInt64]$data;
			}
			else
			{
				[Int64]$int = $key.GetValue($ValueName, 0);
				[decimal]$current = [System.BitConverter]::ToUInt64([System.BitConverter]::GetBytes([Int64]$int), 0);
				Write-Log -Message "Current data: '$($current)'." -Source ${CmdletName};

				if ($Action -eq "Add")
				{
					[decimal]$result = ($current + [decimal]$data);
					while ($result -gt [UInt64]::MaxValue) { $result -= ([UInt64]::MaxValue + [decimal]1); }
					$valueData = $result;
				}
				elseif ($Action -eq "Subtract")
				{
					[decimal]$result = ($current - [decimal]$data);
					if ($result -lt 0) { $result = 0; }
					$valueData = $result;
				}
				else { throw "Unsupported action '$($Action)'."; }
			}
			
			Write-Log -Message "Write data '$($valueData)' [$($valueKind)] to value '$($ValueName)' of registry key '$($key.Name)'." -Source ${CmdletName};
			[Int64]$int = [System.BitConverter]::ToInt64([System.BitConverter]::GetBytes([UInt64]$valueData), 0);
			$key.SetValue($ValueName, $int, $valueKind); # [Microsoft.Win32.RegistryValueKind]::QWord: [Int64]::MinValue .. [Int64]::MaxValue
		}
		catch
		{
			$failed = "Failed to write registry value '$ValueName' [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($key -ne $null)
			{
				$key.Close();
				$key = $null;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Write-RegistryKey {
	<#
	.SYNOPSIS
		Modifies or adds a key in the registry.
	.DESCRIPTION
		Modifies or adds a key in the registry, existing entries in the registry are updated.
	.PARAMETER KeyPath
		Registry Key.
	.PARAMETER SubKeyName
		Name of the subkey to be set.
	.PARAMETER Repair
		Use to ensure that the specified subkey is recreated as needed during a repair installation.
	.PARAMETER Uninstall
		Use to ensure that the specified subkey is deleted during uninstallation.
	.PARAMETER Recurse
		The options enabled under Actions are also applied to the values of any existing subkeys.
	.PARAMETER PreventUninstall
		Prevents script from undoing configuration if running on 'uninstall mode'
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Write-RegistryKey -KeyPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Run' -SubkeyName PackagingPowerBench -Repair -Uninstall
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Write-RegistryKey.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][string]$SubKeyName = "",
		[Parameter(Mandatory=$false)][switch]$Repair = $false,
		[Parameter(Mandatory=$false)][switch]$Uninstall = $false,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		$subKey = $null;
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }

			$subKeyPath = $(if ([string]::IsNullOrEmpty($SubKeyName)) { $KeyPath } else { "$($KeyPath)\$($SubKeyName)" });

			$pdc = Get-PdContext;
			$installMode = $pdc.InstallMode;
			if (($installMode -eq "InstallUserPart") -and ![string]::IsNullOrEmpty($pdc.UserPartInstallMode))
			{
				$installMode = $pdc.UserPartInstallMode;
				Write-Log -Message "Installation mode for user part: '$($installMode)'." -Source ${CmdletName};
			}
				
			if (Test-ReverseMode)
			{
				Write-Log -Message "$($installMode) mode (reverse): Deleting registry key '$($subKeyPath)'." -Source ${CmdletName};
				$subKey = Get-PdRegistryKey -Path $subKeyPath -Writable -Wow64:$Wow64 -AcceptNull;
				if ($subKey -eq $null)
				{
					Write-Log -Message "Done: Key already deleted." -Source ${CmdletName};
				}
				elseif (($installMode -eq "Uninstall") -and !$Uninstall)
				{
					Write-Log -Message "Not deleting: Key is not intended for $($installMode) mode (-Uninstall is not set)." -Source ${CmdletName};
				}
				elseif ($Recurse)
				{
					Write-Log -Message "Deleting (recursively): Key exists." -Source ${CmdletName};
					$subKey.DeleteSubKeyTree(""); # "" -> $subKeyPath
				}
				elseif ($subKey.SubKeyCount -gt 0)
				{
					Write-Log -Message "Not deleting: Key contains subkeys ($($subKey.SubKeyCount): $([string]::Join(', ', $subKey.GetSubKeyNames())))." -Source ${CmdletName};
					if ($subKey.ValueCount -gt 0)
					{
						Write-Log -Message "Deleting values: Key contains values ($($subKey.ValueCount))." -Source ${CmdletName};
						$subKey.GetValueNames() | % { $subKey.DeleteValue($_); }
					}
				}
				else
				{
					Write-Log -Message "Deleting (no subkeys): Key exists." -Source ${CmdletName};
					$subKey.DeleteSubKey(""); # "" -> $subKeyPath
				}
				
				return; # exit from reverse mode
			}
			
			Write-Log -Message "$($installMode) mode: Writing registry key '$($subKeyPath)'." -Source ${CmdletName};
			$subKey = Get-PdRegistryKey -Path $subKeyPath -Wow64:$Wow64 -AcceptNull;
			if ($subKey -ne $null)
			{
				Write-Log -Message "Done: Key already exists." -Source ${CmdletName};
			}
			elseif (($installMode -eq "Repair") -and !$Repair)
			{
				Write-Log -Message "Not recreating: Key is not intended for $($installMode) mode (-Repair is not set)." -Source ${CmdletName};
			}
			else
			{
				Write-Log -Message "Creating: Key does not exist." -Source ${CmdletName};
				$subKey = Get-PdRegistryKey -Path $subKeyPath -Create -Wow64:$Wow64;
			}
		}
		catch
		{
			$failed = "Failed to write registry key '$SubKeyName' [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($subKey -ne $null)
			{
				$subKey.Close();
				$subKey = $null;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-InstallMode {
	<#
	.SYNOPSIS
		Check for installation or execution mode
	.DESCRIPTION
		Checks the mode the current installation is executed with.
		Installation mode: user portion or computer portion
		Execution mode: repair, reinstallation, modification, update, uninstallation
		Due to compatibility with older scripts, a check for reinstallation will also return "true" if the actual execution mode is modification or update.
		In a script with more than one CheckInstallMode, make sure to check for modification or update before checking for reinstallation.
	.PARAMETER InstallMode
		- InstallUserPart: Check for user portion.
		- InstallComputerPart: Check for computer portion.
		- Repair: Check for repair.
		- Reinstall: Check for reinstallation.
		- Modify: Check for modification
		- Update: Check for Update.
		- Uninstall: Check for uninstallation.
	.EXAMPLE
		Test-InstallMode -InstallMode Reinstall
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-InstallMode.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][ValidateSet("InstallUserPart","InstallComputerPart","Install","Repair","Reinstall","Modify","Update","Uninstall")][string]$InstallMode = "Install"
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$result = $false;
			$note = $null;
			
			$pdc = Get-PdContext;
			$currentInstallMode = $pdc.InstallMode;
			if ([string]::IsNullOrEmpty($currentInstallMode))
			{
				$currentInstallMode = "Install";
				Write-Log -Message "No installation mode set - assume: '$($currentInstallMode)'." -Source ${CmdletName};
			}

			if (($InstallMode -eq "InstallComputerPart") -and ($currentInstallMode -ne "InstallUserPart"))
			{
				$note = $(if ($currentInstallMode -ne $InstallMode) { " ('$($InstallMode)' part of '$($currentInstallMode)')" } else { $null });
				$result = $true;
			}
			elseif ($pdc.IncludeUserPart -and ![string]::IsNullOrEmpty($pdc.UserPartInstallMode))
			{
				$currentInstallMode = $pdc.UserPartInstallMode;
				Write-Log -Message "Installation mode for user part: '$($currentInstallMode)'." -Source ${CmdletName};
			}
			
			if ($currentInstallMode -eq $InstallMode)
			{
				$note = $null;
				$result = $true;
			}
			elseif (($InstallMode -eq "Install") -and (($currentInstallMode -eq "InstallUserPart") -or ($currentInstallMode -eq "InstallComputerPart")))
			{
				# special case: combined mode "Install"
				$note = " ('$($currentInstallMode)' part of '$($InstallMode)')";
				$result = $true;
			}
			elseif (($InstallMode -eq "InstallUserPart") -and $pdc.IncludeUserPart)
			{
				# special case: included user part
				$note = " ('$($InstallMode)' part of '$($currentInstallMode)')";
				$result = $true;
			}
			elseif (($InstallMode -eq "Reinstall") -and (($currentInstallMode -eq "Modify") -or ($currentInstallMode -eq "Update")))
			{
				# note: if "Treat Update and Modification as Reinstallation" is set (default), then "Modify" and "Update" return $true for "Reinstall"
				#       ... may define and check any property of $script:PdContext here, like " if ($pdc.TreatUpdateAndModificationAsReinstallation -and ...)
				$note = " (treating '$($currentInstallMode)' as '$($InstallMode)')";
				$result = $true;
			}
			
			Write-Log -Message "Current installation mode '$($currentInstallMode)' matches '$($InstallMode)': $($result)$($note)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test for InstallMode [$InstallMode]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-RegistryKey {
	<#
	.SYNOPSIS
		Existence of a registry key
	.DESCRIPTION
		Checks if a registry key exists.
	.PARAMETER KeyPath
		Registry Key.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.EXAMPLE
		Test-RegistryKey -KeyPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Run'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-RegistryKey.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$key = Get-PdRegistryKey -Path $KeyPath -AcceptNull -Wow64:$Wow64;
			$result = ($key -ne $null);
			Write-Log -Message "Registry key '$($KeyPath)' exists: $($result)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test for registry key [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-RegistryValue {
	<#
	.SYNOPSIS
		Existence of a registry value
	.DESCRIPTION
		Checks if a specific registry value exists (string, binary or DWORD value). The value name is the parameter
	.PARAMETER KeyPath
		Registry Key.
	.PARAMETER ValueName
		Value name.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.EXAMPLE
		Test-RegistryValue -KeyPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Run' -ValueName SecurityHealth
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-RegistryValue.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][string]$ValueName = $null,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$result = $false;
			
			$key = Get-PdRegistryKey -Path $KeyPath -AcceptNull -Wow64:$Wow64;
			if ($key -eq $null) { $result = $false; }
			else { $result = (@($key.GetValueNames() | where { $_ -eq $ValueName }).Count -gt 0); }
			
			Write-Log -Message "Registry value '$($ValueName)' of key '$($KeyPath)' exists: $($result)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test for registry value [$ValueName][$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-RegistryValue {
	<#
	.SYNOPSIS
		Checks a Registry Key value
	.DESCRIPTION
		Compares the value of a Registry key within an IF statement. Supported parameters are: the name of the key, an operator and the value.
		Moreover, it can be selected whether on 64-bit computers the 64-bit branch of the registry should be used or not.
	.PARAMETER KeyPath
		Registry Key.
	.PARAMETER ValueName
		Value name.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.EXAMPLE
		(Get-RegistryValue -KeyPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Run' -ValueName SecurityHealth) -eq '%windir%\system32\SecurityHealthSystray.exe'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Get-RegistryValue.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][string]$ValueName = $null,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		$key = $null;
		try
		{
			$value = $null;
			
			$key = Get-PdRegistryKey -Path $KeyPath -AcceptNull -Wow64:$Wow64;
			if ($key -eq $null) { $value = $null; }
			else { $value = $key.GetValue($ValueName, $null, [Microsoft.Win32.RegistryValueOptions]::DoNotExpandEnvironmentNames); }
			
			Write-Log -Message "Registry value '$($ValueName)' of key '$($KeyPath)': '$($value)'." -Source ${CmdletName};
			return $value; 
		}
		catch
		{
			$failed = "Failed to get registry value [$ValueName][$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($key -ne $null)
			{
				$key.Close();
				$key = $null;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Exit-Package {
	<#
	.SYNOPSIS
		Exit installation
	.DESCRIPTION
		Exits the current installation package with the specified status and optional comment.
	.PARAMETER Status
		Defines the status with which the package is terminated.
		- Done: 	This option causes the status information for successful installation to be written to the registry.
		- Undone: 	This option causes status information about package execution to be written to the registry, 
					but the IsInstalled value is set to 0 and the Status value is set to Undone to document that the package is not considered installed or executed.
		- UndoneContinueParentScript: Some as undone, but proceed executing parent script.
		- Failed: 	This option causes status information about package execution to be written to the registry, but the IsInstalled value is set to 0 and the Status value is set 
					to Failed to document that the package is considered failed and not executed. For packages with a user part, the entries for the Active Setup are not created. 
	.PARAMETER Message
		Comment on the termination of the package, which is recorded in the log file and in the registry's StatusMessage value.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Exit-Package -Status Done
	.EXAMPLE
		Exit-Package -Status Failed -Message 'Failed with errors'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Exit-Package.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][ValidateSet("Done", "Undone", "UndoneContinueParentScript", "Failed")][string]$Status = "Done",
		[Parameter(Mandatory=$false)][string]$Message = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $null) { return; }

			$pdc = Get-PdContext;
			$pdc.Status = $Status;
			$pdc.StatusMessage = $Message;

			Write-Log -Message "Setting exit status code: $($pdc.Status)." -Source ${CmdletName};
			if (![string]::IsNullOrEmpty($Message)) { Write-Log -Message "Setting exit status message: '$($pdc.StatusMessage)'." -Source ${CmdletName}; }
			exit $(if ($Status -eq "Done") {0} else {1}); # exit calling script
		}
		catch
		{
			$failed = "Failed to set exit status '$Status'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Stop-PdProcess {
	<#
	.SYNOPSIS
		Terminates process
	.DESCRIPTION
		Terminates one or more processes. This command scrolls through the list of all running 
		processes and ends the corresponding process(es) or application(s). 
		Unsaved process data may be lost!
	.PARAMETER Name
		Name of the process executable or the window title of the process(es) to be terminated
	.PARAMETER NameOf
		Criterion by which the process or processes to be terminated should be identified.
		- File: The specified criterion is applied to the name of the process executable.
		- Window: The specified criterion is applied to the window title of the process(es).
	.PARAMETER Select
		Defines which processes are to be terminated.
		- FirstMatch: Only the first process found according to the criteria is terminated.
		- AllMatches: All processes found that match the criteria are terminated.
	.PARAMETER SupportUninstall
		If this option is enabled, the processes are also terminated when the package is uninstalled.
	.PARAMETER IncludeChildren
		If this option is activated, subordinate processes (so-called child processes) are also terminated.
	.PARAMETER PreventUninstall
		Prevents Uninstall-MsiProduct from beeing executed when running in 'uninstall mode'
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Stop-PdProcess -Name 'notepad*.exe' -NameOf File -Select AllMatches -IncludeChildren -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Stop-PdProcess.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][ValidateSet("File", "Window")][string]$NameOf = "File",
		[Parameter(Mandatory=$false)][ValidateSet("FirstMatch", "AllMatches")][string]$Select = "FirstMatch",
		[Parameter(Mandatory=$false)][switch]$SupportUninstall = $false,
		[Parameter(Mandatory=$false)][switch]$IncludeChildren = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse:$SupportUninstall -PreventUninstall:$PreventUninstall) { return; }

			$filter = $(if ($NameOf -eq "Window") {{ ($_.MainWindowTitle -like $Name) }} else {{ ($_.ProcessName -like $Name) -or ($_.MainModule.ModuleName -like $Name) }});
			
			$processes = @(Get-Process | where $filter);
			Write-Log -Message "Found processes '$($Name)' [$($NameOf)]: $($processes.Count)." -Source ${CmdletName};
			if ($processes.Count -eq 0) { return; }
			
			if (($processes.Count -gt 1) -and ($Select -eq "FirstMatch"))
			{
				$processes = @($processes[0]);
				Write-Log -Message "Terminating only first match: '$($processes[0].ProcessName)' [ID $($processes[0].Id)]." -Source ${CmdletName};
			}
			
			# (1) note: e.g. some browser-applications restart processes which are terminated after a short delay - so, terminating the list of processes as quickly as possible may prevent this behaviour
			$listItemIndent = "`r`n`t* ";
			$parentIds = @($processes | % { $_.Id });
			Write-Log -Message ("Terminating processes ($($processes.Count)):" + $listItemIndent + [string]::Join($listItemIndent, @($processes | % { "'$($_.ProcessName)' [ID $($_.Id)]" }))) -Source ${CmdletName};
			$processes | % { if (!$_.HasExited) { $_.Kill(); } } # (1)
			if ($IncludeChildren)
			{
				$childIds = @(Get-WmiObject -Class Win32_Process -Property ProcessId, ParentProcessId | where { $parentIds -contains $_.ParentProcessId } | % { $_.ProcessId }); 
				if ($childIds.Count -gt 0)
				{
					$childProcesses = @(Get-Process | where { $childIds -contains $_.Id });
					Write-Log -Message ("Terminating child processes ($($childProcesses.Count)):" + $listItemIndent + [string]::Join($listItemIndent, @($childProcesses | % { "'$($_.ProcessName)' [ID $($_.Id)]" }))) -Source ${CmdletName};
					$childProcesses | % { if (!$_.HasExited) { $_.Kill(); } } # (1)
				}
				else
				{
					Write-Log -Message "No child processes found to terminate." -Source ${CmdletName};
				}
			}
		}
		catch
		{
			$failed = "Failed to kill process '$Name' [$NameOf]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Install-PnpDevice {
	<#
	.SYNOPSIS
		Installing Plug & Play devices
	.DESCRIPTION
		Every package for installing drivers contains the command InstallPnpDevices. 
		This command will search for Drivers in the current directory as well the subdirectories with the names 'Extern$', 'Files' and 'SupportedFiles'
	.EXAMPLE
		Install-PnpDevice
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Install-PnpDevice.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }
			
			$dirs = @("Extern`$", "Files", "SupportFiles", ".");
			$infDir = $($dirs | % { Expand-Path "$($_)\drv" -Wow64:$Wow64 } | where { [System.IO.Directory]::Exists($_) } | select -First 1);
			if ([string]::IsNullOrEmpty($infDir)) { throw "Drivers directory 'drv' not found in '$((Get-Location).ProviderPath)' [$([string]::Join(', ', $dirs))]."; }

			$files = @([System.IO.Directory]::GetDirectories($infDir) | % { [System.IO.Directory]::GetFiles($_, "*.inf"); });
			$files | % { Invoke-PnpUtilExe -Operation "-i -a" -InfName $_; }
			Write-Log -Message "INF-files processed: $($files.Count)" -Source ${CmdletName};
		}
		catch
		{
			$failed = "Failed to install PnP devices";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-MsiLogPath([string]$product, [string]$action = "Install", [switch]$verify = $false)
{
	$dir = $(if ($configToolkitCompressLogs) {$logTempFolder} else {$configMSILogDir});
	if ($verify -and ![System.IO.Directory]::Exists($dir))
	{
		Write-Log -Message "Creating MSI-Log-directory '$($dir)'." -Source ${CmdletName};
		$void = [System.IO.Directory]::CreateDirectory($dir);
	}
	
	$path = [System.IO.Path]::Combine($dir, (& $script:validateFileName "PSPD_MSI_$($product)_$($action).log"));
	Write-Log -Message "Writing MSI-Log to '$($path)'." -Source ${CmdletName};
	return $path;
}

function Install-MsiProduct {
	<# 
	.SYNOPSIS
		Installs an MSI-based software.
	.DESCRIPTION
		Installs an MSI-based software.
	.PARAMETER Path
		Specifies the Windows Installer package (*.msi file) to be installed, including path
	.PARAMETER InstallationMode
		- Install: The product is fully installed at package runtime.
		- Advertise: Only shortcuts are created and the product is actually installed the first time it is called by a user.
		- AsProject: Use 'Advertise' if running in User or UserAsService Context, else Install. 
	.PARAMETER Target
		- AllUsers: For all users on this computer.
		- CurrentUser: Install for current user only.
	.PARAMETER Options
		Custom Parameters for MSI File. Prefix your parameters with params:yourparameter
	.PARAMETER VerifyInstallation
		Append Transform file
	.PARAMETER SecureParameters
		If this option is enabled, logging of the command line in the log file is prevented. 
		Useful if confidential values (e.g. passwords, license numbers or similar) are to be passed to the call via the parameters.
	.PARAMETER LogLevel
		- DEBUG: The Windows Installer log file contains maximum detailed entries.
		- NORMAL: The Windows Installer log file contains the normal level of detail.
	.PARAMETER ResultVariable
		Name of a variable that contains the exit code of the Windows Installer.
	.PARAMETER PreventUninstall
		Prevents Uninstall-MsiProduct from beeing executed when running in 'uninstall mode'
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Install-MsiProduct -Path 'D:\Downloads\blender-2.93.5-windows-x64.msi' -InstallationMode Install -Target CurrentUser -ResultVariable _returnCode -Context User
	.EXAMPLE
		Install-MsiProduct -Path 'D:\Downloads\blender-2.93.5-windows-x64.msi' -InstallationMode Advertise -Target CurrentUser -Options 'params:REBOOT=ReallySuppress' -SecureParameters -ResultVariable _returnCode -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Install-MsiProduct.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][ValidateSet("Install", "Advertise", "AsProject")][string]$InstallationMode = "Install",
		[Parameter(Mandatory=$false)][ValidateSet("AllUsers", "CurrentUser")][string]$Target = "AllUsers",
		[Parameter(Mandatory=$false)][string[]]$Options = @(),
		[Parameter(Mandatory=$false)][switch]$VerifyInstallation = $false,
		[Parameter(Mandatory=$false)][switch]$SecureParameters = $false,
		[Parameter(Mandatory=$false)][ValidateSet("", "NORMAL", "DEBUG")][string]$LogLevel = "",
		[Parameter(Mandatory=$false)][string]$ResultVariable = $null,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		[hashtable]$displayParameters = $PSBoundParameters;
		if ($SecureParameters) { @("Options") | where { $displayParameters.ContainsKey($_) } | % { $displayParameters[$_] = "-hidden-" } }
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $displayParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }

			$Path = Expand-Path $Path;
			
			$productName = $null;
			$productCode = $null;
			try
			{
				$info = Get-MsiTableProperty -Path $Path;
				$productName = $info.ProductName;
				$productCode = $info.ProductCode;
			}
			catch
			{
				Write-Log -Message "Failed to get info from '$($Path)'." -Severity 2 -Source ${CmdletName};
				$productName = [System.IO.Path]::GetFilenameWithoutExtension($Path);
				$productCode = $null;
			}

			$parameters = $null;
			$patternParameters = '^\s*params:\s*(?<value>.*)$';
			$transforms = [string]::Join(";", @($Options | where { ![string]::IsNullOrEmpty($_) } | % { if ($_ -match $patternParameters) { $parameters = ($_ -replace $patternParameters, '${value}').Trim(); } else { $secure = $(if ($_.Trim().StartsWith("|")) {"|"} else {""}); "$($secure)$(Expand-Path $_.Trim(' |'))" } }))

			if ($InstallationMode -eq "AsProject") { $InstallationMode = $(if (($Context -eq "User") -or ($Context -eq "UserPerService")) {"Advertise"} else {"Install"}); }

			$action = $(if (Test-ReverseMode) {"Uninstall"} else {$InstallationMode});
			
			$allUsers = ($Target -eq "AllUsers");
			
			$arguments = $null;
			if ($action -eq "Uninstall") { $arguments = "/x `"$($Path)`""; } # /x <msi|guid>
			elseif ($action -eq "Advertise") { $arguments = "/j$($(if ($allUsers) {'m'} else {'u'})) `"$($Path)`""; } # /j{m|u} <msi>
			else { $arguments = "/i `"$($Path)`""; } # /i <msi> # ($action -eq "Install")
			
			if (![string]::IsNullOrEmpty($LogLevel))
			{
				$logOptions = $(if ($LogLevel -eq "DEBUG") {"*v"} else {"*vx"});
				$logPath = Get-MsiLogPath -product $productName -action $action -verify;
				$arguments += " /l$($logOptions) `"$($logPath)`"";
			}

			$arguments += " /qb-!";

			if (![string]::IsNullOrEmpty($transforms)) { $arguments += " TRANSFORMS=`"$($transforms)`""; }
			
			$arguments += " REBOOT=ReallySuppress";
			
			$arguments += " ALLUSERS=$(if ($allUsers) {'1'} else {'0'})";

			if (![string]::IsNullOrEmpty($parameters)) { $arguments += " $($parameters)"; }
			
			Write-Log -Message "Performing action '$($action)' for MSI-Product '$($Path)' [$($productName)] [$($productCode)]." -Source ${CmdletName};
			$status = Invoke-MsiExecExe -Arguments $arguments -PassThru -SecureParameters:$SecureParameters;
			Write-Log -Message "Result status is: $($status.ExitCode) [$($status.StatusText)] [Success: $($status.Success)] [Reboot: $($status.Reboot)]." -Source ${CmdletName};
			if (![string]::IsNullOrEmpty($ResultVariable)) { Set-PdVar -Name $ResultVariable -Value $status.ExitCode; }
			if ($status.Reboot) { Request-Reboot; }

			if ($status.Success -and $VerifyInstallation)
			{
				try
				{
					$productState = $null;
					$guid = [Guid]::Empty;
					if ([Guid]::TryParse([string]$productCode, [ref]$guid))
					{
						$productState = [PSPD.API]::GetMsiProductState($guid.ToString("B"));
					}
					else
					{
						Write-Log -Message "Using 'Get-InstalledApplication' for '$($productName)' [ProductCode '$($productCode)']." -Source ${CmdletName};
						$product = Get-InstalledApplication -Name $productName;
						$productState = $(if ($product -ne $null) {[PSPD.API+INSTALLSTATE]::DEFAULT} else {[PSPD.API+INSTALLSTATE]::UNKNOWN});
					}
					
					$expected = @{ "Install" = [PSPD.API+INSTALLSTATE]::DEFAULT; "Advertise" = [PSPD.API+INSTALLSTATE]::ADVERTISED; "Uninstall" = [PSPD.API+INSTALLSTATE]::UNKNOWN; }
					$check = $(if ($productState -eq $expected[[string]$action]) {"OK"} else {"expected: $($expected[[string]$action])"});

					Write-Log -Message "Installation state of '$($productName)' is: $($productState) - $($check)." -Source ${CmdletName};
				}
				catch
				{
					Write-Log -Message "Failed to determine installation state of '$($productName)' [$([string]$_)]." -Severity 2 -Source ${CmdletName};
				}
			}
		}
		catch
		{
			$failed = "Failed to process '$Path' [$InstallationMode]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-MsiProduct {
	<# 
	.SYNOPSIS
		Existence of an MSI Package.
	.DESCRIPTION
		Checks if a specific MSI Package is installed.
	.PARAMETER ProductCode
		The product GUID of the MSI file you want to uninstall.
	.PARAMETER DisplayName
		The Product name or Display name
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-MsiProduct -ProductCode "{3033FBAD-BA86-469B-8C6F-ECD41334BD4D}" -DisplayName 'blender 2.93.5' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-MsiProduct.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ProductCode,
		[Parameter(Mandatory=$false)][Alias("ProductName")][string]$DisplayName = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if ([string]::IsNullOrEmpty($DisplayName)) { $DisplayName = $ProductCode; }

			Write-Log -Message "Testing the installation state of '$($ProductCode)' [$($DisplayName)]." -Source ${CmdletName};

			$productState = [PSPD.API]::GetMsiProductState($ProductCode);
			$isInstalled = ($productState -eq [PSPD.API+INSTALLSTATE]::DEFAULT);
			Write-Log -Message "Installation state of '$($ProductCode)' [$($DisplayName)] is: $($productState) -> installed: $($isInstalled)." -Source ${CmdletName};

			return $isInstalled;
		}
		catch
		{
			$failed = "Failed to test MSI product '$ProductCode' [$DisplayName]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Uninstall-MsiProduct {
	<# 
	.SYNOPSIS
		Uninstalls an MSI patch.
	.DESCRIPTION
		Uninstalls an MSI patch.
	.PARAMETER ProductCode
		The product GUID of the MSI file you want to uninstall.
	.PARAMETER DisplayName
		The Product name or Display name
	.PARAMETER AddParameters
		Additional parameters that can be passed to the MSI uninstall command, given in the form of a comma-separated list of property=value pairs, for example REBOOT=ReallySuppress.
	.PARAMETER UiLevelFlags
		Add the value of the desired options to include the option in the repair. Start calculation with value of 0

		UI Default behavior
		- +1 Default: The installer chooses an appropriate level of the user interface itself.
		- +2 Silent: Completely automatic installation in the background in 'silent' mode.
		- +3 Basic: Basic progress and error handling.
		- +4 Reduced: User interface with suppressed wizards and dialog boxes.
		- +5 Full: User interface with assistants, dialogs, progress and errors.
		
		Special options
		- +128 = Show finish dialog
		- +64 = Only show progress
		- +32 = Hide cancellation button
	.PARAMETER UiLevelAsProject
		Adds Setup parameter '/qb-!'
	.PARAMETER ResultVariable
		Name of a variable that contains the exit code of the Windows Installer.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Uninstall-MsiProduct -ProductCode "{3033FBAD-BA86-469B-8C6F-ECD41334BD4D}" -DisplayName 'blender 2.93.5' -UiLevelFlags 1 -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Uninstall-MsiProduct.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ProductCode,
		[Parameter(Mandatory=$false)][Alias("ProductName")][string]$DisplayName = $null,
		[Parameter(Mandatory=$false)][string]$AddParameters = $null,
		[Parameter(Mandatory=$false)][string]$UiLevelFlags = "1",
		[Parameter(Mandatory=$false)][object]$UiLevelAsProject = $false,
		[Parameter(Mandatory=$false)][string]$ResultVariable = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if ([string]::IsNullOrEmpty($DisplayName)) { $DisplayName = $ProductCode; }

			function toBool($v) { [bool]$b = $false; if (($v -is [string]) -and ([bool]::TryParse($v, [ref]$b) -or [int]::TryParse($v, [ref]$b))) { return $b; } else { return [System.Management.Automation.LanguagePrimitives]::IsTrue($v); } }
			$UiLevelAsProject = (toBool $UiLevelAsProject);

			Write-Log -Message "Removing MSI-Product '$($ProductCode)' [$($DisplayName)]." -Source ${CmdletName};
			
			try
			{
				$productState = [PSPD.API]::GetMsiProductState($ProductCode);
				$done = ($productState -eq [PSPD.API+INSTALLSTATE]::UNKNOWN);
				Write-Log -Message "Installation state of '$($ProductCode)' is: $($productState)$(if($done) {' - done'})." -Source ${CmdletName};
				if ($done) { return; }
			}
			catch
			{
				Write-Log -Message "Failed to determine installation state of '$($ProductCode)' [$([string]$_)] - assume installed." -Severity 2 -Source ${CmdletName};
			}

			$action = "Uninstall";
			$allUsers = $true;
			$secureParameters = $false;
			$logLevel = "NORMAL";
			
			$arguments = "/x `"$($ProductCode)`""; # /x <msi|guid>
			
			if (![string]::IsNullOrEmpty($logLevel))
			{
				$logOptions = $(if ($logLevel -eq "DEBUG") {"*v"} else {"*vx"});
				$logPath = Get-MsiLogPath -product $DisplayName -action $action -verify;
				$arguments += " /l$($logOptions) `"$($logPath)`"";
			}

			if ($UiLevelAsProject) { $arguments += " /qb-!"; }
			elseif ([string]::IsNullOrEmpty($UiLevelFlags)) { $arguments += " /qb-!"; }
			else # evaluate flags
			{
				[int]$flags = [int]$UiLevelFlags;
				if ($flags -band 0x40) { $arguments += " /passive"; } # INSTALLUILEVEL_PROGRESSONLY
				switch ($flags -band 7)
				{
					5 { $arguments += " /qf"; } # INSTALLUILEVEL_FULL
					4 { $arguments += " /qr"; } # INSTALLUILEVEL_REDUCED
					3 { $arguments += " /qb$(if ($flags -band 0x80) {'+'} else {'-'})$(if ($flags -band 0x20) {'!'})"; } # INSTALLUILEVEL_BASIC + INSTALLUILEVEL_ENDDIALOG + INSTALLUILEVEL_HIDECANCEL
					2 { $arguments += " /qn$(if ($flags -band 0x80) {'+'})"; } # INSTALLUILEVEL_NONE + INSTALLUILEVEL_ENDDIALOG
					default { $arguments += ""; } # INSTALLUILEVEL_DEFAULT, INSTALLUILEVEL_NOCHANGE
				}
			}
			
			$arguments += " REBOOT=ReallySuppress";
			
			$arguments += " ALLUSERS=$(if ($allUsers) {'1'} else {'0'})";

			if (![string]::IsNullOrEmpty($AddParameters)) { $arguments += " $($AddParameters)"; }
			
			Write-Log -Message "Performing action '$($action)' for MSI-Product '$($ProductCode)' [$($DisplayName)]." -Source ${CmdletName};
			$status = Invoke-MsiExecExe -Arguments $arguments -PassThru -SecureParameters:$secureParameters;
			Write-Log -Message "Result status is: $($status.ExitCode) [$($status.StatusText)] [Success: $($status.Success)] [Reboot: $($status.Reboot)]." -Source ${CmdletName};
			if (![string]::IsNullOrEmpty($ResultVariable)) { Set-PdVar -Name $ResultVariable -Value $status.ExitCode; }
			if ($status.Reboot) { Request-Reboot; }
		}
		catch
		{
			$failed = "Failed to uninstall MSI product '$ProductCode' [$DisplayName]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Install-MsiPatch {
	<# 
	.SYNOPSIS
		Installs an MSI patch.
	.DESCRIPTION
		Installs an MSI patch.
	.PARAMETER Path
		Specifies the Windows Installer patch (*.msp file) to be installed
	.PARAMETER AddParameters
		Contains parameters to be passed to the Windows Installer for patch installation
	.PARAMETER ResultVariable
		Name of a variable that contains the exit code of the Windows Installer.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Install-MsiPatch -Path 'D:\Downloads\FoxitPDFReaderUpd1101_enu.msp' -AddParameters '/verysilent NILOGPARAM=DEBUG' -ResultVariable _returncode -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Install-MsiPatch.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][string]$AddParameters,
		[Parameter(Mandatory=$false)][string]$ResultVariable = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$Path = Expand-Path $Path;
			
			$displayName = $null;
			try
			{
				$info = Get-MsiTableProperty -Path $Path;
				$displayName = $info.DisplayName;
			}
			catch
			{
				Write-Log -Message "Failed to get info from '$($Path)'." -Severity 2 -Source ${CmdletName};
				$displayName = [System.IO.Path]::GetFilenameWithoutExtension($Path);
			}

			$action = "Update";
			$allUsers = $true;
			$secureParameters = $false;
			$logLevel = $null;
			
			if (![string]::IsNullOrEmpty($AddParameters))
			{
				$match = [regex]::Match($AddParameters, '(^|\s+)NILOGPARAM=(?<level>\w+)(\s+|$)');
				if ($match.Success)
				{
					$logLevel = $match.Groups['level'].Value;
					$AddParameters = "$($AddParameters.Substring(0, $match.Index)) $($AddParameters.Substring($match.Index + $match.Length))";
				}
			}
			
			$arguments = "/p `"$($Path)`""; # /p <msp>
			
			if (![string]::IsNullOrEmpty($logLevel))
			{
				$logOptions = $(if ($logLevel -eq "DEBUG") {"*v"} else {"*vx"});
				$logPath = Get-MsiLogPath -product $displayName -action $action -verify;
				$arguments += " /l$($logOptions) `"$($logPath)`"";
			}

			$arguments += " /qb-!";
			
			$arguments += " REBOOT=ReallySuppress";

			$arguments += " REINSTALLMODE=ecmus REINSTALL=ALL";
			
			$arguments += " ALLUSERS=$(if ($allUsers) {'1'} else {'0'})";

			if (![string]::IsNullOrEmpty($AddParameters)) { $arguments += " $($AddParameters)"; }

			Write-Log -Message "Performing action '$($action)' for MSI-Patch '$($Path)' [$($displayName)]." -Source ${CmdletName};
			$status = Invoke-MsiExecExe -Arguments $arguments -PassThru -SecureParameters:$SecureParameters;
			Write-Log -Message "Result status is: $($status.ExitCode) [$($status.StatusText)] [Success: $($status.Success)] [Reboot: $($status.Reboot)]." -Source ${CmdletName};
			if (![string]::IsNullOrEmpty($ResultVariable)) { Set-PdVar -Name $ResultVariable -Value $status.ExitCode; }
			if ($status.Reboot) { Request-Reboot; }
		}
		catch
		{
			$failed = "Failed to install MSI patch '$Path'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Repair-MsiProduct {
	<# 
	.SYNOPSIS
		Repairs an MSI software
	.DESCRIPTION
		Repairs or installs (parts of) an MSI based software again.
	.PARAMETER ProductCode
		The product GUID of the MSI file you want to reinstall.
	.PARAMETER DisplayName
		The Product name or Display name
	.PARAMETER RepairOptionFlags
		Add the value of the desired options to include the option in the repair. Start calculation with value of 0
		- +2 = Reinstall if files are missing
		- +4 = Reinstall if files are missing or outdated
		- +8 = Reinstall if files are missing, outdated or identical
		- +16 = Reinstall if files are missing or identical
		- +32 = Validate Checksum
		- +64 = Force reinstall
		- +256 = Rewrite Registry for HKCU and HKU
		- +128 = Rewrite Registry for HKCR and HKLM
		- +512= Reinstall all Shortcuts and Icons
		- +1024 = Start from the source package and reload the local package into the cache
	.PARAMETER UiLevelFlags
		Add the value of the desired options to include the option in the repair. Start calculation with value of 0

		UI Default behavior
		- +1 Default: The installer chooses an appropriate level of the user interface itself.
		- +2 Silent: Completely automatic installation in the background in 'silent' mode.
		- +3 Basic: Basic progress and error handling.
		- +4 Reduced: User interface with suppressed wizards and dialog boxes.
		- +5 Full: User interface with assistants, dialogs, progress and errors.
		
		Special options
		- +128 = Show finish dialog
		- +64 = Only show progress
		- +32 = Hide cancellation button
	.PARAMETER AssistanceFlags	
		- 2: Logging normal
		- 6: Logging debug
	.PARAMETER ResultVariable
		Name of a variable that contains the exit code of the Windows Installer.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Repair-MsiProduct -ProductCode "{3033FBAD-BA86-469B-8C6F-ECD41334BD4D}" -DisplayName 'blender 2.93.5' -RepairOptionFlags 32 -UiLevelFlags 2 -AssistanceFlags 4 -Context Computer
	.EXAMPLE
		Repair-MsiProduct -ProductCode "{3033FBAD-BA86-469B-8C6F-ECD41334BD4D}" -DisplayName 'blender 2.93.5' -RepairOptionFlags 32 -UiLevelFlags 101 -AssistanceFlags 2 -Context Computer
	.EXAMPLE
		Repair-MsiProduct -ProductCode "{3033FBAD-BA86-469B-8C6F-ECD41334BD4D}" -DisplayName 'blender 2.93.5' -RepairOptionFlags 392 -UiLevelFlags 4 -AssistanceFlags 6 -ResultVariable _returncode -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Repair-MsiProduct.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ProductCode,
		[Parameter(Mandatory=$false)][Alias("ProductName")][string]$DisplayName = $null,
		[Parameter(Mandatory=$false)][string]$RepairOptionFlags = "0",
		[Parameter(Mandatory=$false)][string]$UiLevelFlags = "1",
		[Parameter(Mandatory=$false)][string]$AssistanceFlags = "0",
		[Parameter(Mandatory=$false)][string]$ResultVariable = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$action = "Repair";
			$allUsers = $true;
			$secureParameters = $false;
			if ([string]::IsNullOrEmpty($DisplayName)) { $DisplayName = $ProductCode; }
			
			$repairOptions = "";
			if (![string]::IsNullOrEmpty($RepairOptionFlags))
			{
				[int]$flags = [int]$RepairOptionFlags;
				if ($flags -band 0x00000002) { $repairOptions += "p"; } # REINSTALLMODE_FILEMISSING
				if ($flags -band 0x00000004) { $repairOptions += "o"; } # REINSTALLMODE_FILEOLDERVERSION
				if ($flags -band 0x00000008) { $repairOptions += "e"; } # REINSTALLMODE_FILEEQUALVERSION
				if ($flags -band 0x00000010) { $repairOptions += "d"; } # REINSTALLMODE_FILEEXACT
				if ($flags -band 0x00000020) { $repairOptions += "c"; } # REINSTALLMODE_FILEVERIFY
				if ($flags -band 0x00000040) { $repairOptions += "a"; } # REINSTALLMODE_FILEREPLACE
				if ($flags -band 0x00000080) { $repairOptions += "m"; } # REINSTALLMODE_MACHINEDATA
				if ($flags -band 0x00000100) { $repairOptions += "u"; } # REINSTALLMODE_USERDATA
				if ($flags -band 0x00000200) { $repairOptions += "s"; } # REINSTALLMODE_SHORTCUT
				if ($flags -band 0x00000400) { $repairOptions += "v"; } # REINSTALLMODE_PACKAGE
			}
			$arguments = "/f$($repairOptions) `"$($ProductCode)`""; # /f <guid>

			$logLevel = $null;
			$uiLevelAsProject = $false;
			if (![string]::IsNullOrEmpty($AssistanceFlags))
			{
				[int]$flags = [int]$AssistanceFlags;
				if ($flags -band 1) { $uiLevelAsProject = $false; }
				if ($flags -band 2) { if ($flags -band 4) { $logLevel = "DEBUG"; } else { $logLevel = "NORMAL"; } }
			}
			
			if (![string]::IsNullOrEmpty($logLevel))
			{
				$logOptions = $(if ($logLevel -eq "DEBUG") {"*v"} else {"*vx"});
				$logPath = Get-MsiLogPath -product $DisplayName -action $action -verify;
				$arguments += " /l$($logOptions) `"$($logPath)`"";
			}

			if ($uiLevelAsProject) { $arguments += " /qb-!"; }
			elseif ([string]::IsNullOrEmpty($UiLevelFlags)) { $arguments += " /qb-!"; }
			else # evaluate flags
			{
				[int]$flags = [int]$UiLevelFlags;
				if ($flags -band 0x40) { $arguments += " /passive"; } # INSTALLUILEVEL_PROGRESSONLY
				switch ($flags -band 7)
				{
					5 { $arguments += " /qf"; } # INSTALLUILEVEL_FULL
					4 { $arguments += " /qr"; } # INSTALLUILEVEL_REDUCED
					3 { $arguments += " /qb$(if ($flags -band 0x80) {'+'} else {'-'})$(if ($flags -band 0x20) {'!'})"; } # INSTALLUILEVEL_BASIC + INSTALLUILEVEL_ENDDIALOG + INSTALLUILEVEL_HIDECANCEL
					2 { $arguments += " /qn$(if ($flags -band 0x80) {'+'})"; } # INSTALLUILEVEL_NONE + INSTALLUILEVEL_ENDDIALOG
					default { $arguments += ""; } # INSTALLUILEVEL_DEFAULT, INSTALLUILEVEL_NOCHANGE
				}
			}
			
			$arguments += " REBOOT=ReallySuppress";
			
			$arguments += " ALLUSERS=$(if ($allUsers) {'1'} else {'0'})";
			
			Write-Log -Message "Performing action '$($action)' for MSI-Product '$($ProductCode)' [$($DisplayName)]." -Source ${CmdletName};
			$status = Invoke-MsiExecExe -Arguments $arguments -PassThru -SecureParameters:$secureParameters;
			Write-Log -Message "Result status is: $($status.ExitCode) [$($status.StatusText)] [Success: $($status.Success)] [Reboot: $($status.Reboot)]." -Source ${CmdletName};
			if (![string]::IsNullOrEmpty($ResultVariable)) { Set-PdVar -Name $ResultVariable -Value $status.ExitCode; }
			if ($status.Reboot) { Request-Reboot; }
		}
		catch
		{
			$failed = "Failed to repair MSI product '$ProductCode' [$DisplayName]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Install-MsiFeature {
	<# 
	.SYNOPSIS
		Changes the installation state of features of an already installed MSI-based software.
	.DESCRIPTION
		Changes the installation state of features of an already installed MSI-based software.
	.PARAMETER ProductCode
		The product GUID of the MSI file for which you want to change the installation status of features.
	.PARAMETER DisplayName
		The Product name or Display name
	.PARAMETER HintPath
		Unused Parameter
	.PARAMETER FeatureList
		Change the installation status of the individual features as required. The default value is always "Keep installation status".
	.PARAMETER ResultVariable
		Name of a variable that contains the exit code of the Windows Installer.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Install-MsiFeature -ProductCode "{3033FBAD-BA86-469B-8C6F-ECD41334BD4D}" -DisplayName 'blender 2.93.5' -HintPath 'D:\Downloads\blender-2.93.5-windows-x64.msi' -FeatureList 'CM_C_Libraries|A' -ResultVariable _returncode -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Install-MsiFeature.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ProductCode,
		[Parameter(Mandatory=$false)][Alias("ProductName")][string]$DisplayName = $null,
		[Parameter(Mandatory=$false)][string]$HintPath = $null,
		[Parameter(Mandatory=$true)][string[]]$FeatureList = @(),
		[Parameter(Mandatory=$false)][string]$ResultVariable = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$action = "Update";
			$allUsers = $true;
			$secureParameters = $false;
			$logLevel = "NORMAL";
			if ([string]::IsNullOrEmpty($DisplayName)) { $DisplayName = $ProductCode; }

			$arguments = "/i `"$($ProductCode)`""; # /i <guid>
			
			if (![string]::IsNullOrEmpty($logLevel))
			{
				$logOptions = $(if ($logLevel -eq "DEBUG") {"*v"} else {"*vx"});
				$logPath = Get-MsiLogPath -product $DisplayName -action $action -verify;
				$arguments += " /l$($logOptions) `"$($logPath)`"";
			}

			$arguments += " /qb-!";
			
			$arguments += " REBOOT=ReallySuppress";
			
			$arguments += " ALLUSERS=$(if ($allUsers) {'1'} else {'0'})";

			$features = @{};
			$FeatureList | % { $_.Split(",") } | % { [regex]::Match($_, '^\s*(?<n>.+)\s*\|\s*(?<m>.+)\s*$') } | where Success | % { $m = $_.Groups["m"].Value; if (!$features.ContainsKey($m)) { $features[$m] = @() }; $features[$m] += $_.Groups["n"].Value; }
			$modes = @{ L = "ADDLOCAL"; S = "ADDSOURCE"; V = "ADVERTISE"; A = "REMOVE"; }
			foreach ($kvp in $features.GetEnumerator())
			{
				$value = [string]::Join(",", @($kvp.Value | % { $_ }))
				if (!$modes.ContainsKey($kvp.Key)) { throw "Unknown mode '$($kvp.Key)' for features '$($value)'."; }
				$mode = $modes[$kvp.Key];
				Write-Log -Message "Apply features '$($mode)': $($value)." -Source ${CmdletName};
				$arguments += " $($mode)=`"$($value)`"";
			}

			Write-Log -Message "Performing action '$($action)' for MSI-Product '$($ProductCode)' [$($DisplayName)]." -Source ${CmdletName};
			$status = Invoke-MsiExecExe -Arguments $arguments -PassThru -SecureParameters:$SecureParameters;
			Write-Log -Message "Result status is: $($status.ExitCode) [$($status.StatusText)] [Success: $($status.Success)] [Reboot: $($status.Reboot)]." -Source ${CmdletName};
			if (![string]::IsNullOrEmpty($ResultVariable)) { Set-PdVar -Name $ResultVariable -Value $status.ExitCode; }
			if ($status.Reboot) { Request-Reboot; }
		}
		catch
		{
			$failed = "Failed to install MSI product features of '$ProductCode' [$DisplayName]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Add-PrinterConnection {
	<# 
	.SYNOPSIS
		Connection to a network printer
	.DESCRIPTION
		Establishes the connection to a network printer. This command is suitable for installing network printers. The printer connection is stored for each computer. 
		The driver files are used by the print server. The printer must be installed on the computer where the installation package is created.
	.PARAMETER ConnectionName
		Path under which the shared printer can be reached.
	.PARAMETER PreventUninstall
		Remove-Printer will not be executed on 'uninstall mode'
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Add-PrinterConnection -ConnectionName '\\CHISV01.solys.local\PDFCreator
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Add-PrinterConnection.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ConnectionName,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }

			# Import-Module PrintManagement;

			if (Test-ReverseMode)
			{
				Write-Log -Message "Removing printer connection '$($ConnectionName)'." -Source ${CmdletName};
				Remove-Printer -Name $ConnectionName;
				
				return; # exit from reverse mode
			}

			Write-Log -Message "Adding printer connection '$($ConnectionName)'." -Source ${CmdletName};
			Add-Printer -ConnectionName $ConnectionName;
		}
		catch
		{
			$failed = "Failed to add printer connection '$ConnectionName'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-PrinterConnection {
	<# 
	.SYNOPSIS
 		Delete a connected network printer
	.DESCRIPTION
		Use this command to delete a connected network printer. Please note that the printer needs to 
		be installed on the workstation where you create the installation package.
	.PARAMETER ConnectionName
		Share name of the connected network printer to be deleted.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-PrinterConnection -ConnectionName 'Microsoft XPS Document Writer' -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-PrinterConnection.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ConnectionName,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			# Import-Module PrintManagement;

			Write-Log -Message "Removing printer connection '$($ConnectionName)'." -Source ${CmdletName};
			Remove-Printer -Name $ConnectionName;
		}
		catch
		{
			$failed = "Failed to add printer connection '$ConnectionName'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

$script:KnownFolderIdsLookup = @{};
function Get-KnownFolderIdsLookup([switch]$Wow64 = $false)
{
	$ErrorActionPreference = "Ignore"; # function scope

	IF (!$script:KnownFolderIdsLookup.ContainsKey($Wow64))
	{
		$lookup = @{};
		
		$key = Get-PdRegistryKey -Path "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\FolderDescriptions" -Wow64:$Wow64;
		foreach ($subKeyName in $key.GetSubKeyNames())
		{
			[Guid]$guid = [Guid]::Empty;
			if (![Guid]::TryParse($subKeyName, [ref]$guid)) { continue; }
			$subKey = $key.OpenSubKey($subKeyName, $false);
			
			$name = $subKey.GetValue("Name");
			if (![string]::IsNullOrEmpty($name)) { $lookup[$name] = $guid; }
			
			$subKey.Close();
		}
		
		$key.Close();
		
		$script:KnownFolderIdsLookup[$Wow64] = $lookup;
	}
	
	return $script:KnownFolderIdsLookup[$Wow64];
}

function Get-ShellFolderPath([string]$Folder, [switch]$Common = $false, [switch]$Wow64 = $false)
{
	if ([string]::IsNullOrEmpty($Folder)) { throw "No folder specified."; }
	
	$path = $null;
	
	if ([System.IO.Path]::IsPathRooted($Folder)) # custom path
	{
		$path = $Folder;
		Write-Log -Message "Identified custom folder path: '$($path)'." -Source ${CmdletName};
		return $path;
	}
	elseif ($Folder -match '^\.(\\|/|$)')
	{
		$path = Expand-Path $Folder -Wow64:$Wow64;
		Write-Log -Message "Identified relative custom folder path: '$($path)'." -Source ${CmdletName};
		return $path;
	}
	
	$name, $subDir = $Folder.Split("\/", 2); # <name>\<subdir>
	
	[System.Environment+SpecialFolder]$specialFolder = 0;
	$source = "Registry";
	$key = Get-PdRegistryKey -Path "HKCU:\SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders" -Wow64:$Wow64 -AcceptNull;
	$path = $(if ($key -ne $null) { $key.GetValue($name) } else { $null });
	if ([string]::IsNullOrEmpty($path) -and [System.Environment+SpecialFolder]::TryParse($name.Trim(), $true, [ref]$specialFolder))
	{
		$source = "Environment";
		$path = [System.Environment]::GetFolderPath($specialFolder);
	}
	
	if ([string]::IsNullOrEmpty($path))
	{
		$guid = [Guid]::Empty;
		$knownFolders = Get-KnownFolderIdsLookup -Wow64:$Wow64;
		if ($knownFolders.ContainsKey($name))
		{
			$source = "KnownFolders";
			$guid = $knownFolders[$name];
			$path = [PSPD.API]::GetKnownFolderPath($guid, 0);
		}
		elseif ([Guid]::TryParse($name, [ref]$guid))
		{
			$source = "KnownFolderID";
			$path = [PSPD.API]::GetKnownFolderPath($guid, 0);
		}
	}
	
	if ($Common)
	{
		$commonPath = $null;
		$commonName = $null;
		$commonSource = "Registry";
		$commonKey = Get-PdRegistryKey -Path "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\User Shell Folders" -Wow64:$Wow64;
		foreach ($commonName in @( "Common $($name)", "Common$($name)", $name )) { $commonPath = $commonKey.GetValue($commonName); if (![string]::IsNullOrEmpty($commonPath)) { break; } }

		if ([string]::IsNullOrEmpty($commonPath))
		{
			$commonName = ($(if ($name.StartsWith("Common", [System.StringComparison]::OrdinalIgnoreCase)) {""} else {"Common"}) + $name.Replace(" ", ""));
			if ([System.Environment+SpecialFolder]::TryParse($commonName, $true, [ref]$specialFolder))
			{
				$commonSource = "Environment";
				$commonPath = [System.Environment]::GetFolderPath($specialFolder);
			}
		}
		
		if ([string]::IsNullOrEmpty($commonPath))
		{
			$guid = [Guid]::Empty;
			$commonName = $name;
			$knownFolders = Get-KnownFolderIdsLookup -Wow64:$Wow64;
			if ($knownFolders.ContainsKey($commonName))
			{
				$commonSource = "KnownFolders";
				$guid = $knownFolders[$commonName];
				$commonPath = [PSPD.API]::GetKnownFolderPath($guid, -1);
			}
			elseif ([Guid]::TryParse($commonName, [ref]$guid))
			{
				$commonSource = "KnownFolderID";
				$commonPath = [PSPD.API]::GetKnownFolderPath($guid, -1);
			}
		}
		
		if (![string]::IsNullOrEmpty($commonPath))
		{
			$path = $commonPath;
			$name = $commonName;
			$source = $commonSource;
		}
		else { Write-Log -Message "Could not identify common shell folder path '$($Folder)'." -Source ${CmdletName} -Severity 2; }
	}
	
	if ([string]::IsNullOrEmpty($path)) { throw "Could not identify shell folder path '$($Folder)'."; }
	
	if (![string]::IsNullOrEmpty($subDir)) { $path = [System.IO.Path]::Combine($path, $subDir); }
	if ($Wow64) { $path = Expand-Path -Path $path -Wow64; }
	Write-Log -Message "Identified $(if ($Common) {'common '} else {''})shell folder '$($Folder)' as '$($name)' by '$($source)': '$($path)'." -Source ${CmdletName};
	return $path;
}

function New-Link {
	<# 
	.SYNOPSIS
 		Delete a shortcut
	.DESCRIPTION
		Use this command to delete a shortcut (link).
	.PARAMETER Description
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-Link -Description Homepage -Folder 'Start Menu\CANCOM' -ComputerRelatedLink -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-Link.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Description,
		[Parameter(Mandatory=$true)][string]$CommandLine,
		[Parameter(Mandatory=$false)][string]$WorkingDirectory,
		[Parameter(Mandatory=$false)][string]$Icon,
		[Parameter(Mandatory=$false)][string]$Folder,
		[Parameter(Mandatory=$false)][switch]$ComputerRelatedLink = $false,
		[Parameter(Mandatory=$false)][switch]$RunMinimized = $false,
		[Parameter(Mandatory=$false)][switch]$ManageUserPortion = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			$folderPath = Get-ShellFolderPath -Folder $Folder -Common:$ComputerRelatedLink -Wow64:$Wow64;
			$linkPath = [System.IO.Path]::Combine($folderPath, "$($Description).lnk");

			if (Test-ReverseMode)
			{
				Write-Log -Message "Deleting link file '$($linkPath)'." -Source ${CmdletName};
				Uninstall-SingleFile -Path $linkPath -Delete -DeleteInUse -Wow64:$Wow64;
				
				return; # exit from reverse mode
			}
			
			if (![System.IO.Directory]::Exists($folderPath))
			{
				Write-Log -Message "Creating link folder '$($folderPath)'." -Source ${CmdletName};
				Install-SingleDirectory -Path $folderPath -Recurse -Wow64:$Wow64;
			}

			$targetPath = $null;
			$arguments = $null;
			$match = [regex]::Match($CommandLine, '^(?<path>(([^"]([^"\.]+\.[^" ]+( |$))|"[^"]+"|[^ ]+)))(?<args>.*)$');
			if ($match.Success)
			{
				$targetPath = $match.Groups["path"].Value.Trim().Trim('"');
				$arguments = $match.Groups["args"].Value.Trim();
			}
			else
			{
				$targetPath = $CommandLine.Trim();
				$arguments = "";
			}
			
			$iconLocation = $Icon;
			$iconIndex = $null;
			if (![string]::IsNullOrEmpty($iconLocation))
			{
				[int]$number = 0;
				$index = $iconLocation.LastIndexOf(",");
				if (($index -ge 0) -and [Int32]::TryParse($iconLocation.SubString($index + 1), [ref]$number))
				{
					$iconIndex = $number;
					$iconLocation = $Icon.SubString(0, $index);
				}
			}

			$parameters = @{
				Path = $linkPath;
				TargetPath = $targetPath;
				Description = $Description;
				WindowStyle = $(if ($RunMinimized) {"Minimized"} else {"Normal"});
				# ContinueOnError = $ContinueOnError;
			}

			if (![string]::IsNullOrEmpty($arguments)) { $parameters["Arguments"] = $arguments; }

			if (![string]::IsNullOrEmpty($WorkingDirectory)) { $parameters["WorkingDirectory"] = $WorkingDirectory; }

			if (![string]::IsNullOrEmpty($iconLocation)) { $parameters["IconLocation"] = $iconLocation; }
			
			if ($iconIndex -ne $null) { $parameters["IconIndex"] = $iconIndex; }
			
			if ($ManageUserPortion) { Write-Log -Message "The parameter -ManageUserPortion is not supported." -Source ${CmdletName} -Severity 2; }
			
			Write-Log -Message "Creating link file '$($linkPath)'." -Source ${CmdletName};
			New-Shortcut @parameters;
		}
		catch
		{
			$failed = "Failed to create link '$Description'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-Link {
	<# 
	.SYNOPSIS
 		Delete a shortcut
	.DESCRIPTION
		Use this command to delete a shortcut (link).
	.PARAMETER Description
		Description of the link to be deleted.
	.PARAMETER Folder
		The name of the folder in which this shortcut should be removed
	.PARAMETER ComputerRelatedLink
		Indicates whether it is a computer-related link available to all users of a computer (e.g. through in the All Users profile)
	.PARAMETER Confirm
		If enabled, the user must confirm the removal of an existing program icon in a dialog.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-Link -Description Homepage -Folder 'Start Menu\CANCOM' -ComputerRelatedLink -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-Link.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Description,
		[Parameter(Mandatory=$false)][string]$Folder,
		[Parameter(Mandatory=$false)][switch]$ComputerRelatedLink = $false,
		[Parameter(Mandatory=$false)][switch]$Confirm = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$folderPath = Get-ShellFolderPath -Folder $Folder -Common:$ComputerRelatedLink -Wow64:$Wow64;
			$linkPath = [System.IO.Path]::Combine($folderPath, "$($Description).lnk");
			$urlPath = [System.IO.Path]::Combine($folderPath, "$($Description).url");

			if ($Confirm) { Write-Log -Message "The parameter -Confirm is not supported." -Source ${CmdletName} -Severity 2; }

			if ([System.IO.File]::Exists($linkPath))
			{
				Write-Log -Message "Deleting link file '$($linkPath)'." -Source ${CmdletName};
				Uninstall-SingleFile -Path $linkPath -Delete -DeleteInUse -Wow64:$Wow64;
			}
			elseif (![System.IO.File]::Exists($urlPath))
			{
				Write-Log -Message "'$($Description)' link not found at '$($linkPath)' or '$($urlPath)'." -Source ${CmdletName};
				return;
			}

			if ([System.IO.File]::Exists($urlPath))
			{
				Write-Log -Message "Deleting link file '$($urlPath)'." -Source ${CmdletName};
				Uninstall-SingleFile -Path $urlPath -Delete -DeleteInUse -Wow64:$Wow64;
			}
		}
		catch
		{
			$failed = "Failed to delete link '$Description'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function New-ShellFolder {
	<# 
	.SYNOPSIS
		Creates an system folder
	.DESCRIPTION
		Creates a folder below a defined path.
	.PARAMETER Path
		The target for the link to be created.
	.PARAMETER AllUsers
		Creates the link in the "All Users" area
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.PARAMETER PreventUninstall
		Uninstall-SingleDirectory Cmdlet will not get called on uninstall mode
	.EXAMPLE
		New-ShellFolder -Path 'StartMenu\CANCOM' -AllUsers
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/New-ShellFolder.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$AllUsers = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			$folderPath = Get-ShellFolderPath -Folder $Path -Common:$AllUsers -Wow64:$Wow64;

			if (Test-ReverseMode)
			{
				Write-Log -Message "Deleting folder path '$($folderPath)'." -Source ${CmdletName};
				Uninstall-SingleDirectory -Path $folderPath -DeleteNotEmpty:$false -DeleteAtEndOfScript -Wow64:$Wow64;
				
				return; # exit from reverse mode
			}
			
			Write-Log -Message "Creating folder path '$($folderPath)'." -Source ${CmdletName};
			Install-SingleDirectory -Path $folderPath -Recurse -Wow64:$Wow64;
		}
		catch
		{
			$failed = "Failed to create folder path '$Path'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-ShellFolder {
	<# 
	.SYNOPSIS
		Remove a system folder
	.DESCRIPTION
		Use this command to remove a system folder and all files contained in this folder.
	.PARAMETER Path
		Full path of the folder to be deleted.
	.PARAMETER AllUsers
		Indicates whether it is a computer-related folder that is available to all users of a computer (e.g. in the All Users profile)
	.PARAMETER Recurse
		If not enabled, the system folder is only deleted if it does not contain any subfolders. 
		If the system folder does not contain any subfolders, it will be deleted (along with the files it contains). Otherwise, the command is not executed.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-ShellFolder -Path 'Desktop\Greenshot' -AllUsers -Recurse -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-ShellFolder.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$AllUsers = $false,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$folderPath = Get-ShellFolderPath -Folder $Path -Common:$AllUsers -Wow64:$Wow64;

			Write-Log -Message "Deleting folder path '$($folderPath)'." -Source ${CmdletName};
			Uninstall-SingleDirectory -Path $folderPath -DeleteNotEmpty:$Recurse -Wow64:$Wow64;
		}
		catch
		{
			$failed = "Failed to delete folder path '$Path'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Merge-IniFile {
	<# 
	.SYNOPSIS
		Add entries to an INI file
	.DESCRIPTION
		This command can be used to add multiple entries or entire sections to an INI file. 
		The value specified in Name of INI file can denote both an existing and a new INI file.
	.PARAMETER FileName
		INI file to be changed
	.PARAMETER Lines
		Entries of the INI file to be newly created or existing entries to be updated.
	.PARAMETER ConfirmChanges
		If selected, the user must confirm every change to an existing entry in a dialog box.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Merge-IniFile -FileName 'Win.ini' -Lines @('[MSAPPS]','MSAPPS=W:\MSAPPS','MSINFO=W:\MSAPPS\MSINFO','','[Microsoft System Info]','MSINFO=W:\MSAPPS\MSINFO\MSINFO.EXE') -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Merge-IniFile.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FileName,
		[Parameter(Mandatory=$true)][AllowEmptyString()][string[]]$Lines,
		[Parameter(Mandatory=$false)][switch]$ConfirmChanges = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if ($ConfirmChanges)
			{
				Write-Log -Message "Parameter -ConfirmChanges is not supported." -Severity 2 -Source ${CmdletName};
			}
			
			if ([System.IO.Path]::IsPathRooted($FileName) -or $FileName.StartsWith(".\"))
			{
				$FileName = Expand-Path $FileName -Wow64:$Wow64;
			}

			$dir = [System.IO.Path]::GetDirectoryName($FileName);
			if (![System.IO.Directory]::Exists($dir))
			{
				Write-Log -Message "Creating INI file directory '$($dir)'." -Source ${CmdletName};
				$void = [System.IO.Directory]::CreateDirectory($dir);
			}

			Write-Log -Message "Merging INI file '$($FileName)' - lines: $($Lines.Count)." -Source ${CmdletName};
			$success = [PSPD.API]::UpdateIniFile($FileName, $Lines);
			Write-Log -Message "Merging of INI file '$($FileName)' succeeded: $($success)." -Source ${CmdletName};
		}
		catch
		{
			$failed = "Failed to merge INI file '$FileName'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-IniFileValue {
	<# 
	.SYNOPSIS
		Reads INI file
	.DESCRIPTION
		This command reads information from existing INI files. The dialog box specifies which key is to be read from which section in which INI file.
	.PARAMETER FileName
		INI file to be read out.
	.PARAMETER SectionName
		Name of the section.
	.PARAMETER KeyName
		Name of the key. 
	.PARAMETER ValueVariable
		The name of the variable in which the read-out value is to be stored
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-IniFileValue -FileName "${Env:APPDATA}\Greenshot\Greenshot.ini" -SectionName Core -KeyName Language -ValueVariable _GreenshotLang -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-IniFileValue.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FileName,
		[Parameter(Mandatory=$true)][string]$SectionName,
		[Parameter(Mandatory=$true)][string]$KeyName,
		[Parameter(Mandatory=$true)][string]$ValueVariable,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }
			
			if ([System.IO.Path]::IsPathRooted($FileName) -or $FileName.StartsWith(".\"))
			{
				$FileName = Expand-Path $FileName -Wow64:$Wow64;
			}
			
			Write-Log -Message "Read key '$($KeyName)' of section '$($SectionName)' of INI file '$($FileName)' into variable '$($ValueVariable)'." -Source ${CmdletName};
			$NoValue = "`b" # BS (backspace)
			$value = [PSPD.API]::GetIniString($FileName, $SectionName, $KeyName, $NoValue);
			if ($value -eq $NoValue)
			{
				throw "Key '$($KeyName)' in section '$($SectionName)' in INI file '$($FileName)' not found.";
				# $value = "";
				# Write-Log -Message "Key '$($KeyName)' not found." -Source ${CmdletName};
			}
			else
			{
				Write-Log -Message "Value of '$($KeyName)' is '$($value)'." -Source ${CmdletName};
			}
			
			if (![string]::IsNullOrEmpty($ValueVariable))
			{
				Set-PdVar -Name $ValueVariable -Value $value;
			}
		}
		catch
		{
			$failed = "Failed to read from INI file '$FileName'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Write-IniFileValue {
	<# 
	.SYNOPSIS
		Process values in INI files
	.DESCRIPTION
		This command processes individual values in INI files. The command works in the same way as the Merge-IniFile command, but offers more options and only processes a 
		specific key of a defined section of an INI file. It can also be used for delete operations. 
		The command supports all types of variables. The variables are converted to their current values when the command is executed, 
		for example, ${env:SystemRoot} becomes the Windows directory.
	.PARAMETER FileName
		INI file to be changed. The use of variables is possible.
	.PARAMETER SectionName
		Name of the section. The name is given without the square brackets and can refer to both an existing and a non-existing section. A section that does not yet exist is created.
	.PARAMETER KeyName
		Name of the key. This is specified without the equals sign and can refer both to an existing key and to a key that does not yet exist. A key that does not yet exist is created.
	.PARAMETER Value
		Value of the key specified above.
	.PARAMETER Action
		- Replace: Standard function: Key value is created or updated
		- Append: Appends an additional value to an existing line
		- Delete Deletes the value from an existing line
	.PARAMETER ConfirmChanges
		If selected, the user must confirm the modification of an existing entry in a dialog box.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Write-IniFileValue -FileName "${Env:APPDATA}}\Greenshot\Greenshot.ini" -SectionName Core -KeyName Language -Value 'de-DE' -Action Replace -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Write-IniFileValue.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FileName,
		[Parameter(Mandatory=$true)][string]$SectionName,
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$KeyName,
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$Value,
		[Parameter(Mandatory=$false)][ValidateSet("Replace", "Append", "Delete", "<Replace>", "<Append>", "<Delete>")][string]$Action = "Replace",
		[Parameter(Mandatory=$false)][switch]$ConfirmChanges = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if (![string]::IsNullOrEmpty($Action)) { $Action = $Action.Trim("<>"); }
			
			if ($ConfirmChanges)
			{
				Write-Log -Message "Parameter -ConfirmChanges is not supported." -Severity 2 -Source ${CmdletName};
			}
			
			if ([System.IO.Path]::IsPathRooted($FileName) -or $FileName.StartsWith(".\"))
			{
				$FileName = Expand-Path $FileName -Wow64:$Wow64;
			}
			
			$success = $true;
			if ([string]::IsNullOrEmpty($KeyName))
			{
				Write-Log -Message "Remove section '$($SectionName)' from INI file '$($FileName)'." -Source ${CmdletName};
				$success = [PSPD.API]::DeleteIniSection($FileName, $SectionName);
			}
			elseif ([string]::IsNullOrEmpty($Value))
			{
				Write-Log -Message "Remove key '$($KeyName)' from section '$($SectionName)' of INI file '$($FileName)'." -Source ${CmdletName};
				$success = [PSPD.API]::DeleteIniKey($FileName, $SectionName, $KeyName);
			}
			else
			{
				if ($Action -eq "Append")
				{
					$currentValue = [PSPD.API]::GetIniString($FileName, $SectionName, $KeyName, "");
					Write-Log -Message "Append value '$($Value)' to '$($currentValue)'." -Source ${CmdletName};
					# <Append> in ModifyIni: does not work ... behaves like <Replace> ...
					# here (assumed behaviour): append value string as item to "normalized" space-separated list ...
					$Value = [string]::Join(" ", ($currentValue.Split(" ", [System.StringSplitOptions]::RemoveEmptyEntries) + $Value));
				}
				elseif ($Action -eq "Delete")
				{
					$currentValue = [PSPD.API]::GetIniString($FileName, $SectionName, $KeyName, "");
					Write-Log -Message "Remove value '$($Value)' from '$($currentValue)'." -Source ${CmdletName};
					# <Delete> in ModifyIni: removes any occurrence of the value string and "normalizes" space-separated item list
					$Value = [string]::Join(" ", ($currentValue -replace $Value).Split(" ", [System.StringSplitOptions]::RemoveEmptyEntries));
				}
				
				Write-Log -Message "Set key '$($KeyName)' of section '$($SectionName)' of INI file '$($FileName)' to '$($Value)'." -Source ${CmdletName};
				$success = [PSPD.API]::SetIniString($FileName, $SectionName, $KeyName, $Value);
			}

			Write-Log -Message "Modification of INI file '$($FileName)' succeeded: $($success)." -Source ${CmdletName};
		}
		catch
		{
			$failed = "Failed to write to INI file '$FileName'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-DriveFreeKB {
	<# 
	.SYNOPSIS
		Free disk space
	.DESCRIPTION
		Checks the free space on the specified drive (in KB).
	.EXAMPLE
		Get-DriveFreeKB
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Get-DriveFreeKB.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$DriveName
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (!$DriveName.EndsWith(":")) { $DriveName += ":"; }
			
			$info = Get-WmiObject -Class Win32_LogicalDisk -Property Name, DeviceID, VolumeName, MediaType, Size, FreeSpace -Filter "(Name='$($DriveName)')";
			if ($info -eq $null)
			{
				Write-Log -Message "No information for drive '$($DriveName)'." -Severity 2 -Source ${CmdletName};
				return ,@(); # => ((Get-DriveFreeKB) <operator> <value>) results in $false
			}
			else
			{
				$freePercent = [Math]::Round((($info.FreeSpace / $info.Size) * 100), 3);
				$freeKB = [Math]::Round(($info.FreeSpace / 1KB), 2);
				Write-Log -Message "Drive '$($DriveName)': DeviceID '$($info.DeviceID)', VolumeName '$($info.VolumeName)', Size $($info.Size) Byte, FreeSpace $($info.FreeSpace) Byte ($($freePercent)%), returning $($freeKB) KB." -Source ${CmdletName};
				return $freeKB;
			}
		}
		catch
		{
			$failed = "Failed to get free space of drive '$DriveName'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-Platform {
	<# 
	.SYNOPSIS
		Check OS version
	.DESCRIPTION
		Checks if the operating system and the type of computer correspond to the specified value.
	.PARAMETER Criterion
		- WinXP: Windows XP.
		- WinXPx86: Windows XP x86.
		- WinXPx64: Windows XP x64.
		- WinSrv2003: Windows Server 2003 family.
		- WinSrv2003x86: Windows Server 2003 x86 family.
		- WinSrv2003x64: Windows Server 2003 x64 family.
		- WinVista: Windows Vista.
		- WinVistax86: Windows Vista x86.
		- WinVistax64: Windows Vista x64.
		- WinSrv2008: Windows Server 2008 family.
		- WinSrv2008x86: Windows Server 2008 x86 family.
		- WinSrv2008x64: Windows Server 2008 x64 family.
		- WinSeven: Windows 7.
		- WinSevenx86: Windows 7 x86.
		- WinSevenx64: Windows 7 x64.
		- WinEight: Windows 8.
		- WinEightx86: Windows 8 x86.
		- WinEightx64: Windows 8 x64.
		- WinSrv2012x64: Windows Server 2012 x64.
		- WinSrv2012R2x64: Windows Server 2012 R2 x64.
		- WinEightOne: Windows 8.1.
		- WinEightOnex86: Windows 8.1 x86.
		- WinEightOnex64: Windows 8.1 x64.
		- WinTen: Windows 10.
		- WinTenx86: Windows 10 x86.
		- WinTenx64: Windows 10 x64.
		- WinSrv2016x64: Windows Server 2016.
		- WinSrv2019x64: Windows Server 2019.
		- DC: Domain Controller.
		- BDC: Backup Domain Controller.
		- PDC: Primary Domain Controller.
		- DataCenterSrv: Datacenter Server.
		- TSRemoteAdmin: Terminal Services installed - Remote administration mode.
		- TSAppSrv: Terminal Services installed - Application server mode.
		- WinSrv2003Web: Windows Server 2003 - Web Edition.
		- Winx64: Operating system based on x64 processor architecture.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.EXAMPLE
		Test-FileExists -Path "$(env:ProgramFiles}\CANCOM\PackagingPowerBench\PackagingPowerBench.exe"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-Platform.html
	#>
	[CmdletBinding()]
	param (
		[ValidateSet("WinXP", "WinXPx86", "WinXPx64", "WinSrv2003", "WinSrv2003x86", "WinSrv2003x64", "WinVista", "WinVistax86", "WinVistax64", "WinSrv2008", "WinSrv2008x86", "WinSrv2008x64", 
			"WinSeven", "WinSevenx86", "WinSevenx64", "WinEight", "WinEightx86", "WinEightx64", "WinSrv2012x64", "WinSrv2012R2x64", "WinEightOne", "WinEightOnex86", "WinEightOnex64", 
			"WinTen", "WinTenx86", "WinTenx64", "WinElevenx64", "WinSrv2016x64", "WinSrv2019x64", "WinSrv2022x64", "DC", "BDC", "PDC", "DataCenterSrv", "TSRemoteAdmin", "TSAppSrv", "WinSrv2003Web", "Winx64")]
		[Parameter(Mandatory=$true)][string]$Criterion
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$osReleaseId = [int](Get-ItemProperty -Path "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows NT\CurrentVersion" -Name ReleaseId -ErrorAction SilentlyContinue).ReleaseId;

			# $is64Bit = [System.Environment]::Is64BitOperatingSystem; # >= .NET 4
			# Win32_OperatingSystem.OSArchitecture : "32-Bit", "64-Bit", ...
			$processor = Get-WmiObject Win32_Processor -Filter "(DeviceID='CPU0')" -Property AddressWidth;
			$is64Bit = ($processor.AddressWidth -eq 64);
			$is32Bit = ($processor.AddressWidth -eq 32);

			$os = Get-WmiObject -Class Win32_OperatingSystem;

			$version = [Version]$os.Version;
			$verMajorMinor = New-Object Version $version.Major, $version.Minor;
			$verBuild = $version.Build;

			$isWorkstation = ($os.ProductType -eq 1);
			$isServer = (($os.ProductType -eq 2) -or ($os.ProductType -eq 3));
			
			$cs = Get-WmiObject -Class Win32_ComputerSystem;
			
			$isRelease1809OrBuild17763 = (($osReleaseId -ge 1809) -or ($version.Build -ge 17763));
			$isRelease2009OrBuild20348 = (($osReleaseId -ge 2009) -or ($version.Build -ge 20348));
			
			Write-Log -Message "Detected OS: '$($os.Caption)' $($os.OSArchitecture) version $($version)." -Source ${CmdletName};
			
			$displayName = "?";
			$result = $false;
			
			switch ($Criterion)
			{
				"WinXP"           { $result = ($isWorkstation -and (($verMajorMinor -eq "5.1") -or ($verMajorMinor -eq "5.2"))); $displayName = "Windows XP"; break; }
				"WinXPx86"        { $result = ($isWorkstation -and ($verMajorMinor -eq "5.1")); $displayName = "Windows XP x86"; break; }
				"WinXPx64"        { $result = ($isWorkstation -and ($verMajorMinor -eq "5.2")); $displayName = "Windows XP x64"; break; }
				"WinSrv2003"      { $result = ($isServer -and ($verMajorMinor -eq "5.2")); $displayName = "Windows Server 2003 family"; break; }
				"WinSrv2003x86"   { $result = ($isServer -and ($verMajorMinor -eq "5.2") -and $is32Bit); $displayName = "Windows Server 2003 x86 family"; break; }
				"WinSrv2003x64"   { $result = ($isServer -and ($verMajorMinor -eq "5.2") -and $is64Bit); $displayName = "Windows Server 2003 x64 family"; break; }
				"WinVista"        { $result = ($isWorkstation -and ($verMajorMinor -eq "6.0")); $displayName = "Windows Vista"; break; }
				"WinVistax86"     { $result = ($isWorkstation -and ($verMajorMinor -eq "6.0") -and $is32Bit); $displayName = "Windows Vista x86"; break; }
				"WinVistax64"     { $result = ($isWorkstation -and ($verMajorMinor -eq "6.0") -and $is64Bit); $displayName = "Windows Vista x64"; break; }
				"WinSrv2008"      { $result = ($isServer -and (($verMajorMinor -eq "6.0") -or ($verMajorMinor -eq "6.1"))); $displayName = "Windows Server 2008 family"; break; }
				"WinSrv2008x86"   { $result = ($isServer -and (($verMajorMinor -eq "6.0") -or ($verMajorMinor -eq "6.1")) -and $is32Bit); $displayName = "Windows Server 2008 x86 family"; break; }
				"WinSrv2008x64"   { $result = ($isServer -and (($verMajorMinor -eq "6.0") -or ($verMajorMinor -eq "6.1")) -and $is64Bit); $displayName = "Windows Server 2008 x64 family"; break; }
				"WinSeven"        { $result = ($isWorkstation -and ($verMajorMinor -eq "6.1")); $displayName = "Windows 7"; break; }
				"WinSevenx86"     { $result = ($isWorkstation -and ($verMajorMinor -eq "6.1") -and $is32Bit); $displayName = "Windows 7 x86"; break; }
				"WinSevenx64"     { $result = ($isWorkstation -and ($verMajorMinor -eq "6.1") -and $is64Bit); $displayName = "Windows 7 x64"; break; }
				"WinEight"        { $result = ($isWorkstation -and ($verMajorMinor -eq "6.2")); $displayName = "Windows 8"; break; }
				"WinEightx86"     { $result = ($isWorkstation -and ($verMajorMinor -eq "6.2") -and $is32Bit); $displayName = "Windows 8 x86"; break; }
				"WinEightx64"     { $result = ($isWorkstation -and ($verMajorMinor -eq "6.2") -and $is64Bit); $displayName = "Windows 8 x64"; break; }
				"WinSrv2012x64"   { $result = ($isServer -and ($verMajorMinor -eq "6.2") -and $is64Bit); $displayName = "Windows Server 2012 x64"; break; }
				"WinSrv2012R2x64" { $result = ($isServer -and ($verMajorMinor -eq "6.3") -and $is64Bit); $displayName = "Windows Server 2012 R2 x64"; break; }
				"WinEightOne"     { $result = ($isWorkstation -and ($verMajorMinor -eq "6.3")); $displayName = "Windows 8.1"; break; }
				"WinEightOnex86"  { $result = ($isWorkstation -and ($verMajorMinor -eq "6.3") -and $is32Bit); $displayName = "Windows 8.1 x86"; break; }
				"WinEightOnex64"  { $result = ($isWorkstation -and ($verMajorMinor -eq "6.3") -and $is64Bit); $displayName = "Windows 8.1 x64"; break; }
				"WinTen"          { $result = ($isWorkstation -and ($verMajorMinor -eq "10.0") -and ($verBuild -lt 22000)); $displayName = "Windows 10"; break; }
				"WinTenx86"       { $result = ($isWorkstation -and ($verMajorMinor -eq "10.0") -and ($verBuild -lt 22000) -and $is32Bit); $displayName = "Windows 10 x86"; break; }
				"WinTenx64"       { $result = ($isWorkstation -and ($verMajorMinor -eq "10.0") -and ($verBuild -lt 22000) -and $is64Bit); $displayName = "Windows 10 x64"; break; }
				"WinElevenx64"    { $result = ($isWorkstation -and ($verMajorMinor -eq "10.0") -and ($verBuild -ge 22000) -and $is64Bit); $displayName = "Windows 11 x64"; break; }
				"WinSrv2016x64"   { $result = ($isServer -and ($verMajorMinor -eq "10.0") -and -not $isRelease1809OrBuild17763 -and $is64Bit); $displayName = "Windows Server 2016"; break; }
				"WinSrv2019x64"   { $result = ($isServer -and ($verMajorMinor -eq "10.0") -and $isRelease1809OrBuild17763 -and -not $isRelease2009OrBuild20348 -and $is64Bit); $displayName = "Windows Server 2019"; break; }
				"WinSrv2022x64"   { $result = ($isServer -and ($verMajorMinor -eq "10.0") -and $isRelease1809OrBuild17763 -and $isRelease2009OrBuild20348 -and $is64Bit); $displayName = "Windows Server 2022"; break; }

				"DC"              { $result = ($os.ProductType -eq 2); $displayName = "Domain Controller"; break; }
				"BDC"             { $result = ($cs.DomainRole -eq 4); $displayName = "Backup Domain Controller"; break; }
				"PDC"             { $result = ($cs.DomainRole -eq 5); $displayName = "Primary Domain Controller"; break; }
				"DataCenterSrv"   { $result = (($os.OSProductSuite -band 0x0080) -ne 0); $displayName = "Datacenter Server"; break; }
				"TSRemoteAdmin"   { $result = (($os.OSProductSuite -band 0x0110) -eq 0x0110); $displayName = "Terminal Services installed - Remote administration mode"; break; }
				"TSAppSrv"        { $result = (($os.OSProductSuite -band 0x0110) -eq 0x0010); $displayName = "Terminal Services installed - Application server mode"; break; }
				"WinSrv2003Web"   { $result = (($os.OSProductSuite -band 0x0400) -ne 0); $displayName = "Windows Server 2003 - Web Edition"; break; }
				"Winx64"          { $result = $is64Bit; $displayName = "Operating system based on x64 processor architecture"; break; }

				default           { $result = $false; throw "Invalid criterion '$($Criterion)'."; }
			}
			
			Write-Log -Message "Check platform $($Criterion) ($displayName): $($result)." -Source ${CmdletName};

			return $result;
		}
		catch
		{
			$failed = "Failed to check platform '$Criterion'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-FileExists {
	<# 
	.SYNOPSIS
		Existence of a file
	.DESCRIPTION
		Checks if a specific file exists.
	.PARAMETER Path
		File path.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.EXAMPLE
		Test-FileExists -Path "$(env:ProgramFiles}\CANCOM\PackagingPowerBench\PackagingPowerBench.exe"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-FileExists.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$Path = Expand-Path $Path;

			$result = [System.IO.File]::Exists($Path);
			Write-Log -Message "Existence of '$($Path)': $($result)." -Source ${CmdletName};
			
			if (!$result)
			{
				$result = [System.IO.Directory]::Exists($Path);
				Write-Log -Message "Existence of '$($Path)' (directory): $($result)." -Source ${CmdletName};
			}
			
			return $result;
		}
		catch
		{
			$failed = "Failed to check existence of '$Path'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-FileDate {
	<# 
	.SYNOPSIS
		File date
	.DESCRIPTION
		Checks the date of the specified file.
	.PARAMETER Path
		File path.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.EXAMPLE
		Get-FileDate -Path "$(env:ProgramFiles}\CANCOM\PackagingPowerBench\PackagingPowerBench.exe"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Get-FileDate.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$Path = Expand-Path $Path;

			if (![System.IO.File]::Exists($Path))
			{
				Write-Log -Message "File not found: '$($Path)'." -Severity 2 -Source ${CmdletName};
				return ,@(); # => ((Get-FileDate) <operator> <value>) results in $false
			}

			$dateTimeFormat = "yyyy-MM-dd";
			[datetime]$modified = [System.IO.File]::GetLastWriteTime($Path);
			[datetime]$result = $modified.Date; ## time: 00:00:00
			Write-Log -Message "Modification date of '$($Path)': $($modified), returning $($result) ($($result.ToString($dateTimeFormat)))." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to get modification date of '$Path'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-FileVersion {
	<# 
	.SYNOPSIS
		File version
	.DESCRIPTION
		Checks the version of the specified file.
	.PARAMETER Path
		File path.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.EXAMPLE
		Get-FileVersion -Path "$(env:ProgramFiles}\CANCOM\PackagingPowerBench\PackagingPowerBench.exe"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Get-FileVersion.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$Path = Expand-Path $Path;

			if (![System.IO.File]::Exists($Path))
			{
				Write-Log -Message "File not found: '$($Path)'." -Severity 2 -Source ${CmdletName};
				return ,@(); # => ((Get-FileVersion) <operator> <value>) results in $false
			}

			$result = $null;
			
            $version = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($Path);
			if (($version.FileMajorPart -gt 0) -or ($version.FileMinorPart -gt 0) -or ($version.FileBuildPart -gt 0) -or ($version.FilePrivatePart -gt 0))
			{
				$parts = @([Math]::Max($version.FileMajorPart, 0), [Math]::Max($version.FileMinorPart, 0), [Math]::Max($version.FileBuildPart, 0), [Math]::Max($version.FilePrivatePart, 0));
				$result = New-Object System.Version $parts;
				Write-Log -Message "File version of '$($Path)': $($result)." -Source ${CmdletName};
			}
			else
			{
				Write-Log -Message "No file version detected for '$($Path)'." -Source ${CmdletName};
				return ,@(); # => ((Get-FileVersion) <operator> <value>) results in $false
			}
			
			return $result;
		}
		catch
		{
			$failed = "Failed to get file version of '$Path'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-ProductVersion {
	<# 
	.SYNOPSIS
		Product version
	.DESCRIPTION
		Checks the product version of the specified file.
	.PARAMETER Path
		File path.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.EXAMPLE
		Get-ProductVersion -Path "$(env:ProgramFiles}\CANCOM\PackagingPowerBench\PackagingPowerBench.exe"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Get-ProductVersion.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$Path = Expand-Path $Path;

			if (![System.IO.File]::Exists($Path))
			{
				Write-Log -Message "File not found: '$($Path)'." -Severity 2 -Source ${CmdletName};
				return ,@(); # => ((Get-ProductVersion) <operator> <value>) results in $false
			}

			$result = $null;
			
            $version = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($Path);
			if (($version.ProductMajorPart -gt 0) -or ($version.ProductMinorPart -gt 0) -or ($version.ProductBuildPart -gt 0) -or ($version.ProductPrivatePart -gt 0))
			{
				$parts = @([Math]::Max($version.ProductMajorPart, 0), [Math]::Max($version.ProductMinorPart, 0), [Math]::Max($version.ProductBuildPart, 0), [Math]::Max($version.ProductPrivatePart, 0));
				$result = New-Object System.Version $parts;
				Write-Log -Message "Product version of '$($Path)': $($result)." -Source ${CmdletName};
			}
			else
			{
				Write-Log -Message "No product version detected for '$($Path)'." -Source ${CmdletName};
				return ,@(); # => ((Get-ProductVersion) <operator> <value>) results in $false
			}
			
			return $result;
		}
		catch
		{
			$failed = "Failed to get product version of '$Path'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-SystemRestart {
	<# 
	.SYNOPSIS
		System restart necessary
	.DESCRIPTION
		Checks if a system restart flag was set using commands like Install-FileList.
	.EXAMPLE
		Test-SystemRestart
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-SystemRestart.html
	#>
	[CmdletBinding()]
	param (
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$pdc = Get-PdContext;
			$result = ($pdc.RebootRequests -gt 0);
			Write-Log -Message "Current reboot requests: $($pdc.RebootRequests) - restart flag: $($result)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test restart flag";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Unregister-SystemRestart {
	<# 
	.SYNOPSIS
		Prevents an automatic system restart requested by commands of this script, such as Install-FileList
	.DESCRIPTION
		Prevents an automatic system restart requested by commands of this script, such as Install-FileList
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Unregister-SystemRestart
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Unregister-SystemRestart.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$pdc = Get-PdContext;
			Write-Log -Message "Clearing restart flags (current reboot requests: $($pdc.RebootRequests))." -Source ${CmdletName};
			$pdc.RebootRequests = 0;
			$pdc.StartSystemShutdownParameters = $null;
		}
		catch
		{
			$failed = "Failed to clear restart flags";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Register-SystemRestart {
	<# 
	.SYNOPSIS
		Sets a script internal flag that initiates a restart at the end of the package
	.DESCRIPTION
		Sets a script internal flag that initiates a restart at the end of the package. The flag can be set implicitly within the script 
		by commands such as Install-FileList if they want to manipulate files that are in access and therefore work with temporary files that are replaced at startup. 
	.PARAMETER AlwaysAskUser
		If this option is enabled, the user is also asked whether the required restart should be performed when the command is executed in a service context.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Register-SystemRestart -ForceLogoff
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Register-SystemRestart.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][switch]$AlwaysAskUser = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Request-Reboot;
			
			$AskUser = ($AlwaysAskUser -or ($DeployMode -eq "Interactive"));

			$pdc = Get-PdContext;
			$timeout = $(if ($AskUser) { 30 } else { 0 });
			$message = $null;
			$forceLogoff = !$AskUser;
			$pdc.StartSystemShutdownParameters = @{ Timeout = $timeout; Message = $message; ForceLogoff = $forceLogoff; Restart = $true; AskUser = $AskUser; };
			
			Write-Log -Message "Registered a system reboot." -Source ${CmdletName};
		}
		catch
		{
			$failed = "Failed to register a system reboot";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Register-SystemShutdown {
	<# 
	.SYNOPSIS
		Shut down a Windows system 
	.DESCRIPTION
		Shut down a Windows system and optionally reboots it. This command corresponds to the command line program shutdown.exe
	.PARAMETER Timeout
		Specified in seconds until shutdown is actually initiated.
	.PARAMETER Message
		The additional message to be displayed during system shutdown.
	.PARAMETER ForceLogoff
		If this option is activated, running programs are terminated and all unsaved data is lost. 
		The computer will shut down without informing the user; the user cannot prevent the shutdown.
	.PARAMETER Restart
		If this option is activated, the computer is automatically restarted.
	.PARAMETER AskUser
		If this option is activated, the user is prompted and can cancel the restart.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Register-SystemShutdown -Timeout 30 -Message 'Do you really want to reboot?' -Restart -AskUser
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Register-SystemShutdown.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][int]$Timeout = 30,
		[Parameter(Mandatory=$false)][string[]]$Message = "",
		[Parameter(Mandatory=$false)][switch]$ForceLogoff = $false,
		[Parameter(Mandatory=$false)][switch]$Restart = $false,
		[Parameter(Mandatory=$false)][switch]$AskUser = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Request-Reboot;
			
			$pdc = Get-PdContext;
			$pdc.StartSystemShutdownParameters = @{ Timeout = $Timeout; Message = [string]::Join("`r`n", $Message); ForceLogoff = $ForceLogoff; Restart = $Restart; AskUser = $AskUser; };
			
			Write-Log -Message "Registered a system shutdown." -Source ${CmdletName};
		}
		catch
		{
			$failed = "Failed to register a system shutdown";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Start-SystemShutdown {
	<# 
	.SYNOPSIS
		Triggers a system shutdown
	.DESCRIPTION
		Use this command to trigger a system shutdown
	.PARAMETER Timeout
		Timeout in seconds when to shutdown
	.PARAMETER Message
		A message for the user if AskUser is set
	.PARAMETER ForceLogoff
		Do a force log off
	.PARAMETER Restart
		If set a reboot will be done instead of a shutdown
	.PARAMETER AskUser
		If set the use will get a message promt
	.EXAMPLE
		Start-SystemShutdown -Timeout 30 -Message 'Do you really want to reboot?' -Restart -AskUser
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Start-SystemShutdown.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][int]$Timeout = 30,
		[Parameter(Mandatory=$false)][string]$Message = $null,
		[Parameter(Mandatory=$false)][switch]$ForceLogoff = $false,
		[Parameter(Mandatory=$false)][switch]$Restart = $false,
		[Parameter(Mandatory=$false)][switch]$AskUser = $false,
		[Parameter(Mandatory=$false)][switch]$PassThru = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		[bool]$status = $true;

		try
		{
			$shutdownTimeout = $Timeout;
			if (!$AskUser) { $shutdownTimeout = 0; }
			elseif ($shutdownTimeout -eq 0) { $shutdownTimeout = 30; }

			$shutdownAction = $(if ($Restart) { "restart" } else { "shutdown" });
			
			Write-Log -Message "Starting system $($shutdownAction)." -Source ${CmdletName};
			if ($AskUser)
			{
				# Write-Log -Message "Presenting a dialog to cancel the system $($shutdownAction) is currently not supported." -Severity 2 -Source ${CmdletName};
				$text = "$($configRestartPromptMessage)`r`n`r`n$($configRestartPromptButtonRestartNow)?";
				$aw = Show-DialogBox -Text $text -Buttons YesNo -DefaultButton First -Icon Question -Timeout 30;

				if ($aw -ne "Yes")
				{
					Write-Log -Message "User declined the system $($shutdownAction)." -Source ${CmdletName};
					$status = $false;
				}
			}

			if ($status)
			{
				Invoke-ShutdownExe -Timeout:$Timeout -Message:$Message -ForceLogoff:$ForceLogoff -Restart:$Restart;
			}
		}
		catch
		{
			$failed = "Failed to start the system shutdown";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
			$status = $false;
		}

		if ($PassThru) { return $status; }
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Stop-SystemShutdown {
	<# 
	.SYNOPSIS
		Stops a system shutdown
	.DESCRIPTION
		Use this command to stop a pending shutdown
	.PARAMETER Timeout
		Unused Paramter
	.PARAMETER Message
		Unused Paramter
	.PARAMETER ForceLogoff
		Unused Paramter
	.PARAMETER Restart
		Unused Paramter
	.PARAMETER AskUser
		Unused Paramter
	.EXAMPLE
		Stop-SystemShutdown
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Stop-SystemShutdown.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][int]$Timeout = 30,
		[Parameter(Mandatory=$false)][string[]]$Message = "",
		[Parameter(Mandatory=$false)][switch]$ForceLogoff = $false,
		[Parameter(Mandatory=$false)][switch]$Restart = $false,
		[Parameter(Mandatory=$false)][switch]$AskUser = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			Write-Log -Message "Stopping system shutdown." -Source ${CmdletName};
			Invoke-ShutdownExe -Cancel;
		}
		catch
		{
			$failed = "Failed to stop the system shutdown";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Suspend-ScriptExecution {
	<# 
	.SYNOPSIS
		Pause the installation
	.DESCRIPTION
		Use this command to pause the installation package from being executed for a specified number of seconds.
	.PARAMETER Seconds
		Duration of the script pause in seconds.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Suspend-ScriptExecution -Seconds 10
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Suspend-ScriptExecution.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][int]$Seconds,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $null) { return; }

			Write-Log -Message "Suspending script execution - seconds: $($Seconds) ..." -Source ${CmdletName};
			Start-Sleep -Seconds $Seconds;
			Write-Log -Message "... Continue script execution." -Source ${CmdletName};
		}
		catch
		{
			$failed = "Failed to suspend script execution";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Set-PdVarIncrement {
	<# 
	.SYNOPSIS
		Increment the value
	.DESCRIPTION
		Use this command to increase the value of a variable by the specified value.
	.PARAMETER Name
		Name of the variable whose value is to be increased
	.PARAMETER Increment
		Numerical value by which the previous value of the variable is increased.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-PdVarIncrement -Name _Counter -Increment 1
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-PdVarIncrement.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][int]$Increment = 0,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if ($Increment -ne 0)
			{
				[int]$value = Get-PdVar -Name $Name;
				[int]$result = ($value + $Increment);
				Write-Log -Message "Increment value of variable '$($Name)' ($value) by $($Increment): $($result)." -Source ${CmdletName};
				Set-PdVar -Name $Name -Value $result;
			}
		}
		catch
		{
			$failed = "Failed to increment variable [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Set-PdVarDecrement {
	<# 
	.SYNOPSIS
		Decrease the value
	.DESCRIPTION
		Use this command to decrease the value of a variable by the specified value.
	.PARAMETER Name
		Name of the variable whose value is to be decreased
	.PARAMETER Decrement
		Numerical value by which the previous value of the variable is decreased.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-PdVarDecrement -Name _Countdown -Decrement 1
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-PdVarDecrement.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][int]$Decrement = 0,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if ($Decrement -ne 0)
			{
				[int]$value = Get-PdVar -Name $Name;
				[int]$result = ($value - $Decrement);
				Write-Log -Message "Decrement value of variable '$($Name)' ($value) by $($Decrement): $($result)." -Source ${CmdletName};
				Set-PdVar -Name $Name -Value $result;
			}
		}
		catch
		{
			$failed = "Failed to decrement variable [$Name]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-RunningOnX64 {
	<# 
	.SYNOPSIS
		Computer with 64bit OS
	.DESCRIPTION
		Checks, if a 64bit OS is running on the computer.
	.EXAMPLE
		Test-RunningOnX64
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-RunningOnX64.html
	#>
	[CmdletBinding()]
	param (
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			# $is64Bit = [System.Environment]::Is64BitOperatingSystem; # >= .NET 4
			# Win32_OperatingSystem.OSArchitecture : "32-Bit", "64-Bit", ...
			$processor = Get-WmiObject Win32_Processor -Filter "(DeviceID='CPU0')" -Property AddressWidth;
			$is64Bit = ($processor.AddressWidth -eq 64);
			$is32Bit = ($processor.AddressWidth -eq 32);
			
			Write-Log -Message "Detected 64-Bit OS: $($is64Bit)." -Source ${CmdletName};
			return $is64Bit;
		}
		catch
		{
			$failed = "Failed to test 64-Bit OS";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-RunningOnServerOS {
	<# 
	.SYNOPSIS
		Computer with server OS
	.DESCRIPTION
		Checks, if a server OS is running on the computer.
	.EXAMPLE
		Test-RunningOnServerOS
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-RunningOnServerOS.html
	#>
	[CmdletBinding()]
	param (
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$os = Get-WmiObject -Class Win32_OperatingSystem;
			$isServer = (($os.ProductType -eq 2) -or ($os.ProductType -eq 3));
			
			Write-Log -Message "Detected Server OS: $($isServer)." -Source ${CmdletName};
			return $isServer;
		}
		catch
		{
			$failed = "Failed to test Server OS";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Write-InstalledAppsRegistry {
	<#
	.SYNOPSIS
		Write into Installed Apps Registry
	.DESCRIPTION
		Registers an installed app. Internal Use only
	.PARAMETER ExitCode
		Exit code of the app installation
	.PARAMETER RequireGUID
		Specified that Package GUID is required
	.PARAMETER WriteAllProperties
		Enumerates all properties of the package and writes them to the registry
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Write-InstalledAppsRegistry -ExitCode 0 -WriteAllProperties
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][int]$ExitCode = 0,
		[Parameter(Mandatory=$false)][switch]$RequireGUID = $false,
		[Parameter(Mandatory=$false)][switch]$WriteAllProperties = $false,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$pdc = Get-PdContext;
			$userPart =  ($pdc.InstallMode -eq "InstallUserPart");
			
			$package = $pdc.Package;
			if ($package -eq $null) { throw "No package set in PdContext."; }
			
			$packageID = $null;
			
			[guid]$guid = [guid]::Empty;
			if ([guid]::TryParse($package.ID, [ref]$guid)) { $packageID = $guid.ToString("B").ToUpper(); }
			elseif ($RequireGUID) { throw "No GUID specified for package '$($package.Name)'."; }
			elseif (![string]::IsNullOrEmpty($package.ID)) { $packageID = [string]$package.ID; }
			else { $packageID = [string]$package.Name; }
			
			if ([string]::IsNullOrEmpty($packageID)) { throw "Cannot detect ID for package '$($package.Name)'."; }
			
			$keyRoot = $(if ($userPart) {"HKEY_CURRENT_USER"} else {"HKEY_LOCAL_MACHINE"});
			$keyPath = "$($keyRoot):\$($InstalledAppsRegistryKeyName)\$($packageID)";
			Write-Log -Message "Writing InstalledApps Registry for current package '$($package.Name)' with ID '$($packageID)' to '$($keyPath)'." -Source ${CmdletName};
			$key = Get-PdRegistryKey -Path $keyPath -Create -Writable;
			
			[bool]$success = ([string]::IsNullOrEmpty($pdc.Status) -or ($pdc.Status -eq "Done"));
			[bool]$isInstalled = ($success -and ($pdc.DeploymentType -ne "Uninstall"));
			
			$timestamp = [DateTime]::UtcNow;
			[string]$installTime = $timestamp.ToString("u"); # universal sortable date/time pattern
			$unixUtc = ([DateTime]"1970-01-01T00:00:00Z").ToUniversalTime();
			[byte[]]$installTimeUnixUtc = [BitConverter]::GetBytes([UInt64]($timestamp - $unixUtc).TotalSeconds);
			
			$values = @{
				DeploymentType         = [string]$pdc.DeploymentType;
				InstallName            = [string]$pdc.InstallName;
				InstallMode            = [string]$pdc.InstallMode;
				InstallUser            = [string]$script:ProcessNTAccount;
				Status                 = [string]$pdc.Status;
				StatusMessage          = [string]$pdc.StatusMessage;
				ExitCode               = [int]$ExitCode;
				LastInstallTime        = [string]$installTime;
				LastInstallTimeUnixUtc = [byte[]]$installTimeUnixUtc;
				IsInstalled            = [int]$isInstalled;
				InstallSuccess         = [int]$success;
				
				Name                   = [string]$package.Name;
				Vendor                 = [string]$package.Vendor;
				Version                = [string]$package.Version;
				Revision               = [string]$package.Revision;
				Architecture           = [string]$package.Architecture;
				Language               = [string]$package.Language;
			};

			if ($userPart) { $values["UserPartInstallMode"] = $pdc.UserPartInstallMode; }

			$action = "setting IsInstalled to $($values.IsInstalled)";
			if (!$success -and ($pdc.DeploymentType -eq "Uninstall"))
			{
				$values.Remove("IsInstalled"); # uninstallation failed -> keep current IsInstalled ...
				$value = $key.GetValue("IsInstalled");
				$action = "keeping current IsInstalled ($(if ($value -ne $null) {$value} else {'<not set>'}))";
			}

			Write-Log -Message "[(DeploymentType $($pdc.DeploymentType), InstallSuccess $($success)] package $($pdc.DeploymentType) $(if ($success) {'succeeded'} else {'failed'}) => $($action)." -Source ${CmdletName};

			if ($WriteAllProperties)
			{
				# add remaining package properties
				$package.GetEnumerator() | where { !$values.ContainsKey($_.Key) } | % { $values[$_.Key] = $_.Value; }
			}

			# set Registry values
			$values.GetEnumerator() | % { $key.SetValue($_.Key, $_.Value); }

			if (!$userPart -and $pdc.IncludeUserPart)
			{
				$key.Close();
				$keyRoot = "HKEY_CURRENT_USER";
				$keyPath = "$($keyRoot):\$($InstalledAppsRegistryKeyName)\$($packageID)";
				Write-Log -Message "Deploy mode is '$($pdc.DeployMode)': Writing InstalledApps Registry for current package '$($package.Name)' with ID '$($packageID)' to '$($keyPath)'." -Source ${CmdletName};
				$key = Get-PdRegistryKey -Path $keyPath -Create -Writable;
				$values.GetEnumerator() | % { $key.SetValue($_.Key, $_.Value); }
			}
		}
		catch
		{
			$failed = "Failed to write InstalledApps Registry for current package";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Write-ActiveSetupRegistry {
	<#
	.SYNOPSIS
		Write into Active Setup Registry
	.DESCRIPTION
		Registers an active setup of a script. Internal Use only
	.PARAMETER ScriptPath
		The script thats currently active
	.PARAMETER RequireGUID
		Specified that Package GUID is required
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Write-ActiveSetupRegistry -ScriptPath D:\Files\script.ps1 -RequireGUID
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ScriptPath,
		[Parameter(Mandatory=$false)][string]$Arguments = $null,
		[Parameter(Mandatory=$false)][switch]$RequireGUID = $false,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$pdc = Get-PdContext;
			$hasInternalUserPartHandler = ($pdc.UserPartHandler -match "^InternalSetup(\.|`$)");
			$hasActiveSetupUserPartHandler = ($pdc.UserPartHandler -match "^ActiveSetup(\.|`$)");
			$writeActiveSetupKeys = ([string]::IsNullOrEmpty($pdc.UserPartHandler) -or $hasActiveSetupUserPartHandler -or $hasInternalUserPartHandler);

			if (!$writeActiveSetupKeys) { Write-Log -Message "Skip writing Active Setup Registry: User part handler is '$($pdc.UserPartHandler)'."; return;}

			if (($pdc.InstallMode -eq "InstallUserPart") -and !$hasInternalUserPartHandler) { Write-Log -Message "Skip writing Active Setup Registry: Running in InstallMode $($pdc.InstallMode) initiated by UserPartHandler $($pdc.UserPartHandler)."; return;}
			
			$package = $pdc.Package;
			if ($package -eq $null) { throw "No package set in PdContext."; }
			if (!$package.HasUserPart) { Write-Log -Message "Skip writing Active Setup Registry: Package has no user part."; return;}
			
			$packageID = $null;
			
			[guid]$guid = [guid]::Empty;
			if ([guid]::TryParse($package.ID, [ref]$guid)) { $packageID = $guid.ToString("B").ToUpper(); }
			elseif ($RequireGUID) { throw "No GUID specified for package '$($package.Name)'."; }
			elseif (![string]::IsNullOrEmpty($package.ID)) { $packageID = [string]$package.ID; }
			else { $packageID = [string]$package.Name; }
			
			if ([string]::IsNullOrEmpty($packageID)) { throw "Cannot detect ID for package '$($package.Name)'."; }

			$writeLocalMachineKey = ($pdc.InstallMode -ne "InstallUserPart");
			$writeUserPartHandler = !(($pdc.DeploymentType -eq "Install") -or ($pdc.DeploymentType -eq "InstallComputerPart") -or ($pdc.DeploymentType -eq "InstallUserPart"));
			$writeCurrentUserKey = (($pdc.InstallMode -eq "InstallUserPart") -or $pdc.IncludeUserPart);
			$writeUninstall = ($pdc.DeploymentType -eq "Uninstall");

			$ActiveSetupRegistryKeyName = "SOFTWARE\Microsoft\Active Setup\Installed Components";
			
			$isInstalled = $(if ($writeUninstall) { 0 } else { 1 });
			
			$deployApplicationExePath = [System.IO.Path]::Combine([System.IO.Path]::GetDirectoryName($ScriptPath), "Deploy-Application.exe");
			$commandLine = "`"$($deployApplicationExePath)`" -InstallMode InstallUserPart";
			if ($writeUserPartHandler) { $commandLine += " -UserPartHandler ActiveSetup.$($PdContext.InstallMode)"; }
			if (![string]::IsNullOrEmpty($Arguments)) { $commandLine += " $($Arguments)"; }

			$activeSetupVersion = "1,0,0,0";

			$packageRevision = $package.Revision;
			if ([string]::IsNullOrEmpty($packageRevision)) { $packageRevision = "1"; }

			$packageVersionParts = ([string]$package.Version).Split(".");
			if ([string]::IsNullOrEmpty($package.Version)) { $activeSetupVersion = [string]::Join(",", @($packageRevision, 0,0,0)); }
			elseif (($packageVersionParts.Count -ge 4) -and ($packageVersionParts[3].Trim("0") -ne "")) { $activeSetupVersion = [string]::Join(",", $packageVersionParts[0..3]); }
			else { $activeSetupVersion = [string]::Join(",", @(@($packageVersionParts + @(0,0,0,0))[0..2] + @($packageRevision))); }
			
			$values = @{
				""                     = [string]$package.Name;
				IsInstalled            = [int]$isInstalled;
				StubPath               = [string]$commandLine;
				Version                = [string]$activeSetupVersion;
			};
			
			if ($isInstalled -eq 0) { $values["DontAsk"] = [int]2; }

			# set Registry values

			if ($writeLocalMachineKey)
			{
				$keyRoot = "HKEY_LOCAL_MACHINE";
				$keyPath = "$($keyRoot):\$($ActiveSetupRegistryKeyName)\$($packageID)";
				Write-Log -Message "Writing Active Setup Registry for current package '$($package.Name)' with ID '$($packageID)' to '$($keyPath)'." -Source ${CmdletName};
				$key = Get-PdRegistryKey -Path $keyPath -Create -Writable;
				$values.GetEnumerator() | % { $key.SetValue($_.Key, $_.Value); }
				$key.Close();
				
				if (!$writeUninstall)
				{
					Write-Log -Message "Deployment type is '$($pdc.DeploymentType)', installation mode is $($pdc.InstallMode)): Removing Active Setup Registry for all users." -Source ${CmdletName};

					$profilesKey = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList";
					try
					{
						# @(Get-ChildItem $profilesKey | Get-ItemProperty | % { $_.PSChildName }) | % { Remove-RegistryKey -KeyPath ($keyPath.Replace("$($keyRoot):", "HKEY_USERS\$($_)")) -ContinueOnError };
						$profiles = @(Get-ChildItem $profilesKey | Get-ItemProperty | select @{label="SID"; expression={ $_.PSChildName }}, @{label="ProfilePath"; expression={ $_.ProfileImagePath }}, @{label="NTAccount"; expression={ [System.IO.Path]::GetFileName($_.ProfileImagePath) }});
						# $profiles | % { Remove-RegistryKey -KeyPath ($keyPath.Replace("$($keyRoot):", "HKEY_USERS\$($_.SID)")) -ContinueOnError };
						Invoke-HKCURegistrySettingsForAllUsers -RegistrySettings { Remove-RegistryKey -KeyPath ($keyPath.Replace("$($keyRoot):", "HKEY_USERS\$($UserProfile.SID)")) -ContinueOnError } -UserProfiles $profiles;
					}
					catch
					{
						Write-Log -Message "Failed to remove the Active Setup Registry for all users.`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
					}
				}
			}

			if ($writeCurrentUserKey)
			{
				if ($writeUninstall)
				{
					Write-Log -Message "Deployment type is '$($pdc.DeploymentType)' (user part installation mode is $($pdc.UserPartInstallMode)): Removing Active Setup Registry for current user." -Source ${CmdletName};
					Remove-ActiveSetupRegistry -PackageID $packageID -UserPart -ContinueOnError:$ContinueOnError;
				}
				else
				{
					$keyRoot = "HKEY_CURRENT_USER";
					$keyPath = "$($keyRoot):\$($ActiveSetupRegistryKeyName)\$($packageID)";
					Write-Log -Message "Deploy mode is '$($pdc.DeployMode)': Writing Active Setup Registry for current package '$($package.Name)' with ID '$($packageID)' to '$($keyPath)'." -Source ${CmdletName};
					$key = Get-PdRegistryKey -Path $keyPath -Create -Writable;
					$key.SetValue("Version", $activeSetupVersion);
					$key.Close();
				}
			}
		}
		catch
		{
			$failed = "Failed to write Active Setup Registry for current package";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-ActiveSetupRegistry {
	<#
	.SYNOPSIS
		Removes an Active Setup entry in the registry.
	.DESCRIPTION
		Removes an Active Setup entry in the registry. Internal Use only.
	.PARAMETER PackageID
		The package id to remove.
	.PARAMETER UserPart
		Use the user part in the registry.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-ActiveSetupRegistry -PackageID {2444B599-7E1B-408E-9172-1869542448B4} -UserPart
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$PackageID,
		[Parameter(Mandatory=$false)][switch]$UserPart = $false,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$keyRoot = $(if ($UserPart) {"HKEY_CURRENT_USER"} else {"HKEY_LOCAL_MACHINE"});
			$keyPath = "$($keyRoot):\SOFTWARE\Microsoft\Active Setup\Installed Components";
			# Write-Log -Message "Removing Active Setup Registry for current package '$($package.Name)' with ID '$($packageID)' to '$($keyPath)'." -Source ${CmdletName};
			$key = Get-PdRegistryKey -Path $keyPath -AcceptNull -Writable;
			if ($key -eq $null)
			{
				Write-Log -Message "Active Setup Registry not found at '$($keyPath)'." -Source ${CmdletName};
				return;
			}
			
			$subKeyNames = $key.GetSubKeyNames();
			
			[guid]$guid = [guid]::Empty;
			$normalizedPackageID = $(if ([guid]::TryParse($PackageID, [ref]$guid)) { $guid.ToString("B").ToUpper() } else { $PackageID }); # see Write-ActiveSetupRegistry
			
			if ($subKeyNames -contains $PackageID)
			{
				Write-Log -Message "Removing Package ID '$($PackageID)' from Active Setup Registry at '$($keyPath)'." -Source ${CmdletName};
				$key.DeleteSubKey($PackageID);
			}
			elseif (($PackageID -ne $normalizedPackageID) -and ($subKeyNames -contains $normalizedPackageID))
			{
				Write-Log -Message "Removing Package ID '$($PackageID)' (as '$($normalizedPackageID)') from Active Setup Registry at '$($keyPath)'." -Source ${CmdletName};
				$key.DeleteSubKey($normalizedPackageID);
			}
			else
			{
				Write-Log -Message "Cannot remove Package ID '$($PackageID)' from Active Setup Registry at '$($keyPath)': Not found." -Source ${CmdletName};
			}
		}
		catch
		{
			$failed = "Failed to remove Active Setup Registry for package ID '$PackageID'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Invoke-ActiveSetupUserPart {
	<#
	.SYNOPSIS
		Run Active Setup with User Parts.
	.DESCRIPTION
		Run Active Setup with User Parts. Internal Use only.
	.PARAMETER Wait
		Wait for process.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Invoke-ActiveSetupUserPart -Wait
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][switch]$Wait = $false,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false
	)

	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}

	process
	{
		try
		{
			$applicationName = "C:\Windows\System32\runonce.exe";
			$arguments = "/AlternateShellStartup";
			if ($pdc.SessionZero)
			{
				$process = [PSPD.API]::StartInteractive($applicationName, $arguments, $null, "~");
				Write-Log -Message "Started process '$($process.ProcessName)' with ID $($process.Id) in user session $($process.SessionId)." -Source ${CmdletName};
			}
			else
			{
				$process = [System.Diagnostics.Process]::Start($applicationName, $arguments);
				Write-Log -Message "Started process '$($process.ProcessName)' with ID $($process.Id) in current session." -Source ${CmdletName};
			}

			if ($Wait)
			{
				$process.WaitForExit();
				Write-Log -Message "Process '$($process.ProcessName)' with ID $($process.Id) terminated with exit code $($process.ExitCode)." -Source ${CmdletName};
			}
		}
		catch
		{
			$failed = "Failed to invoke Active Setup user part";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}

	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-InstalledAppsRegistry {
	<#
	.SYNOPSIS
		Get a installed apps package.
	.DESCRIPTION
		Get a installed apps package. Internal Use only.
	.PARAMETER PackageID
		Package ID of the app
	.PARAMETER UserPart
		Use the user part in the registry.
	.PARAMETER AcceptNull
		Dont abort if InstalledApps Registry Key is not found.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Get-InstalledAppsRegistry -PackageID {2444B599-7E1B-408E-9172-1869542448B4} -UserPart
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$PackageID,
		[Parameter(Mandatory=$false)][switch]$UserPart = $false,
		[Parameter(Mandatory=$false)][switch]$AcceptNull = $false,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$keyRoot = $(if ($UserPart) {"HKEY_CURRENT_USER"} else {"HKEY_LOCAL_MACHINE"});
			
			$keyPath = "$($keyRoot):\$($InstalledAppsRegistryKeyName)";
			$key = Get-PdRegistryKey -Path $keyPath -AcceptNull:$AcceptNull;
			if ($key -eq $null)
			{
				Write-Log -Message "InstalledApps Registry not found at '$($keyPath)'." -Source ${CmdletName};
				return $null;
			}
			
			$subKeyNames = $key.GetSubKeyNames();
			$key.Close();
			
			[guid]$guid = [guid]::Empty;
			$normalizedPackageID = $(if ([guid]::TryParse($PackageID, [ref]$guid)) { $guid.ToString("B").ToUpper() } else { $PackageID }); # see Write-InstalledAppsRegistry
			
			$key = $null;
			if (($PackageID -ne $normalizedPackageID) -and ($subKeyNames -contains $normalizedPackageID))
			{
				$keyPath = "$($keyRoot):\$($InstalledAppsRegistryKeyName)\$($normalizedPackageID)";
				Write-Log -Message "Reading InstalledApps Registry for package with ID '$($PackageID)' from '$($keyPath)'." -Source ${CmdletName};
				$key = Get-PdRegistryKey -Path $keyPath -AcceptNull:$AcceptNull;
			}
			else
			{
				$keyPath = "$($keyRoot):\$($InstalledAppsRegistryKeyName)\$($PackageID)";
				Write-Log -Message "Reading InstalledApps Registry for package with ID '$($PackageID)' from '$($keyPath)'." -Source ${CmdletName};
				$key = Get-PdRegistryKey -Path $keyPath -AcceptNull:$AcceptNull;
			}
			
			$result = $null;

			if ($key -ne $null)
			{
				$result = New-Object PSObject;
				foreach ($name in $key.GetValueNames()) { $result | Add-Member -MemberType NoteProperty -Name $name -Value $key.GetValue($name); }
			}

			return $result;
		}
		catch
		{
			$failed = "Failed to read InstalledApps Registry for package ID '$PackageID'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-InstalledAppsRegistry {
	<#
	.SYNOPSIS
		Removes an installed apps package.
	.DESCRIPTION
		Removes an installed apps package. Internal Use only.
	.PARAMETER PackageID
		Package ID of the app
	.PARAMETER UserPart
		Use the user part in the registry.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-InstalledAppsRegistry -PackageID {2444B599-7E1B-408E-9172-1869542448B4} -UserPart
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$PackageID,
		[Parameter(Mandatory=$false)][switch]$UserPart = $false,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$keyRoot = $(if ($UserPart) {"HKEY_CURRENT_USER"} else {"HKEY_LOCAL_MACHINE"});
			$keyPath = "$($keyRoot):\$($InstalledAppsRegistryKeyName)";
			$key = Get-PdRegistryKey -Path $keyPath -AcceptNull -Writable;
			if ($key -eq $null)
			{
				Write-Log -Message "InstalledApps Registry not found at '$($keyPath)'." -Source ${CmdletName};
				return;
			}
			
			$subKeyNames = $key.GetSubKeyNames();
			
			[guid]$guid = [guid]::Empty;
			$normalizedPackageID = $(if ([guid]::TryParse($PackageID, [ref]$guid)) { $guid.ToString("B").ToUpper() } else { $PackageID }); # see Write-InstalledAppsRegistry
			
			if ($subKeyNames -contains $PackageID)
			{
				Write-Log -Message "Removing Package ID '$($PackageID)' from InstalledApps Registry at '$($keyPath)'." -Source ${CmdletName};
				$key.DeleteSubKey($PackageID);
			}
			elseif (($PackageID -ne $normalizedPackageID) -and ($subKeyNames -contains $normalizedPackageID))
			{
				Write-Log -Message "Removing Package ID '$($PackageID)' (as '$($normalizedPackageID)') from InstalledApps Registry at '$($keyPath)'." -Source ${CmdletName};
				$key.DeleteSubKey($normalizedPackageID);
			}
			else
			{
				Write-Log -Message "Cannot remove Package ID '$($PackageID)' from InstalledApps Registry at '$($keyPath)': Not found." -Source ${CmdletName};
			}
		}
		catch
		{
			$failed = "Failed to remove InstalledApps Registry for package ID '$PackageID'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-PackageInstalled {
	<#
	.SYNOPSIS
		Executed installation of a package
	.DESCRIPTION
		Checks if the selected package has been executed already.
	.PARAMETER PackageID
		The Package-ID
	.PARAMETER Hint
		The Display Name of the Package
	.PARAMETER Revision
		The Revision to check
	.PARAMETER UserPart
		Check if the user portion is installed.
	.EXAMPLE
		Test-PackageRevisionInstalled -PackageID "{AC07DC31-0B69-40F7-BCC0-5026641B10DF}" -Hint 'Test Package' -Revision 4 -OrHigher -Completion ComputerPart
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-PackageInstalled.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$PackageID,
		[Parameter(Mandatory=$false)][string]$Hint,
		[Parameter(Mandatory=$false)][switch]$UserPart = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$info = Get-InstalledAppsRegistry -PackageID $PackageID -UserPart:$UserPart -AcceptNull;
			if ($info -eq $null)
			{
				Write-Log -Message "Package ID '$($PackageID)' not found in Registry." -Source ${CmdletName};
				return $false;
			}
			
			$result = ([int]$info.IsInstalled -ne 0);
			Write-Log -Message "Identified package ID '$($PackageID)' as '$($info.Name)'. Property IsInstalled: $($info.IsInstalled) - result: $($result)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test the installation of package ID '$PackageID'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-PackageRevisionInstalled {
	<#
	.SYNOPSIS
		Executed installation of a package
	.DESCRIPTION
		Checks if the selected package has been executed already.
	.PARAMETER PackageID
		The Package-ID
	.PARAMETER Hint
		The Display Name of the Package
	.PARAMETER Revision
		The Revision to check
	.PARAMETER OrHigher	
		Is switch is set, any higher revision will be accpeted
	.PARAMETER Completion
		- AnyPart: Any portion is installed.
		- Complete: The computer portion and the user portion are installed.
		- UserPart: The user portion is installed.
		- ComputerPart: The computer portion is installed.
	.EXAMPLE
		Test-PackageRevisionInstalled -PackageID "{AC07DC31-0B69-40F7-BCC0-5026641B10DF}" -Hint 'Test Package' -Revision 4 -OrHigher -Completion ComputerPart
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-PackageRevisionInstalled.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$PackageID,
		[Parameter(Mandatory=$false)][string]$Hint,
		[Parameter(Mandatory=$false)][AllowEmptyString()][string]$Revision = $null,
		[Parameter(Mandatory=$false)][switch]$OrHigher = $false,
		[Parameter(Mandatory=$false)][ValidateSet("AnyPart", "Complete", "UserPart", "ComputerPart")][string]$Completion = "AnyPart"
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$needAnyPart = ($Completion -eq "AnyPart");
			$needAllParts = ($Completion -eq "Complete");
			$needOnlyUserPart = ($Completion -eq "UserPart");
			$needOnlyComputerPart = ($Completion -eq "ComputerPart");
			Write-Log -Message "Required completion: $($Completion)." -Source ${CmdletName};
			
			$anyRevision = [string]::IsNullOrEmpty($Revision);
			Write-Log -Message "Required revision: $(if ($anyRevision) {'Any'} else {$Revision})$(if ($OrHigher) {' or higher'})." -Source ${CmdletName};
			
			$userPartResult = $false;
			if ($needOnlyUserPart -or $needAnyPart -or $needAllParts)
			{
				$info = Get-InstalledAppsRegistry -PackageID $PackageID -UserPart -AcceptNull;
				if ($info -eq $null)
				{
					Write-Log -Message "User part of Package ID '$($PackageID)' not found in Registry (HKCU)." -Source ${CmdletName};
				}
				else
				{
					Write-Log -Message "Identified user part of package ID '$($PackageID)' as '$($info.Name)'. Property IsInstalled: $($info.IsInstalled), property Revision: $($info.Revision)." -Source ${CmdletName};
					if ([string]::IsNullOrEmpty($info.Revision) -and !$anyRevision) { $info.Revision = 1; Write-Log -Message "Property Revision not set - assume : $($info.Revision)." -Source ${CmdletName}; }
				}
				
				$userPartResult = ([bool]$info.IsInstalled -and ($anyRevision -or ($info.Revision -eq $Revision) -or ($OrHigher -and ($info.Revision -gt $Revision))));
				Write-Log -Message "Calculated result for user part: $($userPartResult)." -Source ${CmdletName};
			}
			
			$computerPartResult = $false;
			if ($needOnlyComputerPart -or $needAnyPart -or $needAllParts)
			{
				$info = Get-InstalledAppsRegistry -PackageID $PackageID -AcceptNull;
				if ($info -eq $null)
				{
					Write-Log -Message "Computer part for Package ID '$($PackageID)' not found in Registry (HKLM)." -Source ${CmdletName};
				}
				else
				{
					Write-Log -Message "Identified computer part of package ID '$($PackageID)' as '$($info.Name)'. Property IsInstalled: $($info.IsInstalled), property Revision: $($info.Revision)." -Source ${CmdletName};
					if ([string]::IsNullOrEmpty($info.Revision) -and !$anyRevision) { $info.Revision = 1; Write-Log -Message "Property Revision not set - assume : $($info.Revision)." -Source ${CmdletName}; }
				}
				
				$computerPartResult = ([bool]$info.IsInstalled -and ($anyRevision -or ($info.Revision -eq $Revision) -or ($OrHigher -and ($info.Revision -gt $Revision))));
				Write-Log -Message "Calculated result for computer part: $($computerPartResult)." -Source ${CmdletName};
			}
			
			$result = $false;
			switch ($Completion)
			{
				"AnyPart" { $result = ($userPartResult -or $computerPartResult); break; }
				"Complete" { $result = ($userPartResult -and $computerPartResult); break; }
				"UserPart" { $result = $userPartResult; break; }
				"ComputerPart" { $result = $computerPartResult; break; }
			}
			Write-Log -Message "Calculated result (completion $($Completion)): $($result)." -Source ${CmdletName};

			return $result;
		}
		catch
		{
			$failed = "Failed to test the installation revision of package ID '$PackageID'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Set-Label {
	<#
	.SYNOPSIS
		Defines a label within the current installation package. 
	.DESCRIPTION
		Defines a label within the current installation package. 
		The label can be used in conjunction with Invoke-Goto to change the sequential processing of an installation package.
	.PARAMETER Name
		Name of the label to be defined.
	.EXAMPLE
		Set-Label -Name "`$BeginUninstallScript"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-Label.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		# Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			Write-Log -Message "Passing label '$($Name)'." -Source ${CmdletName};
			
			# processing $BeginUninstallScript label behaviour
			if (($Name -eq '$BeginUninstallScript') -and !(Set-BeginUninstallScript)) { exit 0; }
		}
		catch
		{
			$failed = "Failed passing label '$Name'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		# Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Invoke-Goto {
	<#
	.SYNOPSIS
		Jumps to a location within the current installation package
	.DESCRIPTION
		Jumps to a location within the current installation package. 
		Invoke-Goto can be used in conjunction with the Set-Label command to change the sequential processing of an installation package.
	.PARAMETER Label
		Name of the label at which the script execution should be continued
	.EXAMPLE
		Invoke-Goto -Label "`$BeginUninstallScript"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Invoke-Goto.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Label,
		[Parameter(Mandatory=$false)][switch]$SupportUninstall = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $null -SupportReverse:$SupportUninstall) { return; }

			$pdc = Get-PdContext;
			
			$scriptPath = $pdc.ScriptPath;
			if ([string]::IsNullOrEmpty($scriptPath)) { throw "ScriptPath not set in PdContext." }
			elseif (![System.IO.File]::Exists($scriptPath)) { throw "ScriptPath '$($scriptPath)' of PdContext does not exist." }
			
			$scriptText = [System.IO.File]::ReadAllText($scriptPath, [System.Text.Encoding]::Default);
			$scriptBlock = [scriptblock]::Create($scriptText);
			$isLabel = {param($a) ($a -is [System.Management.Automation.Language.CommandAst]) -and ($a.GetCommandName() -eq "Set-Label") -and ((($a.CommandElements[1].ParameterName -eq "Name") -and ($a.CommandElements[2].Value -eq $Label)) -or ($a.CommandElements[1].Value -eq $Label))}
			$labelAst = $scriptBlock.Ast.Find($isLabel, $true);
			if ($labelAst -eq $null) { throw "Label '$($Label)' does not exist." }
			
			$blockOf = {param($a, [Type[]]$tl); $p = $a; while ($p -ne $null) { if (($p -eq $rootBlock) -or ([Array]::Exists($tl, [System.Predicate[Type]]{param($t) $p -is $t }))) { return $p; } else { $p = $p.Parent; } }}
			$blockTypes = @([System.Management.Automation.Language.StatementBlockAst], [System.Management.Automation.Language.NamedBlockAst], [System.Management.Automation.Language.IfStatementAst], [System.Management.Automation.Language.SwitchStatementAst], [System.Management.Automation.Language.LoopStatementAst], [System.Management.Automation.Language.FunctionDefinitionAst], [System.Management.Automation.Language.ScriptBlockAst]);
			
			$lines = @();
			$offset = $labelAst.Extent.StartOffset;
			
			$blockAst = . $blockOf $labelAst $blockTypes;
			while ($blockAst -ne $null)
			{
				if ($blockAst -eq $rootBlock)
				{
					$lines += $blockAst.Extent.Text.Substring($offset - $blockAst.Extent.StartOffset);
					$blockAst = $null;
				}
				elseif ($blockAst -is [System.Management.Automation.Language.StatementBlockAst])
				{
					$lines += (". {`r`n" + $blockAst.Extent.Text.Substring($offset - $blockAst.Extent.StartOffset));
					$offset = $blockAst.Extent.EndOffset;
					$blockAst = . $blockOf $blockAst.Parent $blockTypes;
				}
				elseif ($blockAst -is [System.Management.Automation.Language.NamedBlockAst])
				{
					$lines += $blockAst.Extent.Text.Substring($offset - $blockAst.Extent.StartOffset);
					$offset = $blockAst.Extent.EndOffset;
					$blockAst = . $blockOf $blockAst.Parent $blockTypes;
				}
				elseif (($blockAst -is [System.Management.Automation.Language.IfStatementAst]) -or 
						($blockAst -is [System.Management.Automation.Language.SwitchStatementAst]) -or 
						($blockAst -is [System.Management.Automation.Language.LoopStatementAst]) -or 
						($blockAst -is [System.Management.Automation.Language.ScriptBlockAst]))
				{
					$offset = $blockAst.Extent.EndOffset;
					$blockAst = . $blockOf $blockAst.Parent $blockTypes;
				}
				elseif ($blockAst -is [System.Management.Automation.Language.FunctionDefinitionAst])
				{
					$offset = $blockAst.Extent.EndOffset;
					$blockAst = $null;
				}
				else
				{
					Write-Log -Message "Unexpected block type [$($blockAst.GetType().FullName)]" -Severity 2 -Source ${CmdletName};
					$lines += $blockAst.Extent.Text.Substring($offset - $blockAst.Extent.StartOffset);
					$blockAst = $null;
				}
			}

			$code = [string]::Join("`r`n", $lines);
			$pdc.ContinueScript = [scriptblock]::Create($code);
			
			Write-Log -Message "Goto label '$($Label)'." -Source ${CmdletName};
			exit 0;
		}
		catch
		{
			$failed = "Failed to goto label '$Label'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-ProcessRunning {
	<#
	.SYNOPSIS
		Check for Process
	.DESCRIPTION
		Checks if a Windows process is running either by Window title or File name
	.PARAMETER Identifier
		Identifier.
	.PARAMETER Find
		FileName: File name.
		WindowTitle: Window title.
	.EXAMPLE
		Test-ProcessRunning -Identifier 'explorer.exe' -Find FileName
	.EXAMPLE
		Test-ProcessRunning -Identifier 'Calculator' -Find WindowTitle
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-ProcessRunning.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Identifier,
		[Parameter(Mandatory=$false)][ValidateSet("FileName", "WindowTitle")][string]$Find = "FileName"
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$found = @();
			
			if ($Find -eq "FileName")
			{
				$found = @(Get-Process | where { [System.IO.Path]::GetFileName($_.MainModule.FileName) -like $Identifier });
			}
			elseif ($Find -eq "WindowTitle")
			{
				$found = @(Get-Process | where { $_.MainWindowTitle -like $Identifier });
			}
			
			Write-Log -Message "Found process with $($Find) '$($Identifier)': $($found.Count)." -Source ${CmdletName};
			foreach ($process in $found)
			{
				Write-Log -Message "* Process ID $($process.Id), Name '$($process.ProcessName)', Path '$($process.MainModule.FileName)', Title '$($process.MainWindowTitle)'." -Source ${CmdletName};
			}
			
			return ($found.Count -gt 0);
		}
		catch
		{
			$failed = "Failed to find process with $Find '$Identifier'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-StringElement {
	<#
	.SYNOPSIS
		Determines an element from a character string separated by separators and stores it in a variable.
	.DESCRIPTION
		Determines an element from a character string separated by separators and stores it in a variable.
	.PARAMETER String
		List of Strings. The character string from which an element is to be selected.
	.PARAMETER Separator
		The character that separates the individual elements in the string. There are two options
	.PARAMETER Index	
		Specifies which element is to be read. The character string is evaluated from the left; counting starts at 0 (0 = first element).
	.PARAMETER ResultVariable
		Name of the variable in which the read out substring is to be stored.
	.PARAMETER SeparateChars
		If several characters are entered as separators, each individual character is interpreted as a separator.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-StringElement -String $_ProcessList -Separator ';' -Index $_ProcessCounter -ResultVariable _CurrentProcess -SeparateChars
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-StringElement.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$String,
		[Parameter(Mandatory=$true)][string]$Separator,
		[Parameter(Mandatory=$true)][string]$Index,
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][switch]$SeparateChars = $false,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			[int]$typedIndex = $Index;
			$typedSeparator = $(if ($SeparateChars) { [char[]]$Separator.ToCharArray() } else { [string[]]@($Separator) });
			$elements = $String.Split($typedSeparator, [System.StringSplitOptions]::None);
			$result = $null;
			if (($typedIndex -ge 0) -and ($typedIndex -lt $elements.Count))
			{
				$result = $elements[$typedIndex]
				Write-Log -Message "Input: '$($String)', separator '$($Separator)', elements: $($elements.Count) - element with index $($Index) is: '$($result)'." -Source ${CmdletName};
				Set-PdVar -Name $ResultVariable -Value $result;
			}
			else
			{
				$result = "";
				Write-Log -Message "Input: '$($String)', separator '$($Separator)', elements: $($elements.Count) - index $($Index) is out of bounds." -Source ${CmdletName};
				Set-PdVar -Name $ResultVariable -Value $result;
			};
		}
		catch
		{
			$failed = "Failed to extract string element";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-LeftString {
	<#
	.SYNOPSIS
		The command copies a fixed number of characters starting at the left side of a string and stores the resulting string in a variable
	.DESCRIPTION
		The command copies a fixed number of characters starting at the left side of a string and stores the resulting string in a variable
	.PARAMETER String
		Character string from which characters are to be copied.
	.PARAMETER Length
		Number of characters to be copied from the string. If 0 is specified, no character is copied.
	.PARAMETER ResultVariable
		Name of the variable in which the read out substring is to be stored. 
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-LeftString -String $Env:SystemDrive -Length 1 -ResultVariable _DriveLetter
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-LeftString.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$String,
		[Parameter(Mandatory=$true)][string]$Length,
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			[int]$validatedLength = $(if ([int]$Length -lt 0) { 0 } elseif ([int]$Length -gt $String.Length) { $String.Length } else { $Length });
			if ([int]$Length -ne $validatedLength) { Write-Log -Message "Length adjusted to: $($validatedLength)." -Source ${CmdletName}; }
			$result = $String.Substring(0, $validatedLength);
			Write-Log -Message "Input: '$($String)' - left string with length $($Length) is: '$($result)'." -Source ${CmdletName};
			Set-PdVar -Name $ResultVariable -Value $result;
		}
		catch
		{
			$failed = "Failed to extract left string";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-MidString {
	<#
	.SYNOPSIS
		Copies a substring of fixed length from a certain character of a string and stores the substring in a variable.
	.DESCRIPTION
		Copies a substring of fixed length from a certain character of a string and stores the substring in a variable.
	.PARAMETER String
		Character string from which characters are to be copied.
	.PARAMETER Start
		Character of the string from which the substring is determined. Counting starts from the left.
	.PARAMETER Length
		Number of characters to be copied from the string. If 0 is specified, no character is copied.
	.PARAMETER ResultVariable
		Name of the variable in which the read out substring is to be stored. 
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-MidString -String $_UNCPath -Start 3 -Length 8 -ResultVariable _Servername
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-MidString.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$String,
		[Parameter(Mandatory=$true)][string]$Start,
		[Parameter(Mandatory=$true)][string]$Length,
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			[int]$validatedStartIndex = $(if ([int]$Start -lt 1) { 0 } elseif ([int]$Start -gt $String.Length) { $String.Length } else { ($Start - 1) });
			if ([int]$Start -ne ($validatedStartIndex + 1)) { Write-Log -Message "Start adjusted to: $($validatedStartIndex + 1)." -Source ${CmdletName}; }
			[int]$validatedLength = $(if ([int]$Length -lt 0) { 0 } elseif (($validatedStartIndex + [int]$Length) -gt $String.Length) { ($String.Length - $validatedStartIndex) } else { $Length });
			if ([int]$Length -ne $validatedLength) { Write-Log -Message "Length adjusted to: $($validatedLength)." -Source ${CmdletName}; }
			$result = $String.Substring($validatedStartIndex, $validatedLength);
			Write-Log -Message "Input: '$($String)' - partial string starting at $($Start) with length $($Length) is: '$($result)'." -Source ${CmdletName};
			Set-PdVar -Name $ResultVariable -Value $result;
		}
		catch
		{
			$failed = "Failed to extract partial string";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-ReplacePattern {
	<#
	.SYNOPSIS
		Exchanges string with another string.
	.DESCRIPTION
		Searches for a character string in a text line and exchanges it with another character string.
	.PARAMETER String
		Character string from which characters are to be copied.
	.PARAMETER Find
		Substring to be searched for.
	.PARAMETER Replace
		The substring found is replaced by this string.
	.PARAMETER ResultVariable
		Name of the variable in which the read out substring is to be stored. 
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-ReplacePattern -String $_String -Find 'in die' -Replace 'nicht nur' -ResultVariable _replaced
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-ReplacePattern.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$String,
		[Parameter(Mandatory=$true)][string]$Find,
		[Parameter(Mandatory=$true)][string]$Replace,
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$escapedFind = [regex]::Escape($Find);
			$matches = [regex]::Matches($escapedFind, "\\[\*\?]");
			$lastIndex = 0;
			$findAnyStringCount = 0;
			$findAnyCharCount = 0;
			$findPattern = "^";
			for ($i = 0; $i -lt $matches.Count; $i++)
			{
				$match = $matches[$i];
				$findPattern += $escapedFind.Substring($lastIndex, ($match.Index - $lastIndex));
				if ($match.Value -eq "\*")
				{
					$findPattern += "(?<s$($findAnyStringCount)>.*?)";
					$findAnyStringCount++;
				}
				elseif ($match.Value -eq "\?")
				{
					$findPattern += "(?<c$($findAnyCharCount)>.)";
					$findAnyCharCount++;
				}
				else
				{
					$findPattern += $match.Value;
				}
				
				$lastIndex = ($match.Index + $match.Length);
			}
			if ($lastIndex -lt $escapedFind.Length)
			{
				$findPattern += $escapedFind.Substring($lastIndex);
			}
			$findPattern += "`$";

			$matches = [regex]::Matches($Replace, "[\*\?]");
			$lastIndex = 0;
			$replaceAnyStringCount = 0;
			$replaceAnyCharCount = 0;
			$replacePattern = "";
			for ($i = 0; $i -lt $matches.Count; $i++)
			{
				$match = $matches[$i];
				$replacePattern += $Replace.Substring($lastIndex, ($match.Index - $lastIndex));
				if (($match.Value -eq "*") -and ($replaceAnyStringCount -lt $findAnyStringCount))
				{
					$replacePattern += "`${s$($replaceAnyStringCount)}";
					$replaceAnyStringCount++;
				}
				elseif (($match.Value -eq "?") -and ($replaceAnyCharCount -lt $findAnyCharCount))
				{
					$replacePattern += "`${c$($replaceAnyCharCount)}";
					$replaceAnyCharCount++;
				}
				else
				{
					$replacePattern += $match.Value;
				}
				
				$lastIndex = ($match.Index + $match.Length);
			}
			if ($lastIndex -lt $Replace.Length)
			{
				$replacePattern += $Replace.Substring($lastIndex);
			}

			$result = $null;
			if ([regex]::IsMatch($String, $findPattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase))
			{
				$result = [regex]::Replace($String, $findPattern, $replacePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase);
				Write-Log -Message "Input: '$($String)', find pattern '$($Find)' (RegEx: '$($findPattern)'), replace by pattern '$($replace)' (RegEx: '$($replacePattern)') - result is: '$($result)'." -Source ${CmdletName};
			}
			else
			{
				$result = $String;
				Write-Log -Message "Input: '$($String)', find pattern '$($Find)' (RegEx: '$($findPattern)'), replace by pattern '$($replace)' (RegEx: '$($replacePattern)') - no match." -Source ${CmdletName};
			}
			
			Set-PdVar -Name $ResultVariable -Value $result;
		}
		catch
		{
			$failed = "Failed to replace string pattern";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-ReplaceString {
	<#
	.SYNOPSIS
		Exchanges all occurrences of this substring for the specified substring
	.DESCRIPTION
		Searches for a specified substring in a string and exchanges all occurrences of this substring for the specified substring
	.PARAMETER String
		Character string from which characters are to be copied.
	.PARAMETER Find
		Substring to be searched for.
	.PARAMETER Replace
		The substring found is replaced by this string.
	.PARAMETER ResultVariable
		Name of the variable in which the read out substring is to be stored. 
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-ReplaceString -String Abracadabra -Find bra -Replace nim -ResultVariable _replaced
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-ReplaceString.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$String,
		[Parameter(Mandatory=$true)][string]$Find,
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$Replace,
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$result = $String.Replace($Find, $Replace);
			Write-Log -Message "Input: '$($String)', find '$($Find)', replace by '$($replace)' - result is: '$($result)'." -Source ${CmdletName};
			Set-PdVar -Name $ResultVariable -Value $result;
		}
		catch
		{
			$failed = "Failed to replace string";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-RightString {
	<#
	.SYNOPSIS
		The command copies a fixed number of characters starting at the right side of a string and stores the resulting string in a variable.
	.DESCRIPTION
		The command copies a fixed number of characters starting at the right side of a string and stores the resulting string in a variable.
	.PARAMETER String
		Character string from which characters are to be copied.
	.PARAMETER Length
		Number of characters to be copied from the string. If 0 is specified, no character is copied.
	.PARAMETER ResultVariable
		Name of the variable in which the read out substring is to be stored. 
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-RightString -String 'Lorem Ipsum' -Length 5 -ResultVariable _rightStringValue
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-RightString.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$String,
		[Parameter(Mandatory=$true)][string]$Length,
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			[int]$validatedLength = $(if ([int]$Length -lt 0) { 0 } elseif ([int]$Length -gt $String.Length) { $String.Length } else { $Length });
			if ([int]$Length -ne $validatedLength) { Write-Log -Message "Length adjusted to: $($validatedLength)." -Source ${CmdletName}; }
			$result = $String.Substring(($String.Length - $validatedLength), $validatedLength);
			Write-Log -Message "Input: '$($String)' - right string with length $($Length) is: '$($result)'." -Source ${CmdletName};
			Set-PdVar -Name $ResultVariable -Value $result;
		}
		catch
		{
			$failed = "Failed to extract right string";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Add-EnvironmentPath {
	<#
	.SYNOPSIS
		Changes PATH-environment variable.
	.DESCRIPTION
		This command supplements the path information stored in the PATH-environment variable.
	.PARAMETER Path
		Directory path to be added to the PATH The use of variables is possible. 
	.PARAMETER Position
		Since some applications can only evaluate a certain number of characters of the PATH environment variable, this option allows to insert the new path at the beginning. 
		If the option is not activated, the path is appended to the end.
	.PARAMETER DialogTitle
		Title bar of the dialog box, which informs the user if necessary.
	.PARAMETER ConfirmChanges
		This parameter is not supported.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Add-EnvironmentPath -Path ";`$(env:ProgramFiles}\CANCOM\PackagingPowerBench\PackagingPowerBench.exe" -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Add-EnvironmentPath.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][string]$Position = $null,
		[Parameter(Mandatory=$false)][string]$DialogTitle = $null,
		[Parameter(Mandatory=$false)][switch]$ConfirmChanges = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if ($ConfirmChanges -or ![string]::IsNullOrEmpty($DialogTitle) -or ![string]::IsNullOrEmpty($Position))
			{
				Write-Log -Message "The parameters -ConfirmChanges, -DialogTitle and -Position are not supported." -Severity 2 -Source ${CmdletName};
			}
			
			$pdc = Get-PdContext;
			$isUserContext = (($Context -eq "User") -or ($Context -eq "UserPerService") -or (($Context -ne "Computer") -and ($Context -ne "ComputerPerService") -and !$pdc.SessionZero -and ($pdc.DeployMode -eq "Interactive") -and ($pdc.InstallMode -ne "InstallComputerPart")));
			# $keyPath = $(if ($isUserContext) {"HKCU:\Environment"} else {"HKLM:\SYSTEM\CurrentControlSet\Control\Session Manager\Environment"});
			$doPrepend = $Path.EndsWith(";"); # ";<Path>" = append, "<Path>;" = prepend
			$rawPath = Expand-Path $Path.Trim(";") -Wow64:$Wow64; # ";<Path>", "<Path>;" -> "<Path>"
			$target = $(if ($isUserContext) { [System.EnvironmentVariableTarget]::User } else { [System.EnvironmentVariableTarget]::Machine });
			
			$name = "PATH";
			$value = [System.Environment]::GetEnvironmentVariable($name, $target);
			
			if (@($value.Split(";") | % { $_.TrimEnd("\/") }) -contains $rawPath.Replace("/", "\").Replace("\\", "\").TrimEnd("\"))
			{
				Write-Log -Message "Environment variable '$($name)' for target '$($target)' already contains extension '$($rawPath)': '$($value)'." -Source ${CmdletName};
				return;
			}
			
			$result = $value; $action = "?";
			if ([string]::IsNullOrEmpty($value)) { $result = $rawPath; $action = "set"; }
			elseif ($doPrepend) { $result = [string]::Join(";", @($rawPath, $value.TrimStart(";"))); $action = "prepend"; }
			else { $result = [string]::Join(";", @($value.TrimEnd(";"), $rawPath)); $action = "append"; }
			
			Write-Log -Message "Environment variable '$($name)' for target '$($target)' - extension '$($rawPath)', action: $($action) '$($rawPath)', result: '$($result)'." -Source ${CmdletName};
			[System.Environment]::SetEnvironmentVariable($name, $result, $target);
		}
		catch
		{
			$failed = "Failed to extend PATH environment variable";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Edit-OemLine {
	<#
	.SYNOPSIS
		Insert or delete a line in ASCII/ANSI files
	.DESCRIPTION
		This command is specific to ASCII/ANSI files, such as AUTOEXEC.BAT 
		and CONFIG.SYS. These files do not support the "Windows Ini-File syntax" 
		(file > section > key > value), so commands must be specified more precisely.
	.PARAMETER FileName
		File to be changed.
	.PARAMETER Arguments
		<Beginning>: At the start of the document 
		<End>: At the end of the document
		<Replace All>: Replaces all texts form OldLine with NewLine
	.PARAMETER OldLine
		This Text will be overwritten
	.PARAMETER NewLine
		The content to be inserted, deleted or replaced.
	.PARAMETER Action
		- Insert: Inserts a line.
		- Delete: Deletes a line.
		- Replace: Replaces an existing line.
	.PARAMETER ConfirmChanges
		This parameter is not supported.
	.PARAMETER ReturnStatus
		For internal Use only
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Edit-OemLine -FileName "${env:SystemRoot}\system32\drivers\etc\hosts" -Arguments '<End>' -NewLine '127.0.0.1 testaddress.de' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Edit-OemLine.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FileName,
		[Parameter(Mandatory=$false)][string]$Arguments = "",
		[Parameter(Mandatory=$false)][string]$OldLine = "",
		[Parameter(Mandatory=$false)][string]$NewLine = "",
		[Parameter(Mandatory=$false)][switch]$ConfirmChanges = $false,
		[Parameter(Mandatory=$false)][switch]$ReturnStatus = $false, # internal - see Add-OemEnvironmentPath
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		[bool]$success = $false;
		
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if ($ConfirmChanges)
			{
				Write-Log -Message "Parameter -ConfirmChanges is not supported." -Severity 2 -Source ${CmdletName};
			}

			# decode -Arguments
			$afterLine = $null;
			$afterLinePattern = $null;
			$beforeLine = $null;
			$beforeLinePattern = $null;
			$action = $(if ([string]::IsNullOrEmpty($OldLine)) {"Insert"} elseif ([string]::IsNullOrEmpty($NewLine)) {"Delete"} else {"Replace"});
			$instance = "Last";
			$insertWhenNotFound = $true;
			
			if (![string]::IsNullOrEmpty($Arguments))
			{
				$argumentPattern = '<(?<n>([^>]|>[^<$])+)>';
				$positionPattern = '^\s*<(?<i>\w+)\s*"(?<m>.+)"\s*>\s*$';
				$matches = [regex]::Matches($Arguments, $argumentPattern);
				foreach ($match in $matches)
				{
					$position = [regex]::Match($match.Value, $positionPattern);
					if ($position.Success)
					{
						$name = $position.Groups['i'].Value;
						$line = $position.Groups['m'].Value;

						if ($name -eq "After") { $afterLinePattern = ConvertTo-RegexPattern $line; }
						elseif ($name -eq "Before") { $beforeLinePattern = ConvertTo-RegexPattern $line; }
					}
					else
					{
						$name = $match.Groups['n'].Value;
						
						if ($name -eq "Replace") { $insertWhenNotFound = $false; }
						elseif ($name -eq "Replace All") { $instance = "All"; $insertWhenNotFound = $false; }
						elseif ($name -eq "Beginning") { $instance = "First"; }
						elseif ($name -eq "End") { $instance = "Last"; }
					}
				}
			}

			$oldLinePattern, $newLineReplace = ConvertTo-RegexPattern $OldLine $NewLine;
			
			if (![System.IO.Path]::IsPathRooted($FileName) -and !$FileName.StartsWith(".\"))
			{
				$FileName = [System.IO.Path]::GetFullPath([System.IO.Path]::Combine([Environment]::GetFolderPath([Environment+SpecialFolder]::Windows), $FileName.TrimStart(".\/".ToCharArray())));
			}

			$FileName = Expand-Path $FileName -Wow64:$Wow64;
			
			if (!(Test-Path -Path $FileName))
			{
				Write-Log -Message "OEM file '$($FileName)' does not exist." -Source ${CmdletName};
				if (($action -eq "Insert") -or (($action -eq "Replace") -and $insertWhenNotFound))
				{
					$dir = [System.IO.Path]::GetDirectoryName($FileName);
					if (![System.IO.Directory]::Exists($dir))
					{
						Write-Log -Message "Creating OEM file directory '$($dir)'." -Source ${CmdletName};
						$void = [System.IO.Directory]::CreateDirectory($dir);
					}
					Write-Log -Message "Creating OEM file '$($FileName)'." -Source ${CmdletName};
					[System.IO.File]::WriteAllText($FileName, $null);
				}
				else
				{
					Write-Log -Message "Skipping command (action '$($action)', instance mode '$($instance)')." -Source ${CmdletName};
					return;
				}
			}

			$fileInfo = Get-TextWithEncoding -path $FileName;
			$textInfo = Get-TextLines -text $fileInfo.Text;
			[System.Collections.Generic.List[string]]$lines = $textInfo.Lines;

			$hasTrailingLineBreak = (($lines.Count -gt 0) -and [string]::IsNullOrEmpty($lines[$lines.Count - 1]));
			if ($hasTrailingLineBreak) { $lines.RemoveAt($lines.Count - 1) }

			$afterLineIndex = -1;
			if (![string]::IsNullOrEmpty($afterLinePattern))
			{
				$index = $lines.FindIndex([System.Predicate[string]]{ [regex]::IsMatch($args[0], $afterLinePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase) });
				if ($index -ge 0) { $afterLineIndex = $index; }
			}

			$beforeLineIndex = $lines.Count;
			if (![string]::IsNullOrEmpty($beforeLinePattern))
			{
				$index = $lines.FindIndex([System.Predicate[string]]{ [regex]::IsMatch($args[0], $beforeLinePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase) });
				if ($index -ge 0) { $beforeLineIndex = $index; }
			}
			
			if ($afterLineIndex -ge $beforeLineIndex)
			{
				throw "No space between line $($afterLineIndex) matching '$($afterLine)' [$($afterLinePattern)] (after) and line $($beforeLineIndex) matching '$($beforeLine)' [$($beforeLinePattern)] (before).";
			}

			$matchCount = 0;
			$insertCount = 0;
			
			if ($action -eq "Insert")
			{
				if ($instance -eq "First")
				{
					$lines.Insert(($afterLineIndex + 1), $NewLine);
					$insertCount++;
					$success = $true;
				}
				elseif ($instance -eq "Last")
				{
					$lines.Insert($beforeLineIndex, $NewLine);
					$insertCount++;
					$success = $true;
				}
				else
				{
					throw "Invalid instance mode '$($instance)' for action '$($action)'.";
				}
			}
			elseif ($action -eq "Replace")
			{
				if ($instance -eq "First")
				{
					$index = $lines.FindIndex(($afterLineIndex + 1), [System.Predicate[string]]{ [regex]::IsMatch($args[0], $oldLinePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase) });
					if ($index -ge 0) { $lines[$index] = [regex]::Replace($lines[$index], $oldLinePattern, $newLineReplace, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase); $matchCount++; $success = $true; }
					elseif ($insertWhenNotFound) { $lines.Insert(($afterLineIndex + 1), $NewLine); $insertCount++; $success = $true; }
				}
				elseif ($instance -eq "Last")
				{
					$index = $lines.FindLastIndex([math]::Max(($beforeLineIndex - 1), 0), [System.Predicate[string]]{ [regex]::IsMatch($args[0], $oldLinePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase) });
					if ($index -ge 0) { $lines[$index] = [regex]::Replace($lines[$index], $oldLinePattern, $newLineReplace, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase); $matchCount++; $success = $true; }
					elseif ($insertWhenNotFound) { $lines.Insert($beforeLineIndex, $NewLine); $insertCount++; $success = $true; }
				}
				elseif ($instance -eq "All")
				{
					for ($i = ($afterLineIndex + 1); $i -lt $beforeLineIndex; $i++)
					{
						if ([regex]::IsMatch($lines[$i], $oldLinePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase))
						{
							$lines[$i] = [regex]::Replace($lines[$i], $oldLinePattern, $newLineReplace, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase);
							$matchCount++;
							$success = $true;
						}
					}
				}
				else
				{
					throw "Invalid instance mode '$($instance)' for action '$($action)'.";
				}
			}
			elseif ($action -eq "Delete")
			{
				if ($instance -eq "First")
				{
					$index = $lines.FindIndex(($afterLineIndex + 1), [System.Predicate[string]]{ [regex]::IsMatch($args[0], $oldLinePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase) });
					if ($index -ge 0) { $lines.RemoveAt($index); $matchCount++; $success = $true; }
				}
				elseif ($instance -eq "Last")
				{
					$index = $lines.FindLastIndex([math]::Max(($beforeLineIndex - 1), 0), [System.Predicate[string]]{ [regex]::IsMatch($args[0], $oldLinePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase) });
					if ($index -ge 0) { $lines.RemoveAt($index); $matchCount++; $success = $true; }
				}
				elseif ($instance -eq "All")
				{
					[string[]]$copy = @();
					
					for ($i = 0; $i -lt $lines.Count; $i++)
					{
						if (($i -gt $afterLineIndex) -and ($i -lt $beforeLineIndex) -and [regex]::IsMatch($lines[$i], $oldLinePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase))
						{
							$matchCount++;
							$success = $true;
							continue;
						}
						
						$copy += $lines[$i];
					}
					
					$lines = $copy;
				}
				else
				{
					throw "Invalid instance mode '$($instance)' for action '$($action)'.";
				}
			}
			else
			{
				throw "Unknown action '$($action)'.";
			}

			if ($hasTrailingLineBreak) { $lines.Add("") }

			Write-Log -Message "File '$($FileName)'" -Source ${CmdletName};
			$insertWhenNotFoundInfo = $(if (($action -eq "Replace") -and $insertWhenNotFound) {" (insert when not found)"} else {""});
			Write-Log -Message " - action '$($action)'$($insertWhenNotFoundInfo), instance mode '$($instance)'" -Source ${CmdletName};
			Write-Log -Message " - initial lines $($textInfo.Lines.Count), restrict $($afterLineIndex + 1)-$([math]::Max(($beforeLineIndex - 1), 0))" -Source ${CmdletName};
			Write-Log -Message " - find '$($OldLine)' [$($oldLinePattern)], replace by '$($NewLine)' [$($newLineReplace)]" -Source ${CmdletName};
			Write-Log -Message " - success $($success), match count $($matchCount), insert count $($insertCount), resulting lines $($lines.Count)" -Source ${CmdletName};
			
			if ($success)
			{
				$text = [string]::Join($textInfo.NewLine, [string[]]$lines);
				[System.IO.File]::WriteAllText($FileName, $text, $fileInfo.Encoding);
			}
		}
		catch
		{
			$success = $false;
			$failed = "Failed to modify OEM file [$FileName]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($ReturnStatus)
			{
				$success; # return $success;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Edit-OemText {
	<#
	.SYNOPSIS
		Insert or delete multiple lines in ASCII/ANSI files
	.DESCRIPTION
		This command is specifically designed for ASCII/ANSI files and can be used to insert or delete multiple lines
	.PARAMETER FileName
		File to be changed.
	.PARAMETER OldLine
		This Text will be overwritten
	.PARAMETER TcpIpKeyColumn
		Specify the Index Column of the TCP/IP Configuration format:
		
		Example:
		ip      0     IP       # Internet Protocol 
		icmp     1     ICMP     # Internet Control Message Protocol 
		This file has the unique key (=index column) in column 2
	.PARAMETER Action
		- Prepend: Inserts the lines specified in text at the beginning of the file.
		- Append: Appends the lines specified in text to the end of the file.
		- InsertAlphabetically: Sorts the lines specified in text alphabetically into the file.
		- Delete: Deletes the lines specified in text from the file.
	.PARAMETER InsertMode
		- New: Inserts only those lines that are not yet present in the existing file.
		- Existing: Existing lines in the existing file are overwritten.
		- MatchOldLine: Overwrites the specified line with text.
		- Always: Always inserts the lines, regardless of whether they already exist.
	.PARAMETER IsTcpIpConfigFormat
		Using this option you can specify the structure of the file. 
	.PARAMETER Unicode
		Specifies that the file to be modified should be saved in Unicode format.
	.PARAMETER NewLine
		Text to be inserted or deleted.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Edit-OemText -FileName "${env:SystemRoot}\system32\drivers\etc\hosts" -Action Append -InsertMode Existing -NewLine '127.0.0.1 testaddress.de' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Edit-OemText.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FileName,
		[Parameter(Mandatory=$false)][string]$OldLine = "",
		[Parameter(Mandatory=$false)][string]$TcpIpKeyColumn = "",
		[Parameter(Mandatory=$false)][ValidateSet("Prepend", "Append", "InsertAlphabetically", "Delete")][string]$Action = "Append",
		[Parameter(Mandatory=$false)][ValidateSet("New", "Existing", "MatchOldLine", "Always")][string]$InsertMode = "Existing",
		[Parameter(Mandatory=$false)][switch]$IsTcpIpConfigFormat = $false,
		[Parameter(Mandatory=$false)][switch]$Unicode = $false,
		[Parameter(Mandatory=$false)][string[]]$NewLine = @(),
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		[bool]$success = $false;

		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if ($IsTcpIpConfigFormat)
			{
				throw "The parameters -IsTcpIpConfigFormat and -TcpIpKeyColumn are not supported.";
			}

			if (![System.IO.Path]::IsPathRooted($FileName) -and !$FileName.StartsWith(".\"))
			{
				$FileName = [System.IO.Path]::GetFullPath([System.IO.Path]::Combine([Environment]::GetFolderPath([Environment+SpecialFolder]::Windows), $FileName.TrimStart(".\/".ToCharArray())));
			}

			$FileName = Expand-Path $FileName -Wow64:$Wow64;

			if (!(Test-Path -Path $FileName))
			{
				Write-Log -Message "OEM file '$($FileName)' does not exist." -Source ${CmdletName};
				if (($Action -ne "Delete") -and (($InsertMode -eq "New") -or ($InsertMode -eq "Existing") -or ($InsertMode -eq "Always")))
				{
					$dir = [System.IO.Path]::GetDirectoryName($FileName);
					if (![System.IO.Directory]::Exists($dir))
					{
						Write-Log -Message "Creating OEM file directory '$($dir)'." -Source ${CmdletName};
						$void = [System.IO.Directory]::CreateDirectory($dir);
					}
					Write-Log -Message "Creating OEM file '$($FileName)'." -Source ${CmdletName};
					[System.IO.File]::WriteAllText($FileName, $null);
				}
				else
				{
					Write-Log -Message "Skipping command (action '$($Action)', insert mode '$($InsertMode)')." -Source ${CmdletName};
					return;
				}
			}

			$fileInfo = Get-TextWithEncoding -path $FileName;
			$textInfo = Get-TextLines -text $fileInfo.Text;
			[System.Collections.Generic.List[string]]$lines = $textInfo.Lines;

			$hasTrailingLineBreak = (($lines.Count -gt 0) -and [string]::IsNullOrEmpty($lines[$lines.Count - 1]));
			if ($hasTrailingLineBreak) { $lines.RemoveAt($lines.Count - 1) }
			
			$newLineInfo = Get-TextLines -text ([string]::Join($textInfo.NewLine, [string[]]$NewLine));
			$NewLine = $newLineInfo.Lines;

			if ($Action -eq "InsertAlphabetically") { $NewLine = @($NewLine | sort); }

			$oldLinePattern = $(if ($InsertMode -eq "MatchOldLine") { ConvertTo-RegexPattern $OldLine -anySpace } else { $null });

			if ($Action -eq "InsertAlphabetically") { $NewLine = @($NewLine | sort); }
			
			$matchCount = 0;
			$insertCount = 0;

			if ($Action -eq "Delete")
			{
				[System.Collections.Generic.List[string]]$deleteLinePatterns = @($NewLine | % { ConvertTo-RegexPattern $_ -anySpace; });

				[string[]]$copy = @();
				
				for ($i = 0; $i -lt $lines.Count; $i++)
				{
					$line = $lines[$i];
					
					if ($deleteLinePatterns.Exists([System.Predicate[string]]{ [regex]::IsMatch($line, $args[0], [System.Text.RegularExpressions.RegexOptions]::IgnoreCase) }))
					{
						$matchCount++;
						$success = $true;
						continue;
					}

					$copy += $line;
				}
				
				$lines = $copy;
			}
			elseif ($InsertMode -eq "MatchOldLine")
			{
				$replacePatterns = @($NewLine | % { $void, $item = ConvertTo-RegexPattern $OldLine $_ -anySpace; $item });
				
				[string[]]$copy = @();
				
				for ($i = 0; $i -lt $lines.Count; $i++)
				{
					$line = $lines[$i];
					
					if ([regex]::IsMatch($line, $oldLinePattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase))
					{
						$matchCount++;
						$success = $true;
						$copy += @($replacePatterns | % { [regex]::Replace($line, $oldLinePattern, $_, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase) });
					}
					else
					{
						$copy += $line;
					}
				}
				
				$lines = $copy;
			}
			elseif (($InsertMode -eq "New") -or ($InsertMode -eq "Existing") -or ($InsertMode -eq "Always"))
			{
				[string[]]$insertLines = $NewLine;
				
				if ($InsertMode -ne "Always")
				{
					$insertLines = @();
					$overwrite = ($InsertMode -eq "Existing");
					
					foreach ($line in $NewLine)
					{
						$pattern = "^\s*$([regex]::Escape($line.Trim()))\s*`$";
						$index = $lines.FindIndex([System.Predicate[string]]{ [regex]::IsMatch($args[0], $pattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase) });
						if ($index -lt 0) { $insertLines += $line; }
						elseif ($overwrite) { $lines[$index] = $line; $matchCount++; }
					}
				}
				
				$insertCount += $insertLines.Count;
				$success = (($matchCount + $insertCount) -gt 0);
				
				if ($Action -eq "Prepend")
				{
					$lines.InsertRange(0, $insertLines);
				}
				elseif ($Action -eq "Append")
				{
					$lines.AddRange($insertLines);
				}
				elseif ($Action -eq "InsertAlphabetically")
				{
					foreach ($line in $insertLines)
					{
						$trimmedLine = $line.Trim();
						$index = $lines.FindIndex([System.Predicate[string]]{ $trimmedLine -lt $args[0].Trim() })
						if ($index -lt 0) { $lines.Add($line) }
						else { $lines.Insert($index, $line); }
					}
				}
				else
				{
					throw "Unsupported: action '$($Action)', insert mode '$($InsertMode)'.";
				}
			}
			else
			{
				throw "Unsupported: action '$($Action)', insert mode '$($InsertMode)'.";
			}

			if ($hasTrailingLineBreak) { $lines.Add("") }

			Write-Log -Message "File '$($FileName)'" -Source ${CmdletName};
			Write-Log -Message " - action '$($Action)', insert mode '$($InsertMode)'" -Source ${CmdletName};
			Write-Log -Message " - initial lines $($textInfo.Lines.Count)" -Source ${CmdletName};
			if ($InsertMode -eq "MatchOldLine") { Write-Log -Message " - replace '$($OldLine)' [$($oldLinePattern)]" -Source ${CmdletName}; }
			Write-Log -Message " - success $($success), match count $($matchCount), insert count $($insertCount), resulting lines $($lines.Count)" -Source ${CmdletName};
			
			if ($success)
			{
				$text = [string]::Join($textInfo.NewLine, [string[]]$lines);
				
				$encoding = $fileInfo.Encoding;
				if ($Unicode)
				{
					$encoding = [System.Text.Encoding]::Unicode;
					Write-Log -Message "Using Unicode character set [$($encoding.WebName)]." -Source ${CmdletName};
				}
				
				[System.IO.File]::WriteAllText($FileName, $text, $encoding);
			}
		}
		catch
		{
			$failed = "Failed to modify OEM file [$FileName]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Add-OemEnvironmentPath {
	<#
	.SYNOPSIS
		Extension of the PATH statement in any batch file (*.BAT).
	.DESCRIPTION
		Extension of the PATH statement in any batch file (*.BAT).
	.PARAMETER Path
		Directory path to be added to the PATH
	.PARAMETER Position
		Since some applications can only evaluate a certain number of characters of the PATH expression, this option allows to insert the new path at the beginning. 
		If the option is not activated, the path is appended to the end.
	.PARAMETER DialogTitle
		Title bar of the dialog box, which informs the user if necessary.
	.PARAMETER BatchFile
		The name (with path if necessary) of the batch file to be changed.
	.PARAMETER ConfirmChanges
		This Parameter is not supported.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Add-OemEnvironmentPath -Path ';C:\batch' -FileName '.\Files\myBatch.bat' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Add-OemEnvironmentPath.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][string]$Position = "",
		[Parameter(Mandatory=$false)][string]$DialogTitle = $null,
		[Parameter(Mandatory=$true)][string]$BatchFile,
		[Parameter(Mandatory=$false)][switch]$ConfirmChanges = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if ($ConfirmChanges)
			{
				Write-Log -Message "Parameter -ConfirmChanges is not supported." -Severity 2 -Source ${CmdletName};
			}

			if (![System.IO.Path]::IsPathRooted($BatchFile) -and !$BatchFile.StartsWith(".\"))
			{
				$BatchFile = [System.IO.Path]::GetFullPath([System.IO.Path]::Combine([Environment]::GetFolderPath([Environment+SpecialFolder]::Windows), $BatchFile.TrimStart(".\/".ToCharArray())));
			}

			$BatchFile = Expand-Path $BatchFile -Wow64:$Wow64;
			
			if (!(Test-Path -Path $BatchFile))
			{
				Write-Log -Message "Cannot update PATH in '$($BatchFile)': File not found." -Severity 2 -Source ${CmdletName};
				return;
			}

			$success = $false;
			
			$parameters = @{
				FileName = $BatchFile;
				ConfirmChanges = $ConfirmChanges;
				Wow64 = $Wow64;
				# Context = $Context;
				ContinueOnError = $ContinueOnError;
			}
			
			$prependPath = $Path.EndsWith(";");

			if (!$success)
			{
				$oldLine = "PATH *";
				$newLine = $(if ($prependPath) {"PATH $($Path)*"} else {"PATH *$($Path)"});
				$success = Edit-OemLine @parameters -ReturnStatus -Arguments "<Replace>$($Position)" -OldLine $oldLine -NewLine $newLine;
				Write-Log -Message "Replace '$($oldLine)' by '$($newLine)' in '$($BatchFile)': $(if ($success) {'OK'} else {'Not found'})." -Source ${CmdletName};
			}

			if (!$success)
			{
				$oldLine = "SET PATH=*";
				$newLine = $(if ($prependPath) {"SET PATH=$($Path)*"} else {"SET PATH=*$($Path)"});
				$success = Edit-OemLine @parameters -ReturnStatus -Arguments "<Replace>$($Position)" -OldLine $oldLine -NewLine $newLine;
				Write-Log -Message "Replace '$($oldLine)' by '$($newLine)' in '$($BatchFile)': $(if ($success) {'OK'} else {'Not found'})." -Source ${CmdletName};
			}

			if (!$success)
			{
				$newLine = $(if ($prependPath) {"PATH $($Path)%PATH%"} else {"PATH %PATH%$($Path)"});
				$success = Edit-OemLine @parameters -ReturnStatus -Arguments "$($Position)" -NewLine $newLine;
				Write-Log -Message "Append '$($newLine)' to '$($BatchFile)': $(if ($success) {'OK'} else {'Failed'})." -Source ${CmdletName};
			}

			if (!$success)
			{
				Write-Log -Message "No PATH-pattern found to update '$($BatchFile)'." -Severity 2 -Source ${CmdletName};
			}
		}
		catch
		{
			$failed = "Failed to extend PATH value in [$BatchFile]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-ScriptEngineAndArguments([string]$ScriptPath, [string]$CLRVersion = $null, [string]$ExecutionPolicy = $null, [switch]$Wow64 = $false)
{
	$scriptEngines = @{
		".vbs" = @("VB Script", "%windir%\system32\wscript.exe", "`"{0}`" //B //E:VBScript //NoLogo");
		".js"  = @("JScript", "%windir%\system32\wscript.exe", "`"{0}`" //B //E:JScript //NoLogo");
		# ".vbs" = @("VB Script", "%windir%\system32\wscript.exe", "`"{0}`" //E:VBScript //NoLogo"); # interactive
		# ".js"  = @("JScript", "%windir%\system32\wscript.exe", "`"{0}`" //E:JScript //NoLogo"); # interactive
		# ".vbs" = @("VB Script", "%windir%\system32\cscript.exe", "`"{0}`" //E:VBScript //NoLogo");
		# ".js"  = @("JScript", "%windir%\system32\cscript.exe", "`"{0}`" //E:JScript //NoLogo");
		".pl"  = @("Perl Script", "", "`"{0}`"");
		".ps1" = @("PowerShell Script", "%windir%\System32\WindowsPowerShell\v1.0\powershell.exe", "{1}-NoLogo -WindowStyle Hidden -NonInteractive -File `"{0}`"");
	}

	$extension = [System.IO.Path]::GetExtension($ScriptPath);
	if (!$scriptEngines.ContainsKey($extension)) { throw "Unknown script type '$($extension)'."; }

	$name, $path, $format = $scriptEngines[$extension];
	if ([string]::IsNullOrEmpty($path)) { throw "No script engine for type '$($extension)' ($($name))."; }
	
	$path = Expand-Path ([Environment]::ExpandEnvironmentVariables($path)) -Wow64:$Wow64;
	if (![System.IO.File]::Exists($path)) { throw "Script engine for type '$($extension)' ($($name)) not found at '$($path)'."; }
	
	$extra = "";
	if ($extension -eq ".ps1")
	{
		$parameters = @();
		if (($CLRVersion -eq "v2.0") -or ($CLRVersion -eq "v3.0") -or ($CLRVersion -eq "v3.5")) { $parameters += "-Version 2.0"; }
		if (![string]::IsNullOrEmpty($ExecutionPolicy)) { $parameters += "-ExecutionPolicy $($ExecutionPolicy)"; }
		if ($parameters.Count -gt 0) { $extra = "$([string]::Join(' ', $parameters)) "; }
	}
	
	$arguments = [string]::Format($format, $ScriptPath, $extra);
	
	return $path, $arguments;
}

function Invoke-Script {
	<#
	.SYNOPSIS
		Calls an external script that is processed by a script engine.
	.DESCRIPTION
		Calls an external script that is processed by a script engine.
		Windows scripts that are called via Invoke-Script have the standard features of the respective script language. 
		Objects and methods additionally provided by the Windows Script Host, e.g. WshShell, are not supported. These objects can be accessed via WScript.Shell, for example.  
	.PARAMETER ScriptPath
		Input of a script file with path, variables are allowed. Possible Scripts: VB-Script, JScript, Perl Script, PowerShell Script
	.PARAMETER CLRVersion
		This option is necessary when the command is used to run a PowerShell script 
		It specifies the .NET Framework Runtime version under which the script runs.
		Possible Values are: v2.0, v3.0, v3.5, v4.0
		Don't use this Parameter if you want to use latest CLR Version
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Invoke-Script -ScriptPath '.\Files\Test.ps1'
	.EXAMPLE
		Invoke-Script -ScriptPath '.\Files\Test.ps1' -CLRVersion 'v3.5' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Invoke-Script.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ScriptPath,
		[Parameter(Mandatory=$false)][string]$CLRVersion = $null,
		[Parameter(Mandatory=$false)][string]$ExecutionPolicy = "RemoteSigned",
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$ScriptPath = Expand-Path $ScriptPath -Wow64:$Wow64;
			Write-Log -Message "Running script [$($ScriptPath)]" -Source ${CmdletName};

			$enginePath, $arguments = Get-ScriptEngineAndArguments -ScriptPath $ScriptPath -CLRVersion $CLRVersion -ExecutionPolicy $ExecutionPolicy -Wow64:$Wow64;
			
			$parameters = @{
				FilePath = $enginePath;
				Arguments = $arguments;
				Wait = $true;
				WindowStyle = "Hidden";
				RunAs = "CurrentUser";
				OnError = "Failed";
				Wow64 = $Wow64;
				# Context = $Context;
				ContinueOnError = $ContinueOnError;
			}
			
			Write-Log -Message "Starting script engine '$($enginePath)' with arguments '$($arguments)'." -Source ${CmdletName};
			$process = Start-ProgramAs -PassThru @parameters;
			if ($process -eq $null) { Write-Log -Message "No process object available." -Source ${CmdletName}; }
			elseif (!$process.HasExited) { Write-Log -Message "Script engine process [$($process.StartInfo.FileName), ID $($process.Id)] is still running." -Source ${CmdletName}; }
			elseif ($process.ExitCode -eq 0) { Write-Log -Message "Script engine process [$($process.StartInfo.FileName), ID $($process.Id)] terminated with exit code $($process.ExitCode) (0x$($process.ExitCode.ToString('X8')))." -Source ${CmdletName}; }
			else { throw "Script engine process [$($process.StartInfo.FileName), ID $($process.Id)] terminated with exit code $($process.ExitCode) (0x$($process.ExitCode.ToString('X8')))."; }
		}
		catch
		{
			$failed = "Failed to run script [$($ScriptPath)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Invoke-ScriptAs {
	<#
	.SYNOPSIS
		Call external script as another user
	.DESCRIPTION
		Call external script as another user (like the Start-ProgramAs command). Note that this command requires Windows Vista or later.
	.PARAMETER ScriptPath
		Input of a script file with path, variables are allowed. Possible Scripts: VB-Script, JScript, Perl Script, PowerShell Script
	.PARAMETER CLRVersion
		This option is necessary when the command is used to run a PowerShell script 
		It specifies the .NET Framework Runtime version under which the script runs.
		Possible Values are: v2.0, v3.0, v3.5, v4.0
		Don't use this Parameter if you want to use latest CLR Version
	.PARAMETER RunAs
		Specifies the credentials under which the script runs. Possible Values are:
		UserName (default), DsmAccount, LocalSystem, CurrentUser
	.PARAMETER UserName
		The user account under which the application is to run. 
		This is entered as USERNAME for local accounts or DOMAIN\USERNAME for domain accounts.
	.PARAMETER Password
		The password of the specified account.
	.PARAMETER LeastPrivilege
		If User Account Control (UAC) is enabled, 
		Invoke-ScriptAs always executes the specified script as 
		administrator by default, unless this option is activated.
	.PARAMETER Logon
		If you want the script to run under a defined user account, 
		this option determines how the account profile is handled.
		Possible Values:
		- NoProfile: The user profile is not loaded. No changes are written to the user profile after the script is executed.
		- LoadProfile: The user profile is loaded; changes are written to the user profile.
		- NetworkOnly: The script itself is executed in the context of the current user account. The specified user account is only used for network access, i.e. a logon session is opened under the specified user account. 
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Invoke-ScriptAs -ScriptPath '.\Files\test.ps1' -RunAs LocalSystem -Logon NoProfile -Context Computer
	.EXAMPLE
		Invoke-ScriptAs -ScriptPath '.\Files\test.ps1' -CLRVersion 'v3.5' -RunAs CurrentUser -Logon NoProfile -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Invoke-ScriptAs.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ScriptPath,
		[Parameter(Mandatory=$false)][string]$CLRVersion = $null,
		[Parameter(Mandatory=$false)][string]$ExecutionPolicy = "RemoteSigned",
		[Parameter(Mandatory=$false)][ValidateSet("UserName", "DsmAccount", "LocalSystem", "CurrentUser", "LoggedOnUser")][string]$RunAs = "UserName",
		[Parameter(Mandatory=$false)][string]$UserName = $null,
		[Parameter(Mandatory=$false)][string]$Password = $null,
		[Parameter(Mandatory=$false)][switch]$LeastPrivilege = $false,
		[Parameter(Mandatory=$false)][ValidateSet("NoProfile", "LoadProfile", "NetworkOnly")][string]$Logon = "NoProfile",
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		[hashtable]$displayParameters = $PSBoundParameters;
		if ($true) { @("Password") | where { $displayParameters.ContainsKey($_) } | % { $displayParameters[$_] = "-hidden-" } }
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $displayParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$ScriptPath = Expand-Path $ScriptPath -Wow64:$Wow64;
			$runAsInfo = $(if ($RunAs -eq "UserName") { "$($RunAs) '$($UserName)'" } else { $RunAs });
			Write-Log -Message "Running script [$($ScriptPath)] as [$($runAsInfo)]" -Source ${CmdletName};

			$enginePath, $arguments = Get-ScriptEngineAndArguments -ScriptPath $ScriptPath -CLRVersion $CLRVersion -ExecutionPolicy $ExecutionPolicy -Wow64:$Wow64;
			
			$parameters = @{
				FilePath = $enginePath;
				Arguments = $arguments;
				Wait = $true;
				SecureParameters = $false;
				WindowStyle = "Hidden";
				RunAs = $RunAs;
				UserName = $UserName;
				Password = $Password;
				LeastPrivilege = $LeastPrivilege;
				Logon = $Logon;
				OnError = "Failed";
				Wow64 = $Wow64;
				# Context = $Context;
				ContinueOnError = $ContinueOnError;
			}
			
			Write-Log -Message "Starting script engine '$($enginePath)' with arguments '$($arguments)' as $($runAsInfo)." -Source ${CmdletName};
			$process = Start-ProgramAs -PassThru @parameters;
			if ($process -eq $null) { Write-Log -Message "No process object available." -Source ${CmdletName}; }
			elseif (!$process.HasExited) { Write-Log -Message "Script engine process [$($process.StartInfo.FileName), ID $($process.Id)] is still running." -Source ${CmdletName}; }
			elseif ($process.ExitCode -eq 0) { Write-Log -Message "Script engine process [$($process.StartInfo.FileName), ID $($process.Id)] terminated with exit code $($process.ExitCode) (0x$($process.ExitCode.ToString('X8')))." -Source ${CmdletName}; }
			else { throw "Script engine process [$($process.StartInfo.FileName), ID $($process.Id)] terminated with exit code $($process.ExitCode) (0x$($process.ExitCode.ToString('X8')))."; }
		}
		catch
		{
			$failed = "Failed to run script [$($ScriptPath)] as [$($RunAs)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-FileVersion {
	<#
	.SYNOPSIS
		Reads Version of a file.
	.DESCRIPTION
		Retrieves the version of a file and stores the result in a variable.
	.PARAMETER Path
		Path including file name to the file to be read. The use of variables is possible
	.PARAMETER ResultVariable
		Name of the variable. 
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-FileVersion -Path "$($env:USERPROFILE)\Documents\MyDocument.docx" -ResultVariable _version
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-FileVersion.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$Path = Expand-Path $Path -Wow64:$Wow64;

			if (![System.IO.File]::Exists($Path)) { throw (New-Object System.IO.FileNotFoundException); }
			
			$info = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($Path);
			$result = $info.FileVersion;
			
			Write-Log -Message "File version of '$($Path)': $($result)." -Source ${CmdletName};
			Set-PdVar -Name $ResultVariable -Value $result;
		}
		catch
		{
			$failed = "Failed to get version of [$($Path)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-FileInUse {
	<#
	.SYNOPSIS
		Checks the in-use status
	.DESCRIPTION
		Checks if the specified file is opened by another process.
	.PARAMETER Path
		Enter the path together with file name.
	.PARAMETER Wow64
		By default, the PackageDeployment module assumes to run in a 64-bit PowerShell process on 64-bit systems, 
		and therefore defaults to the 64-bit application location (if different from the 32-bit application location) for file and registry accesses.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-FileInUse -Path '$($env:USERPROFILE)\Documents\MyDocument.docx'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-FileInUse.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			# test always ... if (Test-SkipCommand -Context $Context) { return; }

			$Path = Expand-Path $Path -Wow64:$Wow64;
			Write-Log -Message "Checking in-use status of '$($Path)'." -Source ${CmdletName};

			$result = $false;	

			if (!(Test-Path $Path))
			{
				$result = $false;
				Write-Log -Message "File not found  -> returning in-use: $($result)." -Source ${CmdletName};
				return $result;
			}

			$access = "ReadWrite";
			$info = Get-Item -Path $Path;
			if ($info.IsReadOnly)
			{
				$access = "Read";
				Write-Log -Message "File is read-only - probing $($access)-access." -Source ${CmdletName};
			}

			$fs = $null;
			try
			{
				$fs = [System.IO.File]::Open($Path, [System.IO.FileMode]::Open, [System.IO.FileAccess]$access, [System.IO.FileShare]::None);
				$result = $false;
				Write-Log -Message "Probing exclusive $($access)-access succeeded -> returning in-use: $($result)." -Source ${CmdletName};
			}
			catch
			{
				$ex = $_.Exception.GetBaseException();
				$hr = $ex.HResult;
				$result = $true;
				Write-Log -Message "Probing exclusive $($access)-access failed [$($ex.Message)][HResult $($hr) (0x$($hr.ToString('X8')))] -> returning in-use: $($result)." -Source ${CmdletName};
			}
			finally
			{
				if ($fs -ne $null) { $fs.Close(); }
			}

			return $result;
		}
		catch
		{
			$failed = "Failed to get in-use status of [$($Path)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-ComputerName {
	<#
	.SYNOPSIS
		Checks the workstation name
	.DESCRIPTION
		Checks if the workstation has the specified name.
	.PARAMETER Name
		The name of the workstation. Wildcards (? *) can be used.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-ComputerName -Name MyWorkstation
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-ComputerName.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			# test always ... if (Test-SkipCommand -Context $Context) { return; }

			$pattern = ("^" + [regex]::Escape($Name).Replace("\*", ".*").Replace("\?", ".") + "`$");
			$computerName = [System.Environment]::MachineName;
			$result = [regex]::IsMatch($computerName, $pattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase);
			Write-Log -Message "Comparing computer name '$($computerName)' with '$($Name)' [$($pattern)] - result: $($result)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test computer name [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Split-WmiNamespaceObjectPath([string]$wmiNamespaceObjectPath)
{
	$result = New-Object -TypeName PSObject -Property @{ ComputerName = "."; Namespace = $wmiNamespaceObjectPath; };

	$match = [regex]::Match($wmiNamespaceObjectPath, "^[\\/]{2}(?<computer>[^\\/]+)[\\/](?<namespace>.+)`$");
	if ($match.Success)
	{
		$result.ComputerName = $match.Groups["computer"].Value;
		$result.Namespace = $match.Groups["namespace"].Value;
	}

	return $result;
}

function Get-ApprovedWmiFilter([string]$filter)
{
	if ([string]::IsNullOrEmpty($filter)) { return $filter; }

	$approved = "";
	$remaining = $filter;
	while (![string]::IsNullOrEmpty($remaining))
	{
		$operator = [regex]::Match($remaining, "(\s+(and|or|not))+\s+", [System.Text.RegularExpressions.RegexOptions]::IgnoreCase);
		$comparison = $(if ($operator.Success) { $remaining.Substring(0, $operator.Index) } else { $remaining });
		
		$match = [regex]::Match($comparison, "^\s*(?<name>[^=\s]+)\s*=\s*(?<value>.+?)\s*`$");
		if ($match.Success -and ![regex]::IsMatch($match.Groups["value"].Value, '^\s*(''[^'']*''|"[^"]*"|\d+)\s*$'))
		{
			$comparison = "$($match.Groups['name'].Value)='$($match.Groups['value'].Value.Replace('\', '\\').Replace('''', '\'''))'";
		}
		
		$approved += $comparison;
		if ($operator.Success) { $approved += $operator; }
		
		$remaining = $(if ($operator.Success) { $remaining.Substring($operator.Index + $operator.Length) } else { "" });
	}
	
	return $approved;
}

function Get-WmiCache([string]$namespace, [string]$class)
{
	$result = $null;
	$key = "$($class)@$($namespace)";

	$pdc = Get-PdContext;
	$cache = $pdc.WmiCache;
	if ($cache.ContainsKey($key))
	{
		$result = $cache[$key];
		if ($result.Expires -lt [DateTime]::UtcNow)
		{
			# expired -> remove from cache
			$cache.Remove($key);
			$result = $null;
		}
	}
	
	return $result;
}

function Set-WmiCache([string]$namespace, [string]$class, $items = $null, $expires = $null)
{
	$key = "$($class)@$($namespace)";

	$pdc = Get-PdContext;
	$cache = $pdc.WmiCache;
	if ($items -eq $null)
	{
		if ($cache.ContainsKey($key)) { $cache.Remove($key); }
		return;
	}
	
	$seconds = $(if ($expires -ne $null) { $expires } elseif (($pdc.Package -ne $null) -and ($pdc.Package.WmiCacheTimeout -ne $null)) { $pdc.Package.WmiCacheTimeout } else { 10 });
	if ($seconds -gt 0)
	{
		$item = New-Object -TypeName PSObject -Property @{ Namespace = $namespace; Class = $class; Items = $items; Expires = ([DateTime]::UtcNow.AddSeconds($seconds))};
		$cache[$key] = $item;
	}
}

function Read-WmiObject {
	<#
	.SYNOPSIS
		Execute a simple WMI query.
	.DESCRIPTION
		Execute a simple WMI query.
	.PARAMETER Namespace
		Specifies the namespace to which the connection is to be made and in which 
		the class (object) is located. This value is also set using the WMI browser. 
		The default namespace is \\.\root\CIMV2.
	.PARAMETER Class
		This is the relative path within the namespace and corresponds to the class 
		name for which the number of existing instances on the system is to be determined.
	.PARAMETER Filter
		The query specifies a criterion (similar to the WHERE clause in the SQL query) that must be met. If this criterion 
		matches a found instance of the specified class, the properties of this instance are used.
	.PARAMETER ResultVariablePrefix
		Together with the property names, this entry results in the actual variable name, 
		i.e. the prefix is placed in front of the variable name.
	.PARAMETER PropertyList
		Names of properties selected in the WMI Browser.
		Each line together with the variable prefix results in a variable.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-WmiObject -Namespace '\\.\root\CIMV2' -Class Win32_LogicalDisk -Filter 'Caption=C:' -ResultVariablePrefix _WMI -PropertyList @('FreeSpace','Size')
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-WmiObject.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Namespace,
		[Parameter(Mandatory=$true)][Alias("ObjectPath")][string]$Class,
		[Parameter(Mandatory=$true)][Alias("Query")][string]$Filter = $null,
		[Parameter(Mandatory=$true)][string]$ResultVariablePrefix,
		[Parameter(Mandatory=$true)][string[]]$PropertyList = @(),
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$split = Split-WmiNamespaceObjectPath $Namespace;
			
			$parameters = @{
				ComputerName = $split.ComputerName;
				Namespace = $split.Namespace;
				Class = $Class;
				Filter = (Get-ApprovedWmiFilter $Filter);
				Property = $PropertyList;
			};

			Write-Log -Message "Query WMI [$($parameters.Namespace) on $($parameters.ComputerName)] for '$($parameters.Class)' where '$($parameters.Filter)'." -Source ${CmdletName};
			
			$list = @(Get-WmiObject @parameters);
			if ($list.Count -gt 0)
			{
				$item = $list[0];
				if ($list.Count -eq 1) { Write-Log -Message "Matching WMI object found: [$($item)]." -Source ${CmdletName}; }
				else { Write-Log -Message "Matching WMI objects found: $($list.Count), using first [$($item)]." -Source ${CmdletName}; }
				
				foreach ($property in $PropertyList)
				{
					Set-PdVar -Name "$($ResultVariablePrefix)$($property)" -Value $item.($property);
				}
			}
			else
			{
				Write-Log -Message "No matching WMI objects found." -Source ${CmdletName};
				# according to DSM: do not set any property values
			}
		}
		catch
		{
			$failed = "Failed to query WMI [$($Class), filter '$($Filter)']";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-WmiObjectCount {
	<#
	.SYNOPSIS
		Store the number of instances of a class found in a variable.
	.DESCRIPTION
		Store the number of instances of a class found in a variable.
	.PARAMETER Namespace
		Specifies the namespace to which the connection is to be made and in which the class (object) is located. This value is also set using the WMI browser. 
		The default namespace is \\.\root\CIMV2.
	.PARAMETER Class
		This is the relative path within the namespace and corresponds to the class 
		name for which the number of existing instances on the system is to be determined
	.PARAMETER ResultVariable
		Name of the variable in which the result
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-WmiObjectCount -Namespace '\\.\root\CIMV2' -Class Win32_DiskDrive -ResultVariable _DriveCount
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-WmiObjectCount.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Namespace,
		[Parameter(Mandatory=$true)][Alias("ObjectPath")][string]$Class,
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$split = Split-WmiNamespaceObjectPath $Namespace;
			
			$items = @();

			$parameters = @{
				ComputerName = $split.ComputerName;
				Namespace = $split.Namespace;
				Class = $Class;
			};

			$items = @(Get-WmiObject @parameters);
			Set-WmiCache -namespace $Namespace -class $Class -items $items;
			Write-Log -Message "Result from WMI query [$($parameters.Namespace) on $($parameters.ComputerName)] for '$($parameters.Class)' instance count: $($items.Count)." -Source ${CmdletName};
			
			Set-PdVar -Name $ResultVariable -Value $items.Count;
		}
		catch
		{
			$failed = "Failed to get count of WMI [$($Class)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-IndexedWmiObject {
	<#
	.SYNOPSIS
		Reads the values of properties of a WMI class.
	.DESCRIPTION
		The command reads the values of properties of the n-th instance of a WMI class and stores these values in variables. 
		This enables the specific query of available system and software components.
	.PARAMETER Namespace
		Specifies the namespace to which the connection is to be made and in which the class (object) is located. 
		This value is also set using the WMI browser. 
		The default namespace is \\.\root\CIMV2.
	.PARAMETER Class
		This is the relative path within the namespace and corresponds to the class name 
		for which the number of existing instances on the system is to be determined
	.PARAMETER Index
		Index that specifies which of the instances found is to be used. 
		The index is zero-based, that is, the first instance has the instance number 0.
	.PARAMETER ResultVariablePrefix
		Together with the property names, this entry results in the actual variable name, 
		i.e. the prefix is placed in front of the variable name.
	.PARAMETER PropertyList
		Names of properties selected in the WMI Browser.
	.PARAMETER NoCache
		Per default not set. Disables the use of WMI Cache.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-IndexedWmiObject -Namespace '\\.\root\CIMV2' -Class Win32_BIOS -Index 0 -ResultVariablePrefix _WMI -PropertyList @('Manufacturer','Caption','Description')
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-IndexedWmiObject.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Namespace,
		[Parameter(Mandatory=$true)][Alias("ObjectPath")][string]$Class,
		[Parameter(Mandatory=$true)][int]$Index = 0,
		[Parameter(Mandatory=$true)][string]$ResultVariablePrefix,
		[Parameter(Mandatory=$true)][string[]]$PropertyList = @(),
		[Parameter(Mandatory=$false)][switch]$NoCache = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$items = @();
			
			Write-Log -Message "Get instance with index $($Index) of WMI '$($Class)', namespace '$($Namespace)'." -Source ${CmdletName};
			$cache = $(if ($NoCache) { $null } else { Get-WmiCache -namespace $Namespace -class $Class; });
			if ($cache -ne $null)
			{
				$items = $cache.Items;
				Write-Log -Message "Result from WMI cache - instances: $($items.Count)." -Source ${CmdletName};
			}
			else
			{
				$split = Split-WmiNamespaceObjectPath $Namespace;

				$parameters = @{
					ComputerName = $split.ComputerName;
					Namespace = $split.Namespace;
					Class = $Class;
				};

				$items = @(Get-WmiObject @parameters);
				Set-WmiCache -namespace $Namespace -class $Class -items $items;
				Write-Log -Message "Result from WMI query [$($parameters.Namespace) on $($parameters.ComputerName)] - instances: $($items.Count)." -Source ${CmdletName};
			}
			
			$item = $(if ($Index -lt $items.Count) { $items[$Index] } else { throw "Index $($Index) out of range (0..$($items.Count - 1))"; })
			Write-Log -Message "Matching WMI object with index $($Index) found: [$($item)]." -Source ${CmdletName};
			foreach ($property in $PropertyList)
			{
				Set-PdVar -Name "$($ResultVariablePrefix)$($property)" -Value $item.($property);
			}
		}
		catch
		{
			$failed = "Failed to get WMI [$($Class), index $($Index)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-LocalAccountName([Parameter(Mandatory=$true, ValueFromPipeline=$true)][string]$identifier)
{
	process
	{
		try
		{
			$sidPattern = "^(S|s)-\d+-\d+(-\d+)+`$";
			$parts = $identifier.Split("|", 2);
			$sid = [System.Security.Principal.SecurityIdentifier]$(if ($parts.Count -gt 1) { $parts[1] } elseif ([regex]::IsMatch($parts[0], $sidPattern)) { $parts[0] } else { $null });
			$name = $(if ($sid -ne $null) { $sid.Translate([System.Security.Principal.NTAccount]).Value } else { $parts[0] });
			Write-Log -Message "Name of local account '$($identifier)': $($name)." -Source ${CmdletName};
			return $name;
		}
		catch
		{
			throw "Cannot get SID from identifier '$($identifier)'.`r`n$(Resolve-Error)";
		}
	}
}

function Get-LocalAccountSid([Parameter(Mandatory=$true, ValueFromPipeline=$true)][string]$identifier)
{
	process
	{
		try
		{
			$sidPattern = "^(S|s)-\d+-\d+(-\d+)+`$";
			$parts = $identifier.Split("|", 2);
			$sid = [System.Security.Principal.SecurityIdentifier]$(if ($parts.Count -gt 1) { $parts[1] } elseif ([regex]::IsMatch($parts[0], $sidPattern)) { $parts[0] } else { ([System.Security.Principal.NTAccount]$parts[0]).Translate([System.Security.Principal.SecurityIdentifier]) });
			Write-Log -Message "SID of local account '$($identifier)': $($sid.Value)." -Source ${CmdletName};
			return $sid;
		}
		catch
		{
			throw "Cannot get SID from identifier '$($identifier)'.`r`n$(Resolve-Error)";
		}
	}
}

function Add-LocalGroupMember {
	<#
	.SYNOPSIS
		Add new members to a local group
	.DESCRIPTION
		Use this command to add new members to a local group. 
		You can use this command to add both objects from the domain and local objects 
		to existing local groups on the systems running the command. 
		This makes it possible, for example, to form a group of 
		"workstation administrators" who are not also domain administrators.
	.PARAMETER Group
		The name of the group to which the objects are to be added.	
	.PARAMETER Member
		Names of the objects to be added to the group. Domain objects are referenced according to the scheme <DOMAIN>\<Object>, local objects only with their names. 
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Add-LocalGroupMember -Group TestUsers -Member @('MaxA','TimB') -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Add-LocalGroupMember.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Group,
		[Parameter(Mandatory=$true)][string[]]$Member = @(),
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$groupSID = Get-LocalAccountSid $Group;
			$memberSIDs = @($Member | Get-LocalAccountSid | % { $_.Value });

			$groupMemberSIDs = @(Microsoft.PowerShell.LocalAccounts\Get-LocalGroupMember -SID $groupSID | % { $_.SID.Value });
			$addMemberSIDs = $memberSIDs | where { $groupMemberSIDs -notcontains $_ }

			if ($addMemberSIDs.Count -eq 0) { Write-Log -Message "All specified members are already in local group '$($Group)': $($Member.Count)." -Source ${CmdletName}; return; }
			elseif ($addMemberSIDs.Count -lt $Member.Count) { $skipMemberSIDs = $memberSIDs | where { $groupMemberSIDs -contains $_ }; Write-Log -Message "Skipping members already in local group '$($Group)' ($($skipMemberSIDs.Count)): $([string]::Join(', ', $skipMemberSIDs))." -Source ${CmdletName}; }

			Write-Log -Message "Adding members to the local group '$($Group)': $($addMemberSIDs.Count)." -Source ${CmdletName};
			Microsoft.PowerShell.LocalAccounts\Add-LocalGroupMember -SID $groupSID -Member $addMemberSIDs;
	}
		catch
		{
			$failed = "Failed to add members to local group [$($Group)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function New-LocalGroup {
	<#
	.SYNOPSIS
		Create a new local group
	.DESCRIPTION
		This command creates a new local group on the system on which it is executed.
	.PARAMETER Name
		The name of the group to be created. 	
	.PARAMETER Description
		Description of the new group, as displayed in Computer Management.	
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		New-LocalGroup -Name TestUsers -Description 'Test users only' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/New-LocalGroup.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][string]$Description = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Write-Log -Message "Creating local group '$($Name)'." -Source ${CmdletName};
			$group = Microsoft.PowerShell.LocalAccounts\New-LocalGroup -Name $Name -Description $Description;
		}
		catch
		{
			$failed = "Failed to create local group [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-LocalGroupMember {
	<#
	.SYNOPSIS
		Remove members from a local group.
	.DESCRIPTION
		Use this command to remove members from a local group.
	.PARAMETER Member
		The name(s) of the object(s) for which the membership in the group is to be deleted.
	.PARAMETER Group
		The name of the group from which existing members are to be removed. 	
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-LocalGroupMember -Group TestUsers -Member @('Max Mustermann','Tim Buktu') -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-LocalGroupMember.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Group,
		[Parameter(Mandatory=$true)][string[]]$Member = @(),
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$groupSID = Get-LocalAccountSid $Group;
			$memberSIDs = @($Member | Get-LocalAccountSid | % { $_.Value });

			$groupMemberSIDs = @(Microsoft.PowerShell.LocalAccounts\Get-LocalGroupMember -SID $groupSID | % { $_.SID.Value });
			$removeMemberSIDs = $memberSIDs | where { $groupMemberSIDs -contains $_ }

			if ($removeMemberSIDs.Count -eq 0) { Write-Log -Message "All specified members are already removed from local group '$($Group)': $($Member.Count)." -Source ${CmdletName}; return; }
			elseif ($removeMemberSIDs.Count -lt $Member.Count) { $skipMemberSIDs = $memberSIDs | where { $groupMemberSIDs -notcontains $_ }; Write-Log -Message "Skipping members already removed from local group '$($Group)' ($($skipMemberSIDs.Count)): $([string]::Join(', ', $skipMemberSIDs))." -Source ${CmdletName}; }

			Write-Log -Message "Removing members from the local group '$($Group)': $($removeMemberSIDs.Count)." -Source ${CmdletName};
			Microsoft.PowerShell.LocalAccounts\Remove-LocalGroupMember -SID $groupSID -Member $removeMemberSIDs;
		}
		catch
		{
			$failed = "Failed to remove members from local group [$($Group)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Add-LocalUserMembership {
	<#
	.SYNOPSIS
		Add User Account to Multiple Local Groups
	.DESCRIPTION
		Use this command to add a user account to multiple local groups.
	.PARAMETER User
		Enter the local user.
	.PARAMETER Group
		Displays the local groups to which you want to add the specified local user.	
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Add-LocalUserMembership -User 'Administrator|S-1-5-21-3681506545-2675960050-2856003502-500' -Group TestUsers -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Add-LocalUserMembership.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$User,
		[Parameter(Mandatory=$true)][string[]]$Group = @(),
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Write-Log -Message "Adding memberships to the local user '$($User)': $($Group.Count)." -Source ${CmdletName};
			$userSID = (Get-LocalAccountSid $User).Value;
			$Group | % { Microsoft.PowerShell.LocalAccounts\Add-LocalGroupMember -SID (Get-LocalAccountSid $_) -Member $userSid; }
		}
		catch
		{
			$failed = "Failed to add memberships to local user [$($User)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function New-LocalUser {
	<#
	.SYNOPSIS
		Create a new local user
	.DESCRIPTION
		Use this command to create a new local user on the system on which it is executed.
	.PARAMETER Name
		The name of the user account. A user name must not be the same as any other user or group name on the currently managed computer. It can be up to 20 characters in upper or lower case except for the following: "  /  \  [  ]  :  ;  |  =  ,  +  *  ?  <  >
	.PARAMETER FullName
		The full name of the user
	.PARAMETER Description
		Any text explaining the user account or the user as displayed in the user administration.
	.PARAMETER Password
		Password with maximum length of 14 characters
	.PARAMETER UserMayNotChangePassword
		Prevents the user from changing the assigned password.
	.PARAMETER PasswordNeverExpires
		This option prevents a password from expiring and overrides the Maximum Password Age setting 
	.PARAMETER Disabled
		Enable this option to only prepare an account and prevent direct use of the newly created account.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		New-LocalUser -Name Test -FullName 'Max Mustermann' -Description 'Test Account' -Password 'Test123' -UserMayNotChangePassword -PasswordNeverExpires -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/New-LocalUser.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][string]$FullName = $null,
		[Parameter(Mandatory=$false)][string]$Description = $null,
		[Parameter(Mandatory=$false)][string]$Password = $null,
		[Parameter(Mandatory=$false)][switch]$UserMayNotChangePassword = $false,
		[Parameter(Mandatory=$false)][switch]$PasswordNeverExpires = $false,
		[Parameter(Mandatory=$false)][switch]$Disabled = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Write-Log -Message "Creating local user '$($Name)'." -Source ${CmdletName};
			
			$parameters = @{
				Name = $Name;
				FullName = $FullName;
				Description = $Description;
				UserMayNotChangePassword = $UserMayNotChangePassword;
				PasswordNeverExpires = $PasswordNeverExpires;
				Disabled = $Disabled;
			};
			
			if ([string]::IsNullOrEmpty($StartPassword)) { $parameters.Password = (New-Object System.Net.NetworkCredential $Name, (ConvertTo-PlainText $Password)).SecurePassword; } else { $parameters.NoPassword = $true }
			
			$user = Microsoft.PowerShell.LocalAccounts\New-LocalUser @parameters;
		}
		catch
		{
			$failed = "Failed to create local user [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Rename-LocalUser {
	<#
	.SYNOPSIS
		Rename local user account.
	.DESCRIPTION
		Use this command to rename an existing local user account.
	.PARAMETER Name
		The name of the user account to be renamed. 
	.PARAMETER NewName
		The new name to which the user should be renamed. A user name must not match any other user or group name on the currently managed computer. It can be up to 20 characters in upper or lower case, except for the following "  /  \  [  ]  :  ;  |  =  ,  +  *  ?  <  >.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Rename-LocalUser -Name 'Administrator|S-1-5-21-3681506545-2675960050-2856003502-500' -NewName 'Mary XMas' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Rename-LocalUser.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$true)][string]$NewName,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Write-Log -Message "Renaming the local user '$($Name)' to '$($NewName)'." -Source ${CmdletName};
			Microsoft.PowerShell.LocalAccounts\Rename-LocalUser -SID (Get-LocalAccountSid $Name) -NewName $NewName;
		}
		catch
		{
			$failed = "Failed to rename local user [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Set-LocalUserFlags {
	<#
	.SYNOPSIS
		Change properties of a local user account.
	.DESCRIPTION
		By running this command, some properties (according to the account policies) of a local user account can be changed.
	.PARAMETER Name
		The name of the user account whose properties are to be changed. 
	.PARAMETER UserMayNotChangePassword
		Prevents the user from changing the assigned password. This option is normally only applied to accounts that are used by multiple users. 
	.PARAMETER PasswordNeverExpires
		This option prevents a password from expiring and overrides the Maximum Password Age setting in the Account Policies dialog box.
	.PARAMETER Disabled
		Enable this option to only prepare an account and prevent direct use of the newly created account.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-LocalUserFlags -Name 'Administrator|S-1-5-21-3681506545-2675960050-2856003502-500' -UserMayNotChangePassword -PasswordNeverExpires -Disabled -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-LocalUserFlags.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][switch]$UserMayNotChangePassword = $false,
		[Parameter(Mandatory=$false)][switch]$PasswordNeverExpires = $false,
		[Parameter(Mandatory=$false)][switch]$Disabled = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Write-Log -Message "Setting flags for the local user '$($Name)': UserMayNotChangePassword $([bool]$UserMayNotChangePassword), PasswordNeverExpires $([bool]$PasswordNeverExpires), Disabled $([bool]$Disabled)." -Source ${CmdletName};
			$user = Microsoft.PowerShell.LocalAccounts\Get-LocalUser -SID (Get-LocalAccountSid $Name);
			Microsoft.PowerShell.LocalAccounts\Set-LocalUser -SID $user.SID -UserMayChangePassword (![bool]$UserMayNotChangePassword) -PasswordNeverExpires ([bool]$PasswordNeverExpires);
			if ($user.Enabled -eq !$Disabled) { Write-Log -Message "The local user '$($Name)' is already $(if ($Disabled) {'disabled'} else {'enabled'})." -Source ${CmdletName}; }
			elseif ($Disabled) { Write-Log -Message "Disabling the local user '$($Name)'." -Source ${CmdletName}; Disable-LocalUser -SID $user.SID; }
			elseif (!$Disabled) { Write-Log -Message "Enabling the local user '$($Name)'." -Source ${CmdletName}; Enable-LocalUser -SID $user.SID; }
		}
		catch
		{
			$failed = "Failed to set flags for local user [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Set-LocalUserPassword {
	<#
	.SYNOPSIS
		Change the password of a local user account.
	.DESCRIPTION
		Use this command to change the password of a local user account.
	.PARAMETER Name
		The name of the user account whose password is to be changed
	.PARAMETER Password
		The password to set. Maximum 14 characters. Note: Please note that the password for the user is only stored obfuscated in the script and can therefore possibly be determined by experienced users.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-LocalUserPassword -Name 'Administrator|S-1-5-21-3681506545-2675960050-2856003502-500' -Password 'Test123' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-LocalUserPassword.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][string]$Password,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Write-Log -Message "Changing the password for the local user '$($Name)'." -Source ${CmdletName};
			Microsoft.PowerShell.LocalAccounts\Set-LocalUser -SID (Get-LocalAccountSid $Name) -Password (New-Object System.Net.NetworkCredential $Name, (ConvertTo-PlainText $Password)).SecurePassword;
		}
		catch
		{
			$failed = "Failed to change the password for local user [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Set-LocalUserProfile {
	<#
	.SYNOPSIS
		Assign a path for a user profile.
	.DESCRIPTION
		This command allows you to assign a path for a user profile, login script or home directory to a local user account.
	.PARAMETER Name
		The name of the user account for which the profile settings are to be changed
	.PARAMETER ProfilePath
		The path via which the user profile is to be accessed and where it is to be stored. 
	.PARAMETER LogonScript
		The name of the login script for this user. A login script can be assigned to a single or multiple user account(s).
	.PARAMETER HomeLocalPath
		To optionally set a home directory on this system, enter the local path to this directory in the Local Path field.
	.PARAMETER HomeDriveLetter
		Enter a drive letter 
	.PARAMETER HomeRemotePath
		UNC path to the shared directory in the "to" field.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-LocalUserProfile -Name 'MyUser|S-1-5-21-3681506545-2675960050-2856003502-500' -ProfilePath '\\ServerName\Profiles' -LogonScript 'logon.cmd' -HomeDriveLetter 'H:' -HomeRemotePath Homes -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-LocalUserProfile.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][string]$ProfilePath = $null,
		[Parameter(Mandatory=$false)][string]$LogonScript = $null,
		[Parameter(Mandatory=$false)][string]$HomeLocalPath = $null,
		[Parameter(Mandatory=$false)][string]$HomeDriveLetter = $null,
		[Parameter(Mandatory=$false)][string]$HomeRemotePath = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$user = Microsoft.PowerShell.LocalAccounts\Get-LocalUser -SID (Get-LocalAccountSid $Name);
			$homePath = $HomeLocalPath;
			$driveLetter = $null;
			if (![string]::IsNullOrEmpty($HomeDriveLetter) -and ![string]::IsNullOrEmpty($HomeRemotePath)) { $homePath = $HomeRemotePath; $driveLetter = $HomeDriveLetter; }
			elseif (![string]::IsNullOrEmpty($HomeDriveLetter)) { Write-Log -Message "Value of -HomeRemotePath is empty, ignoring -HomeDriveLetter ($HomeDriveLetter)." -Source ${CmdletName}; }
			elseif (![string]::IsNullOrEmpty($HomeRemotePath)) { Write-Log -Message "Value of -HomeDriveLetter is empty, ignoring -HomeRemotePath ($HomeRemotePath)." -Source ${CmdletName}; }
			Write-Log -Message "Setting profile informationm for the local user '$($Name)': profile path '$($ProfilePath)', logon script '$($LogonScript)', home path '$($homePath)', home drive letter '$($driveLetter)'." -Source ${CmdletName};
			[PSPD.API]::SetLocalUserProfile($user.Name, $ProfilePath, $LogonScript, $homePath, $HomeDriveLetter);
		}
		catch
		{
			$failed = "Failed to change profile information for local user [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-PendingReboot {
	<#
	.SYNOPSIS
		System restart pending
	.DESCRIPTION
		Checks diverse indications for a pending system restart.
	.EXAMPLE
		Test-PendingReboot
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-PendingReboot.html
	#>

	[CmdletBinding()]
	param (
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$status = Get-PendingReboot;
			Write-Log -Message "Pending reboot status: System: $($status.IsSystemRebootPending) [CBS: $([bool]$status.IsCBServicingRebootPending), Windows-Update: $([bool]$status.IsWindowsUpdateRebootPending), SCCM-Client: $([bool]$status.IsSCCMClientRebootPending), File-Rename: $([bool]$status.IsFileRenameRebootPending)], App-V: $([bool]$status.IsAppVRebootPending)." -Source ${CmdletName};
			$result = ($status.IsSystemRebootPending -or [bool]$status.IsAppVRebootPending);
			Write-Log -Message "Returning result: $($result)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test pending reboot";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Search-RegistryKey {
	<#
	.SYNOPSIS
		Searches a registry key
	.DESCRIPTION
		This command searches below the specified registry key for the key in which the specified value is set.
	.PARAMETER KeyPath
		Enter the registry key from which the specified value is searched.
	.PARAMETER ValueName
		Optionally the default value of the key or the name of another value.
	.PARAMETER Value
		Search for a specific value content. This is useful if there are several values.
	.PARAMETER ParentIndex
		Zero-based index in reverse order. As many key registry levels are truncated on the right as are specified as index numbers.
	.PARAMETER ResultVariable
		The result variable to be set. Specify only the name of the variable.
	.PARAMETER ValueKind
		String or DWORD.
	.PARAMETER FullPathResult
		If checked, the full path is stored in the variable. The result in the example below would be HKEY_CURRENT_USER\Software\Search-RegistryKey\F1\F2\F3.
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Search-RegistryKey -KeyPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Run' -ValueName OneDrive -Value "OneDrive.exe" -ParentIndex index -ResultVariable keyPath -ValueKind String -FullPathResult
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Search-RegistryKey.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,
		[Parameter(Mandatory=$false)][string]$ValueName = $null,
		[Parameter(Mandatory=$false)][string]$Value = $null,
		[Parameter(Mandatory=$false)][int]$ParentIndex = 0,
		[Parameter(Mandatory=$true)][string]$ResultVariable = $null,
		[Parameter(Mandatory=$false)][ValidateSet("String", "DWord")][Microsoft.Win32.RegistryValueKind]$ValueKind = "DWord",
		[Parameter(Mandatory=$false)][switch]$FullPathResult = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $script:PdContext.RegistryCommandsContinueOnError
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }
			
			function searchKey($key, [string]$name, [Microsoft.Win32.RegistryValueKind]$kind, $value)
			{
				$result = $null;

				$matches = @($key.GetValueNames() | where { ($_ -eq $name) -and ($key.GetValueKind($_) -eq $kind) -and ([string]::IsNullOrEmpty($value) -or ($key.GetValue($_) -eq $value)) });
				if ($matches.Count -gt 0)
				{
					Write-Log -Message "Found name '$($name)' of kind '$($kind)' with value '$($key.GetValue($name))' at key '$($key.Name)'." -Source ${CmdletName};
					$result = $key.Name;
				}
				else
				{
					foreach ($subKeyName in $key.GetSubKeyNames())
					{
						try
						{
							$subKey = $key.OpenSubKey($subKeyName, $false);
							$result = searchKey -key $subKey -name $name -kind $kind -value $value;
							$subKey.Close();

							if ($result -ne $null) { break; }
						}
						catch
						{
							Write-Log -Message "Failed to search sub-key '$($subKeyName)' of key '$($key.Name)': $($_)" -Source ${CmdletName} -Severity 2;
							$result = $null;
						}
					}
				}

				return $result;
			}
			
			if (($ValueKind -eq "DWord") -and ![string]::IsNullOrEmpty($Value)) { $Value = [string]([int]$Value); }
			if ($ParentIndex -lt 0) { $ParentIndex = 0; }
			
			$result = $null;
			Write-Log -Message "Searching for value name '$($ValueName)' of kind '$($ValueKind)' with value '$($Value)' in key '$($key.Name)'." -Source ${CmdletName};
			$key = Get-PdRegistryKey -Path $KeyPath -Wow64:$Wow64 -AcceptNull;
			if ($key -ne $null)
			{
				$result = searchKey -key $key -name $ValueName -kind $ValueKind -value $Value;
				$key.Close();
			}
			
			if (![string]::IsNullOrEmpty($result))
			{
				$index = $result.Length;
				for ($pos = $ParentIndex; (($pos -ge 0) -and ($index -gt 0)); $pos--)
				{
					$result = $result.Substring(0, $index);
					$index = $result.LastIndexOf("\");
				}

				if (!$FullPathResult) { $result = $result.Substring($index + 1); }
			}
			else { Write-Log -Message "Not found." -Source ${CmdletName}; }

			if (![string]::IsNullOrEmpty($ResultVariable))
			{
				Write-Log -Message "Writing result '$($result)' (parent-index $($ParentIndex), full path $($FullPathResult)) to variable '$($ResultVariable)'." -Source ${CmdletName};
				Set-PdVar -Name $ResultVariable -Value $result;
			}
		}
		catch
		{
			$failed = "Failed to search for registry value '$ValueName' [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Send-SmtpMail {
	<#
	.SYNOPSIS
		Sends an e-mail via SMTP 
	.DESCRIPTION
		Sends an e-mail via SMTP (Simple Mail Transfer Protocol) provided that TCP/IP is installed as the network protocol. The sender can be freely assigned. Several recipients can be specified, separated by semicolons.
	.PARAMETER Sender
		The Mail Adress and the sender's name that appears on the recipient's screen. Format the string like this: mail|name Example:'Max.Mustermann@muster.de|Max Mustermann'
	.PARAMETER Recipients
		Enter the recipients that you want to address directly here. Multiple recipients are separated by horizontal slash (|).
	.PARAMETER Subject
		The subject of the mail.
	.PARAMETER Server
		Enter the e-mail server using the IP address or the computer name and other parameters for the connection to the mail server in following format: hostname|port|ssl|useAuthentication|username|password (Password can be clear text or base64)
	.PARAMETER Attachments
		Files to be attached to the e-mail. Multiple files can be attached.
	.PARAMETER Message
		The actual message text.
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Send-SmtpMail -Sender 'Max.Mustermann@muster.de|Max Mustermann' -Recipients 'max.mustermann@gmx.de|maria.musterfrau@gmx.de' -Subject 'Test E-Mail' -Server 'localhost|25|1|1|Test|Test' -Attachments '.\Files\MyAttachment.txt' -Message @('This is a mail test')
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Send-SmtpMail.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Sender,
		[Parameter(Mandatory=$true)][string]$Recipients,
		[Parameter(Mandatory=$true)][string]$Subject,
		[Parameter(Mandatory=$true)][string]$Server,
		[Parameter(Mandatory=$false)][string]$Attachments = $null,
		[Parameter(Mandatory=$false)][string[]]$Message = @(),
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }
			
			$PartsSeparator = "|";
			$ListSeparator = ";";
			$ServerInfoFlag_True = "1";
			
			$senderAddress, $senderName = $Sender.Split($PartsSeparator, 2);
			$senderAddress = $senderAddress.Trim();
			$senderName = $senderName.Trim();
			$from = $(if ([string]::IsNullOrEmpty($senderName)) { $senderAddress } else { "$($senderName) <$($senderAddress)>" });

			$to, $cc, $bcc = $Recipients.Split($PartsSeparator, 3);
			[string[]]$to = @(([string]$to).Split($ListSeparator, [System.StringSplitOptions]::RemoveEmptyEntries) | % { $_.Trim() } | where { ![string]::IsNullOrEmpty($_) });
			[string[]]$cc = @(([string]$cc).Split($ListSeparator, [System.StringSplitOptions]::RemoveEmptyEntries) | % { $_.Trim() } | where { ![string]::IsNullOrEmpty($_) });
			[string[]]$bcc = @(([string]$bcc).Split($ListSeparator, [System.StringSplitOptions]::RemoveEmptyEntries) | % { $_.Trim() } | where { ![string]::IsNullOrEmpty($_) });
			
			$smtpServer, $port, $useSsl, $authenticationRequired, $userName, $password = $Server.Split($PartsSeparator, 6);
			$smtpServer = $smtpServer.Trim();
			$port = $port.Trim();
			[bool]$useSsl = ($useSsl -eq $ServerInfoFlag_True);
			[bool]$authenticationRequired = ($authenticationRequired -eq $ServerInfoFlag_True);
			$credential = $(if ($authenticationRequired) { $credential = New-Object PSCredential $userName, (ConvertTo-SecureString -String (ConvertTo-PlainText $password) -AsPlainText -Force) } else { $null });
			
			$files = @($Attachments.Split($ListSeparator, [System.StringSplitOptions]::RemoveEmptyEntries) | % { $_.Trim() } | where { ![string]::IsNullOrEmpty($_) } | % { Expand-Path  $_ -Wow64:$Wow64 });
			
			$body = [string]::Join("`r`n", $Message);
			
			$parameters = @{
				To = $to;
				From = $from;
				Subject = $Subject;
				Body = $body;
				SmtpServer = $smtpServer;
				Port = $port;
				UseSsl = $useSsl;
				Encoding = "UTF8";
			};
			
			if ($cc.Count -gt 0) { $parameters.Cc = $cc; }
			if ($bcc.Count -gt 0) { $parameters.Bcc = $bcc; }
			if ($files.Count -gt 0) { $parameters.Attachments = $files; }
			if ($credential -ne $null) { $parameters.Credential = $credential; }

			Write-Log -Message "Sending e-mail '$($subject)' from '$($from)' to '$([string]$to)' [server: $($smtpServer), port: $($port), SSL: $($useSsl), authenticated: $($authenticationRequired)]." -Source ${CmdletName};
			Send-MailMessage @parameters;
		}
		catch
		{
			$failed = "Failed to send e-mail '$Subject' [$Sender]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-Line {
	<#
	.SYNOPSIS
		Existence of a line
	.DESCRIPTION
		Checks whether a specific line exists in the specified file. Wildcards can be used also.
	.PARAMETER Line
		Content of the line. Wildcards (? *) can be used.
	.PARAMETER Path
		File path.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-Line -Line "Version" -Path "${env:SystemDrive}\temp\myfile.txt"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-Line.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$true)][string]$Line,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			# test always ... if (Test-SkipCommand -Context $Context) { return; }

			$Path = Expand-Path $Path -Wow64:$Wow64;
			if (![System.IO.File]::Exists($Path))
			{
				Write-Log -Message "File '$($Path)' not found." -Source ${CmdletName};
				return $false;
			}
			
			$result = $false;
			
			$pattern = ("^" + [regex]::Escape($Line).Replace("\*", ".*").Replace("\?", ".") + "`$");
			$lines = Get-Content -Path $Path # -Encoding Default
			foreach ($item in $lines)
			{
				$result = [regex]::IsMatch($item, $pattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase);
				if ($result) { break; }
			}

			Write-Log -Message "Searching line '$($Line)' [$($pattern)] in file '$($Path)' - result: $($result)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test lines in [$($Path)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Set-NtfsSecurity {
	<#
	.SYNOPSIS
		Changes directory and file level access permissions on NTFS partitions. 
	.DESCRIPTION
		Changes directory and file level access permissions on NTFS partitions. Use this command to change the access permissions on local drives and shared network directories.
	.PARAMETER Path
		The directory in which access permissions are to be changed. Use the "..." button to select an existing directory (on the packaging system) or enter the path to be created manually.
	.PARAMETER Files
		Only active if you have activated the option "SetFiles". You can separate multiple specifications by spaces or by semicolons, periods, or quotation marks. 
		You should note the following: If the file name already contains spaces or periods, the individual file names should be placed in inverted commas to ensure unique identification. 
		If you want to set different rights for files and directories, you must assign the rights in separate commands.
	.PARAMETER SetDirectory
		If the option is activated, access permissions are only changed at directory level. These changes have no effect on the access permissions of existing files.
	.PARAMETER SetSubDirectories
		If the option is activated, the access permissions are also changed in all subdirectories.
	.PARAMETER SetFiles
		If the option is enabled, the access permissions are changed at the file level. In the text field "Files" you can specify the desired file specification.
	.PARAMETER KeepInherited
		If this option is activated, the permission for all newly created files ("file inherit ACE") within the selected directory will be kept on the existing settings, 
		independent of the new directory permission ("container ACE").
	.PARAMETER Mode
		Determines whether the entries from the user and group list should be added, deleted or replaced in the access permission.
		- Reset: Adds the entries from the user and group list to the existing access permissions. 
			 If an inherited access authorization already exists for a user or group, it is converted into a local access authorization and changed accordingly. 
			 All other inherited access authorizations are then also converted into local access authorizations..
		- Add: Adds or changes the entries from the user and group list as local access permissions. If an inherited access authorization already exists for a user or group, it remains unchanged.
		- Remove: For the entries from the user and group list, the respective rights are deleted from the existing access authorizations. If an inherited access authorization already exists for a user or group, it is converted into a local access authorization and changed accordingly. All other inherited access authorizations are then also converted into local access authorizations..
		- Set: The entries from the user and group list completely replace the existing access permissions.
	.PARAMETER AccountRights
		Sets the type of access to be defined
			"R" = Read
			"W" = Write
			"X" = ExecuteFile
			"D" = Delete
			"P" = ChangePermissions
			"O" = TakeOwnership
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-NtfsSecurity -Path "${env:SystemDrive}\temp" -Files '*.*' -Mode Add -SetDirectory -SetSubDirectories -SetFiles -AccountRights 'Administrators|S-1-5-32-544 RX' -Context Computer
	.EXAMPLE
		Set-NtfsSecurity -Path "${env:SystemDrive}\temp" -Files '' -Mode Reset -SetDirectory -SetSubDirectories -AccountRights 'Administrators|S-1-5-32-544 RWXDPO' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-NtfsSecurity.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][string]$Files = $null,
		[Parameter(Mandatory=$false)][switch]$SetDirectory = $false,
		[Parameter(Mandatory=$false)][switch]$SetSubDirectories = $false,
		[Parameter(Mandatory=$false)][switch]$SetFiles = $false,
		[Parameter(Mandatory=$false)][switch]$KeepInherited = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Reset", "Add", "Remove", "Set")][string]$Mode = "Reset",
		[Parameter(Mandatory=$false)][string[]]$AccountRights = @(),
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$Path = Expand-Path $Path -Wow64:$Wow64;
			Write-Log -Message "Setting NTFS security for directory '$($Path)'." -Source ${CmdletName};
			if (![System.IO.Directory]::Exists($Path)) { throw "Directory '$($Path)' does not exist."; }

			$rightsLookup = @{
				"R" = [System.Security.AccessControl.FileSystemRights]::Read;
				"W" = [System.Security.AccessControl.FileSystemRights]::Write;
				"X" = [System.Security.AccessControl.FileSystemRights]::ExecuteFile;
				"D" = [System.Security.AccessControl.FileSystemRights]::Delete;
				"P" = [System.Security.AccessControl.FileSystemRights]::ChangePermissions;
				"O" = [System.Security.AccessControl.FileSystemRights]::TakeOwnership;
			};

			$AsFullControlFileSystemRights = (
				[System.Security.AccessControl.FileSystemRights]::Read -bor 
				[System.Security.AccessControl.FileSystemRights]::Write -bor 
				[System.Security.AccessControl.FileSystemRights]::ExecuteFile -bor 
				[System.Security.AccessControl.FileSystemRights]::Delete -bor 
				[System.Security.AccessControl.FileSystemRights]::ChangePermissions -bor 
				[System.Security.AccessControl.FileSystemRights]::TakeOwnership);

			$NoFileSystemRights = [System.Security.AccessControl.FileSystemRights]0;

			function getFileSystemRights([string]$value)
			{
				if ([string]::IsNullOrEmpty($value)) { return $NoFileSystemRights; }
				
				$result = $NoFileSystemRights;
				
				foreach ($c in $value.ToCharArray())
				{
					$key = $c.ToString();
					if ($rightsLookup.ContainsKey($key))
					{
						$result = ($result -bor $rightsLookup[$key]);
					}
				}
				
				return $result;
			}
			
			$accounts = @();
			foreach ($item in $AccountRights)
			{
				if ([string]::IsNullOrEmpty($item)) { continue; }

				$index = $item.LastIndexOfAny(" `t");
				$sid = Get-LocalAccountSid $(if ($index -ge 0) { $item.Substring(0, $index).Trim() } else { $item });
				$rights = [System.Security.AccessControl.FileSystemRights](getFileSystemRights $(if ($index -ge 0) { $item.Substring($index).Trim() } else { "" }));
				$isFullControl = (($rights -band $AsFullControlFileSystemRights) -eq $AsFullControlFileSystemRights);
				$account = New-Object PSObject -Property @{ Raw = $item; SID = $sid; Rights = $rights; IsFullControl = $isFullControl; };
				Write-Log -Message "Setting account rights '$($account.Raw)' (SID: $($account.SID), Rights ($([int]$account.Rights)): $($account.Rights) - Full Control: $($account.IsFullControl))." -Source ${CmdletName};
				$accounts += $account;
			}
			
			$fileSpecifications = @();
			if ($SetFiles)
			{
				$pattern = "(^|;+| +)(`"(?<spec>[^`"]+)`"|'(?<spec>[^']+)'|(?<spec>[^ ;]+))"; # '
				$fileSpecifications = @([regex]::Matches($Files, $pattern) | % { $_.Groups["spec"].Value; });
				Write-Log -Message "File specifications: '$([string]::Join(''', ''', $fileSpecifications))'." -Source ${CmdletName};
			}
			
			$accountLookup = @{}; $accounts | % { $accountLookup[$_.SID.Value] = $_ }
			$checkAccessRuleProtection = {param($security) return (@($security.GetAccessRules($false, $true, [System.Security.Principal.SecurityIdentifier]) | where { $accountLookup.ContainsKey($_.IdentityReference.Value) }).Count -gt 0) };
			
			$accessRuleModification = "Set";
			$setAccessRuleProtection = { $false; };
			$preserveInheritance = $true;
			$removeAllAccess = $false;
			
			if ($Mode -eq "Reset") # Add, replace inherited access permissions
			{
				$accessRuleModification = "Set";
				$setAccessRuleProtection = $checkAccessRuleProtection;
				$preserveInheritance = $true;
				$removeAllAccess = $false;
			}
			elseif ($Mode -eq "Add") # Add, keep inherited access permissions
			{
				$accessRuleModification = "Set";
				$setAccessRuleProtection = { $false; };
				$preserveInheritance = $true;
				$removeAllAccess = $false;
			}
			elseif ($Mode -eq "Remove") # Remove
			{
				$accessRuleModification = "Remove";
				$setAccessRuleProtection = $checkAccessRuleProtection;
				$preserveInheritance = $true;
				$removeAllAccess = $false;
			}
			elseif ($Mode -eq "Set") # Replace all
			{
				$accessRuleModification = "Set";
				$setAccessRuleProtection = { $true; };
				$preserveInheritance = $false;
				$removeAllAccess = $true;
			}
			
			$modified = $false;

			if ($SetDirectory)
			{
				$accessControlType = [System.Security.AccessControl.AccessControlType]::Allow;

				$inheritanceFlags = [System.Security.AccessControl.InheritanceFlags]::ContainerInherit;
				if (!$KeepInherited) { $inheritanceFlags = ($inheritanceFlags -bor [System.Security.AccessControl.InheritanceFlags]::ObjectInherit); }

				$propagationFlags = [System.Security.AccessControl.PropagationFlags]::None;

				$directorySecurity = [System.IO.Directory]::GetAccessControl($Path);

				if ($removeAllAccess)
				{
					Write-Log -Message "Removing all access rules from '$($Path)' that are not inherited." -Source ${CmdletName};
					$directorySecurity.Access | where { !$_.IsInherited } | % { $directorySecurity.RemoveAccessRule($_); }
					[System.IO.Directory]::SetAccessControl($Path, $directorySecurity);
					$directorySecurity = [System.IO.Directory]::GetAccessControl($Path);
				}

				if (& $setAccessRuleProtection $directorySecurity)
				{
					Write-Log -Message "Disable access rule inheritance for '$($Path)', preserve inherited access rules: $($preserveInheritance)." -Source ${CmdletName};
					$directorySecurity.SetAccessRuleProtection($true, $preserveInheritance);
					[System.IO.Directory]::SetAccessControl($Path, $directorySecurity);
					$directorySecurity = [System.IO.Directory]::GetAccessControl($Path);
				}

				$rules = @($accounts | % { New-Object System.Security.AccessControl.FileSystemAccessRule $_.SID, $(if ($_.IsFullControl) { [System.Security.AccessControl.FileSystemRights]::FullControl } else { $_.Rights }), $inheritanceFlags, $propagationFlags, $accessControlType });
				Write-Log -Message "Apply access rules to '$($Path)' (access control: '$($accessControlType)', inheritance: '$($inheritanceFlags.ToString())', propagation: '$($propagationFlags.ToString())'): $($rules.Count)." -Source ${CmdletName};
				$rules | % { $void = $directorySecurity.ModifyAccessRule($accessRuleModification, $_, [ref]$modified); }

				[System.IO.Directory]::SetAccessControl($Path, $directorySecurity);
			}

			if ($SetFiles -and (!$SetDirectory -or $KeepInherited))
			{
				$accessControlType = [System.Security.AccessControl.AccessControlType]::Allow;
				$inheritanceFlags = [System.Security.AccessControl.InheritanceFlags]::None;
				$propagationFlags = [System.Security.AccessControl.PropagationFlags]::None;
				$rules = @($accounts | % { New-Object System.Security.AccessControl.FileSystemAccessRule $_.SID, $(if ($_.IsFullControl) { [System.Security.AccessControl.FileSystemRights]::FullControl } else { $_.Rights }), $inheritanceFlags, $propagationFlags, $accessControlType });
				
				$searchOption = $(if ($SetSubDirectories) { [System.IO.SearchOption]::AllDirectories } else { [System.IO.SearchOption]::TopDirectoryOnly });
				$filePaths = @($fileSpecifications | % { [System.IO.Directory]::GetFiles($Path, $_, $searchOption) } | sort -Unique);
				foreach ($filePath in $filePaths)
				{
					$fileSecurity = [System.IO.File]::GetAccessControl($filePath);

					if ($removeAllAccess)
					{
						Write-Log -Message "Removing all access rules from '$($filePath)' that are not inherited." -Source ${CmdletName};
						$fileSecurity.Access | where { !$_.IsInherited } | % { $fileSecurity.RemoveAccessRule($_); }
						[System.IO.File]::SetAccessControl($filePath, $fileSecurity);
						$fileSecurity = [System.IO.File]::GetAccessControl($filePath);
					}

					if (& $setAccessRuleProtection $fileSecurity)
					{
						Write-Log -Message "Disable access rule inheritance for '$($filePath)', preserve inherited access rules: $($preserveInheritance)." -Source ${CmdletName};
						$fileSecurity.SetAccessRuleProtection($true, $preserveInheritance);
						[System.IO.File]::SetAccessControl($filePath, $fileSecurity);
						$fileSecurity = [System.IO.File]::GetAccessControl($filePath);
					}

					Write-Log -Message "Apply access rules to '$($filePath)' (access control: '$($accessControlType)'): $($rules.Count)." -Source ${CmdletName};
					$rules | % { $void = $fileSecurity.ModifyAccessRule($accessRuleModification, $_, [ref]$modified); }

					[System.IO.File]::SetAccessControl($filePath, $fileSecurity);
				}
			}
		}
		catch
		{
			$failed = "Failed to set NTFS security for [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}
function Set-RegistrySecurity {
	<#1
	.SYNOPSIS
		Change Permissions in the Registry. 
	.DESCRIPTION
		Use this command to change the permissions in the registry.
	.PARAMETER KeyPath
		Specify the registry key you want to change the permissions for.
	.PARAMETER SetSubKeys
		Changes the permissions for all subkeys.
	.PARAMETER Mode
		Determines whether the entries from the user and group list should be added, deleted or replaced in the access permission.
		- Add: Adds the entries contained in the user and group list to the list of existing access permissions.(Existing access permissions are automatically extended.)
		- Remove: Deletes the entries contained in the user and group list from the list of existing access permissions.
		- Set: Replaces the entries contained in the user and group list with the entries on the list of existing access permissions.
	.PARAMETER AccountRights
		Sets the type of access to be defined and enter the users or groups in this list you want to add to, remove from or replace in the access permissions.
			"D" = Delete
			"O" = TakeOwnership
			"P" = ChangePermissions
			"R" = ReadKey
			"W" = WriteKey
			"X" = ExecuteKey
			"Q" = QueryValues
			"S" = SetValue
			"C" = CreateSubKey
			"E" = EnumerateSubKeys
			"N" = Notify
			"L" = CreateLink
			"A" = ReadPermissions
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 registry.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-RegistrySecurity -KeyPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Mozilla\Firefox' -Mode Add -SetSubKeys -AccountRights 'Gast RWXDPO'
	.EXAMPLE
		Set-RegistrySecurity -KeyPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Mozilla\Firefox' -Mode Reset -SetSubKeys -AccountRights 'Gast RWXDPOQSCENLA' -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-RegistrySecurity.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KeyPath,  
		[Parameter(Mandatory=$false)][switch]$SetSubKeys = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Add", "Delete", "Replace")][string]$Mode = "Add",
		[Parameter(Mandatory=$false)][string[]]$AccountRights = @(),
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		$key = $null;
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$key = Get-PdRegistryKey -Path $KeyPath -Wow64:$Wow64 -AcceptNull -Writable;
			if ($key -eq $null) { throw "Registrykey '$($KeyPath)' does not exist."; }

			Write-Log -Message "Setting Registry security for key '$($key.Name)'." -Source ${CmdletName};
			

			$rightsLookup = @{
				"D" = [System.Security.AccessControl.RegistryRights]::Delete;
				"O" = [System.Security.AccessControl.RegistryRights]::TakeOwnership;
				"P" = [System.Security.AccessControl.RegistryRights]::ChangePermissions;
				"R" = [System.Security.AccessControl.RegistryRights]::ReadKey;
				"W" = [System.Security.AccessControl.RegistryRights]::WriteKey;
				"X" = [System.Security.AccessControl.RegistryRights]::ExecuteKey;
				"Q" = [System.Security.AccessControl.RegistryRights]::QueryValues;
				"S" = [System.Security.AccessControl.RegistryRights]::SetValue;
				"C" = [System.Security.AccessControl.RegistryRights]::CreateSubKey;
				"E" = [System.Security.AccessControl.RegistryRights]::EnumerateSubKeys;
				"N" = [System.Security.AccessControl.RegistryRights]::Notify;
				"L" = [System.Security.AccessControl.RegistryRights]::CreateLink;
				"A" = [System.Security.AccessControl.RegistryRights]::ReadPermissions;
			};

			$AsFullControlRegistryRights = [System.Security.AccessControl.RegistryRights]::FullControl;

			$NoRegistryRights = [System.Security.AccessControl.RegistryRights]0;

			function getRegRights([string]$value)
			{
				if ([string]::IsNullOrEmpty($value)) { return $NoFileSystemRights; }
				
				$result = $NoRegistryRights;
				
				foreach ($c in $value.ToCharArray())
				{
					$char = $c.ToString();
					if ($rightsLookup.ContainsKey($char))
					{
						$result = ($result -bor $rightsLookup[$char]);
					}
				}
				
				return $result;
			}
			
			$accounts = @();
			foreach ($item in $AccountRights)
			{
				if ([string]::IsNullOrEmpty($item)) { continue; }

				$index = $item.LastIndexOfAny(" `t");
				$sid = Get-LocalAccountSid $(if ($index -ge 0) { $item.Substring(0, $index).Trim() } else { $item });
				$rights = [System.Security.AccessControl.RegistryRights](getRegRights $(if ($index -ge 0) { $item.Substring($index).Trim() } else { "" }));
				$isFullControl = (($rights -band $AsFullControlRegistryRights) -eq $AsFullControlRegistryRights);
				$account = New-Object PSObject -Property @{ Raw = $item; SID = $sid; Rights = $rights; IsFullControl = $isFullControl; };
				Write-Log -Message "Setting account rights '$($account.Raw)' (SID: $($account.SID), Rights ($([int]$account.Rights)): $($account.Rights) - Full Control: $($account.IsFullControl))." -Source ${CmdletName};
				$accounts += $account;
			}
			
			$accountLookup = @{}; $accounts | % { $accountLookup[$_.SID.Value] = $_ }
			$checkAccessRuleProtection = {param($security) return (@($security.GetAccessRules($false, $true, [System.Security.Principal.SecurityIdentifier]) | where { $accountLookup.ContainsKey($_.IdentityReference.Value) }).Count -gt 0) };
			
			$accessRuleModification = "Set";
			$setAccessRuleProtection = { $false; };
			$preserveInheritance = $true;
			
			if ($Mode -eq "Add") 
			{
				$accessRuleModification = "Set";
				$setAccessRuleProtection = { $false; };
				$preserveInheritance = $true;
			}
			elseif ($Mode -eq "Delete") 
			{
				$accessRuleModification = "Remove";
				$setAccessRuleProtection = $checkAccessRuleProtection;
				$preserveInheritance = $true;
			}
			elseif ($Mode -eq "Replace") 
			{
				$accessRuleModification = "Set";
				$setAccessRuleProtection = { $true; };
				$preserveInheritance = $false;
			}
			
			$modified = $false;

			$accessControlType = [System.Security.AccessControl.AccessControlType]::Allow;

			$inheritanceFlags = [System.Security.AccessControl.InheritanceFlags]::None;

			if($SetSubKeys) { $inheritanceFlags = ($inheritanceFlags -bor [System.Security.AccessControl.InheritanceFlags]::ContainerInherit); }

			$propagationFlags = [System.Security.AccessControl.PropagationFlags]::None;

			$accessControl = $key.GetAccessControl();
			if (& $setAccessRuleProtection $accessControl)
			{
				Write-Log -Message "Disable access rule inheritance for '$($key.Name)', preserve inherited access rules: $($preserveInheritance)." -Source ${CmdletName};
				$accessControl.SetAccessRuleProtection($true, $preserveInheritance);
				$key.SetAccessControl($directorySecurity);
				$accessControl = $key.GetAccessControl();
			}

			$rules = @($accounts | % { New-Object System.Security.AccessControl.RegistryAccessRule $_.SID, $(if ($_.IsFullControl) { [System.Security.AccessControl.RegistryRights]::FullControl } else { $_.Rights }), $inheritanceFlags, $propagationFlags, $accessControlType });
			Write-Log -Message "Apply access rules to '$($key.Name)' (access control: '$($accessControlType)', inheritance: '$($inheritanceFlags.ToString())', propagation: '$($propagationFlags.ToString())'): $($rules.Count)." -Source ${CmdletName};
			$rules | % { $void = $accessControl.ModifyAccessRule($accessRuleModification, $_, [ref]$modified); }

			$key.SetAccessControl($accessControl);

		}
		catch
		{
			$failed = "Failed to set Registry security for [$KeyPath]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
		finally
		{
			if ($key -ne $null)
			{
				$key.Close();
				$key = $null;
			}
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Select-XmlNode([string]$path, [string]$query)
{
	$xml = New-Object System.Xml.XmlDocument;
	$xml.PreserveWhitespace = $true;
	$xml.Load($path);

	$namespaceManager = $null;

	$matches = [regex]::Matches($query, ',\s*xmlns(:(?<name>\w+))?\s*=\s*"(?<url>[^"]+)"');
	if ($matches.Count -gt 0)
	{
		$namespaceManager = New-Object System.Xml.XmlNamespaceManager $xml.NameTable
		$matches | % { $namespaceManager.AddNamespace($_.Groups["name"].Value, $_.Groups["url"].Value); }
		$query = $query.Substring(0, $matches[0].Index);
		Write-Log -Message "Using XmlNamespaceManager namespaces $($matches.Count): $([string]::Join(', ', @($namespaceManager.GetEnumerator() | % { $namespaceManager.LookupNamespace($_) })))." -Source ${CmdletName};
	}

	$nodes = $xml.SelectNodes($query, $namespaceManager);
	Write-Log -Message "Performed query '$($query)' on XML file '$($path)' - result nodes: $($nodes.Count)." -Source ${CmdletName};
	
	return $nodes;
}

function Remove-Hint([string]$path)
{
	# format: <path>'<#'<hint>'#>'
	$match = [regex]::Match($path, "^(?<path>.*)\s*<#\s*(?<hint>.*)\s*#>\s*`$");
	if ($match.Success) { $path = $match.Groups["path"].Value.Trim(); }
	
	return $path;
}

function Set-XmlNode {
	<#
	.SYNOPSIS
		Make Changes to to an XML file add, modify or delete existing entries
	.DESCRIPTION
		This command adds new entries to an XML file or changes or deletes existing entries. The modified file is only saved if it is a "well-formed" XML file, but the schema is not validated.
	.PARAMETER NodeType
		Entry type
		Selection of the type of the part of the XML file to be edited:
		- Element - Basic structural unit of an XML file, which usually has a start and end tag.
		If an element is to be deleted, the element identified under XPath query is completely deleted with all subelements and contents. If a new element is to be added, the content of the Entry value field is added to the XML file under the specified element.
		If no XPath query is specified, the content of the Entry value field is used to replace the full content of the XML file. This is how you can create a new XML file.
		- Attribute of an element
		If an attribute is to be deleted, the attribute is completely deleted from the element identified under XPath query.
		If an attribute is to be added or changed, the attribute is entered in the identified element. The command automatically detects if the attribute already exists and reacts accordingly. 
		- Content - this refers to the text content of an element between the start and end tag (without sub-elements)
		If the content of an element is to be deleted, it is completely deleted from the element identified under XPath query.
		If the content of an element is to be added or changed, the content is entered into the identified element or replaces the existing content. Subelements are not treated as content and are therefore ignored.
	.PARAMETER Action
		The following actions are available:
		- Add
		- Change 
		- Delete
	.PARAMETER Path
		XML file to be modified. The use of variables is possible. Use the "..." button to select an existing file (on the packaging system) or enter the path manually.
	.PARAMETER XPathQuery
		dentification of the part of the XML file to be edited using XPath syntax. With a click on the "..." button you open the XPath browser for the specified XML file, 
		which enables a convenient selection of the searched element including its conversion into a unique XPath query. 
		Namespaces are also taken into account. The file is only displayed if it is a "well-formed" XML file.
	.PARAMETER Index
		Defines which elements should be processed if the query returns multiple hits in the XML file. The following elements are available for selection:
			- All machtes (Default)
			- First match
			- Last match
			- Specific index
	.PARAMETER Attribute
		Contains the name of the attribute searched for if Entry type = attribute was selected. The placeholder * is allowed.
	.PARAMETER HintPath
		Deprecated: Unused Parameter
	.PARAMETER Content
		Defines the XML content to be replaced or added to the XML file for the Add and Change actions. To enter complex and multi-line contents, open the XML file by clicking on the "..." button to open an editor. The use of variables is possible. 
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-XmlNode -NodeType Attribute -Action Add -Path '.\Package.xml' -XPathQuery '/Package/DisplayName' -Index 1 -Attribute value -Content 1000 -Context Computer
	.EXAMPLE
		Set-XmlNode -NodeType Element -Action Delete -Path '.\Package.xml' -XPathQuery '/Package/Parameters' -Index 1 -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-XmlNode.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][ValidateSet("Element", "Attribute", "Content")][string]$NodeType = "Element",
		[Parameter(Mandatory=$true)][ValidateSet("Add", "Change", "Delete")][string]$Action = "Add",
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][string]$XPathQuery = $null,
		[Parameter(Mandatory=$false)][int]$Index = 0,
		[Parameter(Mandatory=$false)][string]$Attribute = $null,
		[Parameter(Mandatory=$false)][string]$HintPath = $null,
		[Parameter(Mandatory=$false)][string[]]$Content = @(),
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if (($NodeType -eq "Attribute") -and [string]::IsNullOrEmpty($Attribute)) { throw "Parameter -Attribute is required for -NodeType '$($NodeType)'."; }
			if (($NodeType -eq "Element") -and ($Action -eq "Change")) { throw "Invalid -Action '$($Action)' for -NodeType '$($NodeType)'."; }
			if ([string]::IsNullOrEmpty($XPathQuery) -and (($NodeType -ne "Element") -or ($Action -ne "Add"))) { throw "Parameter -XPathQuery is required for -Action '$($Action)' on -NodeType '$($NodeType)'."; }
			
			$Path = Remove-Hint $Path;
			$Path = Expand-Path $Path -Wow64:$Wow64;
			if (([System.IO.Path]::GetExtension($Path) -eq "") -and ![System.IO.File]::Exists($Path)) { $Path += ".xml"; Write-Log -Message "Appending '.xml' to -Path -> '$($Path)'." -Source ${CmdletName}; }
			
			$value = [string]::Join("`r`n", $Content);
			
			if ([string]::IsNullOrEmpty($XPathQuery) -and ($NodeType -eq "Element") -and ($Action -eq "Add"))
			{
				Write-Log -Message "No -XPathQuery specified for -Action '$($Action)' on -NodeType '$($NodeType)' -> creating XML-file '$($Path)' from specified -Content value." -Source ${CmdletName};
				$xml = New-Object System.Xml.XmlDocument;
				$xml.PreserveWhitespace = $false;
				$xml.InnerXml = $value;
				# if ($xml.FirstChild -isnot [System.Xml.XmlDeclaration]) { $void = $xml.InsertBefore($xml.CreateXmlDeclaration('1.0', 'utf-8', $null), $xml.FirstChild); }
				$xml.Save($Path);
				return;
			}

			$nodes = @(Select-XmlNode -path $Path -query $XPathQuery);
			
			if ($nodes.Count -eq 0) { Write-Log -Message "No nodes found - done." -Source ${CmdletName}; return; }
			elseif ($Index -gt $nodes.Count) { Write-Log -Message "[-Index = $($Index)] Count of result nodes less than $($Index) - done." -Source ${CmdletName}; return; }
			elseif ($Index -eq 0) { Write-Log -Message "[-Index = $($Index)] Selecting all nodes ($($nodes.Count))." -Source ${CmdletName}; }
			elseif ($Index -eq 1) { Write-Log -Message "[-Index = $($Index)] Selecting first node." -Source ${CmdletName}; $nodes = @( $nodes[0] ); }
			elseif ($Index -lt 0) { Write-Log -Message "[-Index = $($Index)] Selecting last node (#$($nodes.Count))." -Source ${CmdletName}; $nodes = @( $nodes[$nodes.Count - 1] ); }
			else { Write-Log -Message "Selecting node #$($Index) of $($nodes.Count)." -Source ${CmdletName}; $nodes = @( $nodes[$Index - 1] ); }

			Write-Log -Message "Performing action '$($Action)' on node type '$($NodeType)' of selected nodes ($($nodes.Count))." -Source ${CmdletName};
			if (($Action -eq "Delete") -and ($NodeType -eq "Element")) { $nodes | % { $void = $_.ParentNode.RemoveChild($_); } }
			elseif (($Action -eq "Delete") -and ($NodeType -eq "Attribute")) { $nodes | % { $_.RemoveAttribute($Attribute); } }
			elseif (($Action -eq "Delete") -and ($NodeType -eq "Content")) { $nodes | % { $_.IsEmpty = $true; } } # note: $_.InnerText = "" -> <node></node> ; $_.IsEmpty = $true --> <node />
			elseif ($NodeType -eq "Attribute") { $nodes | % { $_.SetAttribute($Attribute, $value); } } # Add or Change
			elseif ($NodeType -eq "Content") { $nodes | % { $_.InnerText = $(if ($Action -eq "Add") { [string]::Concat($_.InnerText, $value) } else { $value }); } }
			elseif (($Action -eq "Add") -and ($NodeType -eq "Element"))
			{
				$fragment = New-Object System.Xml.XmlDocument;
				$root = $fragment.AppendChild($fragment.CreateElement("root"));
				$root.InnerXml = $value;
				foreach ($node in $nodes) { $root.ChildNodes | % { $void = $node.AppendChild($node.OwnerDocument.ImportNode($_, $true)); } }
			}
			else { throw "Unsupported -Action '$($Action)' for -NodeType '$($NodeType)'."; }
			
			$xml = $nodes[0].OwnerDocument;
			$xml.Save($Path);
		}
		catch
		{
			$failed = "Failed to modify XML file [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-XmlNode {
	<#
	.SYNOPSIS
		Reads existing XML files.
	.DESCRIPTION
		This command reads information from existing XML files. The dialog box specifies which information (element, attribute, content) should be read from which XML file.
		To identify the part of the XML file to be processed, DSM uses specifications of the XML Path Language (XPath) 2.0. See: https://www.w3.org/TR/xpath20/
	.PARAMETER NodeType
		Selection of the type of the part of the XML file to be edited:
		- Element - Basic structural unit of an XML file, which usually has a start and end tag.
		If an element is searched for, the result variable contains the entire element with start and end tags and the complete content between these tags after the command is executed. 
		The search for an element returns the complete element: e.g. <Object type="STRING">QueenMary</Object>.
		- Attribute
		If an attribute is searched for, the result variable contains the value of the attribute after the command is executed. The search for the attribute type in the above-mentioned example element results STRING. 
		- Content - this refers to the text content of an element between the start and end tag (without sub-elements)
		If the content is searched for, the result variable contains the entire text content of the searched element between the start and end tag after the command has been executed. Subordinate elements with their associated text contents are not included. The search for the content in the above example element results in QueenMary.
	.PARAMETER Path
		XML file to be modified. The use of variables is possible. Use the "..." button to select an existing file (on the packaging system) or enter the path manually
	.PARAMETER XPathQuery
		Identification of the part of the XML file to be edited using XPath syntax. With a click on the "..." button you open the XPath browser for the specified XML file, 
		which enables a convenient selection of the searched element including its conversion into a unique XPath query. 
		Namespaces are also taken into account. The file is only displayed if it is a "well-formed" XML file.
	.PARAMETER Attribute
		Enthält den Namen des gesuchten Attributs, wenn für Typ des Eintrags der Wert "Attribut" ausgewählt wurde. Der Platzhalter * ist erlaubt. 
	.PARAMETER Index
		Defines which elements should be read if the query returns multiple hits in the XML file. The following elements are available for selection:
		First match
		Last match
		Specific index
		To do this, you must specify the position of the desired element in the hit list, e.g. 3 for the third occurrence in the XML file.
	.PARAMETER ResultVariable
		Specifies the variable in which the result of the command is to be stored. Specify only the name of the variable, not the $ sign and curly brackets.
	.PARAMETER IndexVariable
		Defines the variable in which the index of the used hit should be stored. Specify only the name of the variable, not the $ sign and curly brackets. 
		This variable can be used to determine the maximum value for a query loop. 
		In this case you should enter the command Read-XmlNode several times in the script: first only to determine the maximum index and then in a corresponding loop to read out the individual values found.
	.PARAMETER HintPath
		Deprecated: Unused Parameter
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-XmlNode -NodeType Content -Path '.\Package.xml' -XPathQuery '/Package/DisplayName' -Index 1 -ResultVariable _content -IndexVariable _foundOnIndex
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-XmlNode.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][ValidateSet("Element", "Attribute", "Content")][string]$NodeType = "Element",
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$true)][string]$XPathQuery,
		[Parameter(Mandatory=$false)][string]$Attribute = $null,
		[Parameter(Mandatory=$false)][int]$Index = 1,
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][string]$IndexVariable = $null,
		[Parameter(Mandatory=$false)][string]$HintPath = $null,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if (($NodeType -eq "Attribute") -and [string]::IsNullOrEmpty($Attribute)) { throw "Parameter -Attribute is required for -NodeType '$($NodeType)'."; }
			
			$Path = Remove-Hint $Path;
			$Path = Expand-Path $Path -Wow64:$Wow64;
			if (([System.IO.Path]::GetExtension($Path) -eq "") -and ![System.IO.File]::Exists($Path)) { $Path += ".xml"; Write-Log -Message "Appending '.xml' to -Path -> '$($Path)'." -Source ${CmdletName}; }

			$nodes = @(Select-XmlNode -path $Path -query $XPathQuery);
			
			$resultNode = $null;
			$resultIndex = 0;
			Set-PdVar -Name $ResultVariable -Value $result;
			
			if ($nodes.Count -eq 0) { Write-Log -Message "No nodes found." -Source ${CmdletName}; $resultNode = $null; $resultIndex = 0; }
			elseif ($Index -gt $nodes.Count) { Write-Log -Message "[-Index = $($Index)] Count of result nodes less than $($Index)." -Source ${CmdletName}; $resultNode = $null; $resultIndex = 0; }
			elseif (($Index -eq 1) -or ($Index -eq 0)) { Write-Log -Message "[-Index = $($Index)] Selecting first node." -Source ${CmdletName}; $resultNode = $nodes[0]; $resultIndex = 1; }
			elseif ($Index -lt 0) { Write-Log -Message "[-Index = $($Index)] Selecting last node (#$($nodes.Count))." -Source ${CmdletName}; $resultNode = $nodes[$nodes.Count - 1]; $resultIndex = $nodes.Count; }
			else { Write-Log -Message "Selecting node #$($Index) of $($nodes.Count)." -Source ${CmdletName}; $resultNode = $nodes[$Index - 1]; $resultIndex = $Index; }
			
			if (![string]::IsNullOrEmpty($IndexVariable)) { Set-PdVar -Name $IndexVariable -Value $resultIndex; }

			if (($resultNode -ne $null) -and ($NodeType -eq "Attribute") -and ($Attribute.IndexOfAny("*?") -ge 0))
			{
				$pattern = ConvertTo-RegexPattern $Attribute;
				$names = @($resultNode.Attributes | where { [regex]::IsMatch($_.Name, $pattern) } | % { $_.Name });
				$using = $(if ($names.Count -gt 0) { $names[0] } else { $Attribute });
				Write-Log -Message "Attribute names matching '$($Attribute)' [$($pattern)]: $($names.Count) ($([string]::Join(', ', $names))) - using '$($using)'." -Source ${CmdletName};
				$Attribute = $using;
			}

			$source = "?";
			$value = $null;
			if ($resultNode -eq $null) { $source = "-none-"; $value = $null; }
			elseif ($NodeType -eq "Element") { $source = "OuterXml"; $value = $resultNode.OuterXml; }
			elseif ($NodeType -eq "Attribute") { $source = "attribute '$($Attribute)'"; $value = $resultNode.GetAttribute($Attribute); }
			elseif ($NodeType -eq "Content") { $source = "InnerText"; $value = $resultNode.InnerText; }
			else { throw "Unsupported -NodeType '$($NodeType)'."; }

			if ($resultNode -eq $null) { Write-Log -Message "No result node." -Source ${CmdletName}; }
			else { Write-Log -Message "Reading $($source) from result node: '$(if (($value -eq $null) -or ($value.Length -le 32)) { $value } else { [string]::Concat($value.Substring(0, 29), '...') })'." -Source ${CmdletName}; }

			if (![string]::IsNullOrEmpty($ResultVariable)) { Set-PdVar -Name $ResultVariable -Value $value; }
		}
		catch
		{
			$failed = "Failed to read XML node [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-XmlNode {
	<#
	.SYNOPSIS
		The value exists in an XML file
	.DESCRIPTION
		Checks if a value exists in a specific XML file. Apart from specifying the actual XML file, you also have to specify which entry you want to check in the XML file. 
		For this reason, specify the entry type and an XPath query. This query works like the Read-XmlNode command. In addition, you must specify the search string. 
		The wildcards * and ? can be used. If you leave the field empty, the system checks for empty contents.
	.PARAMETER NodeType
		Select the type of the part of the XML file you want to read:
		Element - Basic structural unit of an XML file that usually has a tag at the beginning and the end.
		If you search for an element, the result variable contains the complete element with beginning and end tag and the complete contents between the tags. Searching for an element returns the complete element, for example: <Object type="STRING">QueenMary</Object>.
		Attribute of an element
		If you search for an attribute, the result variable returns the value of the attribute. Searching for the attribute type in the above example element results in: STRING.
		Contents of an element
		If you search for the contents, the result variable contains the complete text contents of the searched element between the beginning and the end tag. This does not include subelements with their text contents. Searching for the contents in the above example element results in: QueenMary.
	.PARAMETER Path
		Path and name of the XML file you want to read. If not specified, the extension .XML is used automatically.
	.PARAMETER XPathQuery
		Used to identify the part of the XML file you want to read with the help of the XPath syntax. XPath query example: //Sample/node1/node2/Object[@type="STRING"]. 
		In this example, the system queries the Object element in Sample/node1/node2 to which the type="STRING" applies. 
		In most cases, the system finds several elements that match the query; therefore, specify an Index in addition.
	.PARAMETER Attribute
		This entry provides the name of the queried attribute if you select Entry type = Attribute. You are allowed to use the * wild card.
	.PARAMETER Index
		The index determines which element is to be read if the query finds several matches in the XML file. The following options are available: 
		First match(default), Last match, Specific Index. Specify the location of the specific element on the match list, for example 3 for the third incidence in the XML file.
	.PARAMETER Content
		Specifies the XML contents (for the actions Add and Change) that will be replaced and/or added to the XML file. You can use variables.
	.PARAMETER HintPath
		Deprecated: Unused Parameter
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-XmlNode -NodeType Element -Path '.\Package.xml' -XPathQuery "/Package/Name[text()=`"New Package`"]" -Index 1 -Content "New"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-XmlNode.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][ValidateSet("Element", "Attribute", "Text")][string]$NodeType = "Element",
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$true)][string]$XPathQuery,
		[Parameter(Mandatory=$false)][string]$Attribute = $null,
		[Parameter(Mandatory=$false)][int]$Index = 1,
		[Parameter(Mandatory=$false)][string]$Content = $null,
		[Parameter(Mandatory=$false)][string]$HintPath = $null,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			# test always ... if (Test-SkipCommand -Context $Context) { return; }

			if (($NodeType -eq "Attribute") -and [string]::IsNullOrEmpty($Attribute)) { throw "Parameter -Attribute is required for -NodeType '$($NodeType)'."; }
			
			$Path = Remove-Hint $Path;
			$Path = Expand-Path $Path -Wow64:$Wow64;
			if (([System.IO.Path]::GetExtension($Path) -eq "") -and ![System.IO.File]::Exists($Path)) { $Path += ".xml"; Write-Log -Message "Appending '.xml' to -Path -> '$($Path)'." -Source ${CmdletName}; }

			$nodes = @(Select-XmlNode -path $Path -query $XPathQuery);
			
			$resultNode = $null;
			$resultIndex = 0;
			
			if ($nodes.Count -eq 0) { Write-Log -Message "No nodes found." -Source ${CmdletName}; $resultNode = $null; $resultIndex = 0; }
			elseif ($Index -gt $nodes.Count) { Write-Log -Message "[-Index = $($Index)] Count of result nodes less than $($Index)." -Source ${CmdletName}; $resultNode = $null; $resultIndex = 0; }
			elseif (($Index -eq 1) -or ($Index -eq 0)) { Write-Log -Message "[-Index = $($Index)] Selecting first node." -Source ${CmdletName}; $resultNode = $nodes[0]; $resultIndex = 1; }
			elseif ($Index -lt 0) { Write-Log -Message "[-Index = $($Index)] Selecting last node (#$($nodes.Count))." -Source ${CmdletName}; $resultNode = $nodes[$nodes.Count - 1]; $resultIndex = $nodes.Count; }
			else { Write-Log -Message "Selecting node #$($Index) of $($nodes.Count)." -Source ${CmdletName}; $resultNode = $nodes[$Index - 1]; $resultIndex = $Index; }
			
			$result = $false;
			if ($resultNode -eq $null) { Write-Log -Message "No result node, returning $($result)." -Source ${CmdletName}; return $result; }

			if (($NodeType -eq "Attribute") -and ($Attribute.IndexOfAny("*?") -ge 0))
			{
				$pattern = ConvertTo-RegexPattern $Attribute;
				$names = @($resultNode.Attributes | where { [regex]::IsMatch($_.Name, $pattern) } | % { $_.Name });
				$using = $(if ($names.Count -gt 0) { $names[0] } else { $Attribute });
				Write-Log -Message "Attribute names matching '$($Attribute)' [$($pattern)]: $($names.Count) ($([string]::Join(', ', $names))) - using '$($using)'." -Source ${CmdletName};
				$Attribute = $using;
			}

			$source = "?";
			$value = $null;
			if ($NodeType -eq "Element") { $source = "OuterXml"; $value = $resultNode.OuterXml; }
			elseif ($NodeType -eq "Attribute") { $source = "attribute '$($Attribute)'"; $value = $resultNode.GetAttribute($Attribute); }
			elseif ($NodeType -eq "Content") { $source = "InnerText"; $value = $resultNode.InnerText; }
			else { throw "Unsupported -NodeType '$($NodeType)'."; }

			$pattern = ConvertTo-RegexPattern $Content;
			$result = [regex]::IsMatch($value, $pattern, [System.Text.RegularExpressions.RegexOptions]::IgnoreCase);
			Write-Log -Message "Comparing $($source) of result node ('$(if (($value -eq $null) -or ($value.Length -le 32)) { $value } else { [string]::Concat($value.Substring(0, 29), '...') })') with '$($Content)' [$($pattern)]: $($result)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test XML node [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Request-EndInstallerSession {
	<#
	.SYNOPSIS
		This command writes a registry value of type DWord EndInstallerSession in the key HKEY_LOCAL_MACHINE\SOFTWARE\CANCOM\Package Deployment\Session Management\Current and sets the content to 1.
	.DESCRIPTION
		This command writes a registry value of type DWord EndInstallerSession in the key HKEY_LOCAL_MACHINE\SOFTWARE\CANCOM\Package Deployment\Session Management\Current and sets the content to 1.If several packages are pending  installation, the caller can check after a package to see if the above value exists and is set to 1 and terminate a running installation session. If a corresponding package is executed via the provided wrapper in Ivanti DSM, the installer session is terminated with the EndInstallerSession command, if necessary.
		The Current key is a so-called volatile key that is automatically deleted at the next system reboot, including the values contained therein. So you do not need to worry about removing the value and the key yourself.
		The command has no dialog and has no parameters.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Request-EndInstallerSession
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Request-EndInstallerSession.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$pdc = Get-PdContext;
			$userPart =  ($pdc.InstallMode -eq "InstallUserPart");

			$keyRoot = $(if ($userPart) {"HKEY_CURRENT_USER"} else {"HKEY_LOCAL_MACHINE"});
			$keyPath = "$($keyRoot):\$($SessionManagementRegistryKeyName)";
			$volatileKeyName = "Current";
			$valueName = "EndInstallerSession";
			$value = 1;
			
			Write-Log -Message "Writing the $($valueName) value '$($value)' to the Session Management Registry at '$($keyPath)\$($volatileKeyName)'." -Source ${CmdletName};
			$key = Get-PdRegistryKey -Path $keyPath -Create -Writable;
			
			$volatileKey = $key.CreateSubKey($volatileKeyName, $true, [Microsoft.Win32.RegistryOptions]::Volatile);
			$key.Close();
			
			$volatileKey.SetValue($valueName, $value);
			$key.Close();
		}
		catch
		{
			$failed = "Failed to write the EndInstallerSession Registry";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function New-InternetLink {
	<#
	.SYNOPSIS
		Creates an Internet shortcut (link) that is opened with the default browser at startup.
	.DESCRIPTION
		Creates an Internet shortcut (link) that is opened with the default browser at startup.
	.PARAMETER Description
		Description of the link. This text appears under the link.
	.PARAMETER Url
		The URL to which the link refers.
	.PARAMETER WorkingDirectory
		The path to the working directory of the shortcut (if necessary).
	.PARAMETER Icon
		The program icon to be used. By default, the icon that is assigned to .htm(l) files is used.
	.PARAMETER Folder
		The name of the subfolder, below the destination specified in "Create at", in which this shortcut is to be created. If the specified folder does not yet exist, it will be created.
	.PARAMETER ComputerRelatedLink
		Indicates whether it is a computer-related link available to all users of a computer (e.g. through in the All Users profile).
	.PARAMETER RunMinimized
		If set to true, Windows Style will be minimized instead of normal
	.PARAMETER ManageUserPortion
		Deprecated. Please dont use this Parameter
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER PreventUninstall
		If set, Remove-InternetLink will not be processed on 'uninstallation Mode'
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		New-InternetLink -Description 'CANCOM GmbH' -Url 'https://www.cancom.de' -WorkingDirectory '' -Icon '' -Folder 'Desktop\' -Context User
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/New-InternetLink.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Description,
		[Parameter(Mandatory=$true)][string]$Url,
		[Parameter(Mandatory=$false)][string]$WorkingDirectory,
		[Parameter(Mandatory=$false)][string]$Icon,
		[Parameter(Mandatory=$false)][string]$Folder,
		[Parameter(Mandatory=$false)][switch]$ComputerRelatedLink = $false,
		[Parameter(Mandatory=$false)][switch]$RunMinimized = $false,
		[Parameter(Mandatory=$false)][switch]$ManageUserPortion = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }
			
			$folderPath = Get-ShellFolderPath -Folder $Folder -Common:$ComputerRelatedLink -Wow64:$Wow64;
			$linkPath = [System.IO.Path]::Combine($folderPath, "$($Description).url");

			if (Test-ReverseMode)
			{
				Write-Log -Message "Deleting link file '$($linkPath)'." -Source ${CmdletName};
				Uninstall-SingleFile -Path $linkPath -Delete -DeleteInUse -Wow64:$Wow64;
				
				return; # exit from reverse mode
			}
			
			if (![System.IO.Directory]::Exists($folderPath))
			{
				Write-Log -Message "Creating link folder '$($folderPath)'." -Source ${CmdletName};
				Install-SingleDirectory -Path $folderPath -Recurse -Wow64:$Wow64;
			}

			$targetPath = $Url;
			
			$iconLocation = $Icon;
			$iconIndex = $null;
			if (![string]::IsNullOrEmpty($iconLocation))
			{
				[int]$number = 0;
				$index = $iconLocation.LastIndexOf(",");
				if (($index -ge 0) -and [Int32]::TryParse($iconLocation.SubString($index + 1), [ref]$number))
				{
					$iconIndex = $number;
					$iconLocation = $Icon.SubString(0, $index);
				}
			}

			$parameters = @{
				Path = $linkPath;
				TargetPath = $targetPath;
				Description = $Description;
				WindowStyle = $(if ($RunMinimized) {"Minimized"} else {"Normal"});
				# ContinueOnError = $ContinueOnError;
			}

			if (![string]::IsNullOrEmpty($arguments)) { $parameters["Arguments"] = $arguments; }

			if (![string]::IsNullOrEmpty($WorkingDirectory)) { $parameters["WorkingDirectory"] = $WorkingDirectory; }

			if (![string]::IsNullOrEmpty($iconLocation)) { $parameters["IconLocation"] = $iconLocation; }
			
			if ($iconIndex -ne $null) { $parameters["IconIndex"] = $iconIndex; }
			
			if ($ManageUserPortion) { Write-Log -Message "The parameter -ManageUserPortion is not supported." -Source ${CmdletName} -Severity 2; }
			
			Write-Log -Message "Creating link file '$($linkPath)'." -Source ${CmdletName};
			New-Shortcut @parameters;
		}
		catch
		{
			$failed = "Failed to create link '$Description'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-PackageCacheEntryPath([string]$scriptPath = $null)
{
	$pdc = Get-PdContext;

	$cacheDir = $pdc.Package.PackageCachePath;
	if ([string]::IsNullOrEmpty($cacheDir))
	{
		$dir = [Environment]::GetFolderPath([System.Environment+SpecialFolder]::CommonApplicationData);
		$cacheDir = [System.IO.Path]::Combine($dir, $PackageCacheDirectoryName);
	}
	
	$entryName = $null;
	if (![string]::IsNullOrEmpty($pdc.Package.ID)) { $entryName = $pdc.Package.ID; }
	elseif (![string]::IsNullOrEmpty($pdc.PackageDirectory)) { $entryName = [System.IO.Path]::GetFileName($pdc.PackageDirectory); }
	elseif (![string]::IsNullOrEmpty($scriptPath)) { $entryName = [System.IO.Path]::GetFileName([System.IO.Path]::GetDirectoryName($scriptPath)); }
	
	if ([string]::IsNullOrEmpty($entryName)) { throw "Cannot determine package cache entry name: No current package and no script path specified."; }
	
	$entryName = (& $script:validateFileName $entryName);
	return [System.IO.Path]::Combine($cacheDir, $entryName);
}

function New-PackageCacheEntry {
	<#
	.SYNOPSIS
		Creates a new cache directory for a package
	.DESCRIPTION
		Creates a new cache directory for a package
	.PARAMETER ScriptPath
		The path to the script
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		New-PackageCacheEntry -ScriptPath '$($env:USERPROFILE)\Documents\PPB Packages\MyPackage\Deploy-Package.ps1'
	.EXAMPLE
		New-PackageCacheEntry -ScriptPath '$($env:USERPROFILE)\Documents\PPB Packages\MyPackage\Deploy-Package.ps1' -ContinueOnError
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/New-PackageCacheEntry.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][string]$ScriptPath = $null,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$pdc = Get-PdContext;
			
			$packageDir = $null;
			if (![string]::IsNullOrEmpty($ScriptPath)) { $packageDir = [System.IO.Path]::GetDirectoryName($ScriptPath); }
			elseif (![string]::IsNullOrEmpty($pdc.PackageDirectory)) { $packageDir = $pdc.PackageDirectory; }
			
			if ([string]::IsNullOrEmpty($packageDir)) { throw "No current package and -ScriptPath not specified."; }
			Write-Log -Message "Package path (source): '$($packageDir)'.";
			
			$cacheEntryDir = Get-PackageCacheEntryPath -scriptPath $ScriptPath;
			Write-Log -Message "Package cache entry path (target): '$($cacheEntryDir)'.";
			
			if ([System.IO.Directory]::Exists($cacheEntryDir))
			{
				Write-Log -Message "Updating existing cache entry at '$($cacheEntryDir)'.";
			}
			
			$updateAcl = {
				param([string]$path)
				
				Write-Log -Message "Updating ACL of '$($path)'." -Source ${CmdletName} -DebugMessage;
				$acl = Get-Acl -Path $path;
				$acl.SetAccessRuleProtection($false, $false);
				Set-Acl -Path $path -AclObject $acl;
			}
			
			$copyContent = {
				param([string]$sourceDir, [string]$targetDir)
				
				if (![System.IO.Directory]::Exists($targetDir))
				{
					Write-Log -Message "Creating target directory '$($targetDir)'." -Source ${CmdletName} -DebugMessage;
					$void = [System.IO.Directory]::CreateDirectory($targetDir);
				}
				else
				{
					foreach ($targetFile in [System.IO.Directory]::GetFiles($targetDir))
					{
						$sourceFile = [System.IO.Path]::Combine($sourceDir, [System.IO.Path]::GetFileName($targetFile));
						if (![System.IO.File]::Exists($sourceFile))
						{
							try
							{
								Write-Log -Message "Deleting Hardlink '$($targetFile)' - not existing '$($sourceFile)'." -Source ${CmdletName} -DebugMessage;
								[System.IO.File]::Delete($targetFile);
							}
							catch
							{
								$failed = "Failed to delete [$targetFile]";
								Write-Log -Message "$($failed): $($_.Exception.Message)" -Source ${CmdletName} -Severity 2;
							}
						}
					}
				}

				foreach ($sourceFile in [System.IO.Directory]::GetFiles($sourceDir))
				{
					$targetFile = [System.IO.Path]::Combine($targetDir, [System.IO.Path]::GetFileName($sourceFile));
					if (![System.IO.File]::Exists($targetFile))
					{
						Write-Log -Message "Creating Hardlink '$($targetFile)' for '$($sourceFile)'." -Source ${CmdletName} -DebugMessage;
						[PSPD.API]::CreateHardLink($targetFile, $sourceFile);
						& $updateAcl -path $targetFile;
					}
					elseif (![PSPD.API]::MatchingHardLinks($targetFile, $sourceFile))
					{
						Write-Log -Message "Deleting Hardlink '$($targetFile)' - not matching '$($sourceFile)'." -Source ${CmdletName} -DebugMessage;
						[System.IO.File]::Delete($targetFile);

						Write-Log -Message "Recreating Hardlink '$($targetFile)' for '$($sourceFile)'." -Source ${CmdletName} -DebugMessage;
						[PSPD.API]::CreateHardLink($targetFile, $sourceFile);
						& $updateAcl -path $targetFile;
					}
					else
					{
						Write-Log -Message "Keeping existing Hardlink '$($targetFile)' for '$($sourceFile)'." -Source ${CmdletName} -DebugMessage;
					}
				}

				foreach ($targetSubDir in [System.IO.Directory]::GetDirectories($targetDir))
				{
					$sourceSubDir = [System.IO.Path]::Combine($sourceDir, [System.IO.Path]::GetFileName($targetSubDir))
					if (![System.IO.Directory]::Exists($sourceSubDir))
					{
						try
						{
							Write-Log -Message "Deleting directory '$($targetSubDir)' - not existing '$($sourceSubDir)'." -Source ${CmdletName} -DebugMessage;
							[System.IO.Directory]::Delete($targetSubDir, $true);
						}
						catch
						{
							$failed = "Failed to delete [$targetSubDir]";
							Write-Log -Message "$($failed): $($_.Exception.Message)" -Source ${CmdletName} -Severity 2;
						}
					}
				}

				foreach ($sourceSubDir in [System.IO.Directory]::GetDirectories($sourceDir))
				{
					$targetSubDir = [System.IO.Path]::Combine($targetDir, [System.IO.Path]::GetFileName($sourceSubDir))
					& $copyContent -sourceDir $sourceSubDir -targetDir $targetSubDir;
				}
			};
			
			& $copyContent -sourceDir $packageDir -targetDir $cacheEntryDir;
			
			return $cacheEntryDir;
		}
		catch
		{
			$failed = "Failed to create the cache entry for the current package";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-PackageCacheEntry {
	<#
	.SYNOPSIS
		Deletes the cache directory of an package
	.DESCRIPTION
		Deletes the cache directory of an package
	.PARAMETER ScriptPath
		The path to the script
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-PackageCacheEntry -ScriptPath '$($env:USERPROFILE)\Documents\PPB Packages\MyPackage\Deploy-Package.ps1'
	.EXAMPLE
		Remove-PackageCacheEntry -ScriptPath '$($env:USERPROFILE)\Documents\PPB Packages\MyPackage\Deploy-Package.ps1' -ContinueOnError
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-PackageCacheEntry.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][string]$ScriptPath = $null,
		[Parameter(Mandatory=$false)][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$cacheEntryDir = Get-PackageCacheEntryPath -scriptPath $ScriptPath;
			Write-Log -Message "Package cache entry path: '$($cacheEntryDir)'.";
			
			if ([System.IO.Directory]::Exists($cacheEntryDir))
			{
				Write-Log -Message "Deleting existing cache entry at '$($cacheEntryDir)'.";
				[System.IO.Directory]::Delete($cacheEntryDir, $true);
			}
			else
			{
				Write-Log -Message "Package cache entry path '$($cacheEntryDir)' does not exist.";
			}
			
		}
		catch
		{
			$failed = "Failed to delete the cache entry for the current package";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Set-ApplicationCompatibilityFlags {
	<#
	.SYNOPSIS
		Sets the compatibility mode for an application
	.DESCRIPTION
		Sets the compatibility mode for an application like reduced colors, fixed display resolution and run as administrator.
	.PARAMETER Path
		The Path to the application
	.PARAMETER Flags
		Flags:
		'- Run for specific operation system: ~ 
		   Possible Values are:
		      Windows Vista: VISTARTM
			  Windows Vista SP1: VISTASP1
			  Windows Vista SP2: VISTASP2
			  Windows 7: WIN7RTM
			  Windows 8: WIN8RTM
			
		- Reduced Color: 256COLOR
		- Fixed Display Resolution: 640X480 
		- High DPA Aware: HIGHDPIAWARE 
		- Run as administrator: RUNASADMIN
	.PARAMETER AllUsers
		Set this setting for all users
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-ApplicationCompatibilityFlags -Path 'C:\Windows\System32\cmd.exe' -Flags 'RUNASADMIN' -Context Computer
	.EXAMPLE
		Set-ApplicationCompatibilityFlags -Path 'C:\Windows\System32\cmd.exe' -Flags '~ WIN7RTM RUNASADMIN' -AllUsers -Context Computer
	.EXAMPLE
		Set-ApplicationCompatibilityFlags -Path 'C:\Windows\System32\cmd.exe' -Flags '~ VISTASP2 256COLOR 640X480 HIGHDPIAWARE RUNASADMIN' -AllUsers -Wow64 -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-ApplicationCompatibilityFla1.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path = $null,
		[Parameter(Mandatory=$true)][AllowNull()][AllowEmptyString()][string]$Flags = $null,
		[Parameter(Mandatory=$false)][switch]$AllUsers = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$Path = Expand-Path $Path -Wow64:$Wow64;

			$action = $(if (![string]::IsNullOrEmpty($Flags)) { "Set" } else { "Set" });
			if ([string]::IsNullOrEmpty($Flags)) { $Flags = $null; }

			$root = $(if ($AllUsers) { "HKEY_LOCAL_MACHINE" } else { "HKEY_CURRENT_USER" });
			$keyPath = "$($root)\Software\Microsoft\Windows NT\CurrentVersion\AppCompatFlags\Layers";
			
			if (![string]::IsNullOrEmpty($Flags))
			{
				Write-Log -Message "Setting compatibility flags for [$($Path)] to [$($Flags)], AllUsers $($AllUsers)." -Source ${CmdletName};
				
				$params = @{
					KeyPath = $keyPath;
					ValueName = $Path;
					Value = $Flags;
					ValueKind = "String";
					Action = "Set";
					Wow64 = $Wow64;
					ContinueOnError = $ContinueOnError;
				}
				
				Write-RegistryValue @params;
			}
			elseif ([string]::IsNullOrEmpty($Path))
			{
				throw "No -Path specified.";
			}
			else
			{
				Write-Log -Message "Removing compatibility flags for [$($Path)] to [$($Flags)], AllUsers $($AllUsers)." -Source ${CmdletName};

				$params = @{
					KeyPath = $keyPath;
					ValueName = $Path;
					Wow64 = $Wow64;
					ContinueOnError = $ContinueOnError;
				}
				
				Remove-RegistryKey @params;
			}
		}
		catch
		{
			$failed = "Failed to set compatibility flags for $($Path)";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-LocalizedString($value, $language = [System.Globalization.CultureInfo]::CurrentUICulture)
{
	return [PSPD.API]::GetLocalizedString($value, $language);
}

function Show-PdMessageBox {
	<#
	.SYNOPSIS
		Displays a dialog box with any message and various buttons.
	.DESCRIPTION
		Displays a dialog box with any message and various buttons; the value of the pressed button is stored in a variable if required. In the options you control the activation of the selected buttons and set a time interval when the dialog box is closed if no user action has taken place.
	.PARAMETER Text
		Text that is displayed in the message window. 
	.PARAMETER ResultVariable
		Optional - Name of a variable containing the pressed button. Specify only the name of the variable, not the $ sign and the curly braces. Depending on which button was pressed, the variable contains one of the following values:
		OK: OK 
		Cancel: CANCEL (ABORT when using option "Abort, Ignore, Retry")
		Yes: YES 
		No: NO 
		Ignore: IGNORE 
		Abort: ABORT 
		Retry: RETRY 
	.PARAMETER Buttons
		Specifies which buttons are displayed in the message window. The following combinations are available:
		- OK
		- OKCancel
		- AbortRetryIgnore
		- YesNoCancel
		- YesNo
		- RetryCancel
		- CancelTryAgainContinue
	.PARAMETER TimeoutSeconds
		Time span in seconds after which the message is automatically closed.
	.PARAMETER TimeoutAction
		Specifies what value the variable should have after automatic closing, as if the user had pressed the corresponding key. -1 = Timeout, 0 = OK/YES,  1 = Cancel/No, 2 = Third
	.PARAMETER Caption
		Text that will be displayed in the title bar of the message window.
	.PARAMETER Icon
		Specifies whether and if so, which icon is displayed in the message window. The following options are available:
		None (default)
		Error
		Warning
		Note
		Information
	.PARAMETER DefaultButton
		The default button that is used when timeout is reached. Default = Timeout
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Show-PdMessageBox -Text "This is a message" -ResultVariable _userAnswer -Buttons OKCancel -TimeoutSeconds 300 -TimeoutAction 0 -Caption Question -Icon Information
	.EXAMPLE
		Show-PdMessageBox -Text "Do you want to shutdown the system?" -ResultVariable _reboot -Buttons YesNo -TimeoutSeconds 300 -TimeoutAction 1 -Caption "Reboot?" -Icon Warning
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Show-PdMessageBox.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][AllowEmptyCollection()][AllowEmptyString()][AllowNull()][string[]]$Text = $null,
		[Parameter(Mandatory=$false)][string]$ResultVariable = $null,
		[Parameter(Mandatory=$false)][string]$Buttons = $null,
		[Parameter(Mandatory=$false)][string]$TimeoutSeconds = $null,
		[Parameter(Mandatory=$false)][string]$TimeoutAction = $null,
		[Parameter(Mandatory=$false)][string]$Caption = $null,
		[Parameter(Mandatory=$false)][string]$Icon = $null,
		[Parameter(Mandatory=$false)][string]$DefaultButton = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if ($Text -eq $null) { $Text = ""; }
			
			# convert string-array to single string
			$message = [string]::Join("`r`n", $Text);

            # replace "<cr>" by CRLF for single-line DSM-strings
            if ([regex]::IsMatch($message, '^[^\r\n]+$')) { $message = [regex]::Replace($message, "<cr>", "`r`n", [System.Text.RegularExpressions.RegexOptions]::IgnoreCase); }
			
			$message = Get-LocalizedString $message;
			
			$title = $Caption;
			if ([string]::IsNullOrEmpty($title)) { $title = "{en} $($moduleName)-Message {de} $($moduleName)-Nachricht {fr} $($moduleName)-Message {es} $($moduleName)-Mensaje {pt} $($moduleName)-Mensagem"; }
			$title = Get-LocalizedString $title;

			if ([string]::IsNullOrEmpty($DefaultButton) -and ![string]::IsNullOrEmpty($TimeoutAction) -and ($TimeoutAction -ne "-1")) { $DefaultButton = $TimeoutAction; }

			function getButtonsAlias($value) { return $(if ([string]::IsNullOrEmpty($value)) { "OK"  } else { $value }); }

			$defaultButtonAlias = @{ "0" = "First"; "1" = "Second"; "2" = "Third"; };
			function getDefaultButtonAlias($value) { return $(if ($defaultButtonAlias.ContainsKey([string]$value)) { $defaultButtonAlias[[string]$value] } else { "First" }); }

			$iconAlias = @{ None = "None"; Hand = "Stop"; Question = "Question"; Exclamation = "Exclamation"; Asterisk = "Information"; Stop = "Stop"; Error = "Stop"; Warning = "Exclamation"; Information = "Information"; };
			function getIconAlias($value) { return $(if ($iconAlias.ContainsKey($value)) { $iconAlias[$value] } else { "None" }); }

			function getTimeoutAlias($value) { return $(if ([string]::IsNullOrEmpty($value)) { "0" } else { [int]$value }); }

			function getNotEmpty([string]$value, [string]$forEmpty = " ") { return $(if ([string]::IsNullOrEmpty($value)) { $forEmpty } else { $value }); }
			
			$topMost = $true;
			
			$parameters = @{
				Text = (getNotEmpty $message);
				Title = (getNotEmpty $title);
				Buttons = (getButtonsAlias $Buttons);
				DefaultButton = (getDefaultButtonAlias $DefaultButton);
				Icon = (getIconAlias $Icon);
				Timeout = (getTimeoutAlias $TimeoutSeconds);
				TopMost = $topMost;
			}

			$result = Show-DialogBox @parameters;

			if (([string]$result) -eq $null) { $result = "SKIPPED"; } # Show-DialogBox returned "nothing" ( ([string]"nothing") resolves to $null )
			elseif ($result -eq $null) { $result = "NULL"; }
			else { $result = ([string]$result).Replace(" ", "").ToUpper(); }

			if ($result -eq "TIMEOUT")
			{
				Write-Log "Dialog is timed out." -Source ${CmdletName};
				
				$index = 0;
				if ([string]::IsNullOrEmpty($TimeoutAction) -or ($TimeoutAction -eq "-1"))
				{
					Write-Log "Timeout allowed - setting result to '$($result)'." -Source ${CmdletName};
				}
				elseif (![int]::TryParse($TimeoutAction, [ref]$index))
				{
					$result = $TimeoutAction;
					Write-Log "Timeout action specified - setting result to '$($result)'." -Source ${CmdletName};
				}
				else
				{
					$list = @();
					switch ($Buttons)
					{
						"OK" { $list = @( "OK" ) }
						"OKCancel" { $list = @( "OK", "CANCEL" ) }
						"AbortRetryIgnore" { $list = @( "ABORT", "RETRY", "IGNORE" ) }
						"YesNoCancel" { $list = @( "YES", "NO", "CANCEL" ) }
						"YesNo" { $list = @( "YES", "NO" ) }
						"RetryCancel" { $list = @( "RETRY", "CANCEL" ) }
						"CancelTryAgainContinue" { $list = @( "CANCEL", "TRYAGAIN", "CONTINUE" ) }
					}
					
					if (($index -ge 0) -and ($index -lt $list.Count))
					{
						$result = $list[$index];
						Write-Log "Timeout action index $($index) specified - setting result to '$($result)'." -Source ${CmdletName};
					}
					else
					{
						Write-Log "Timeout action index $($index) specified for $($list.Count) buttons $([string]::Join(', ', $list)) - using result '$($result)'." -Source ${CmdletName};
					}
				}
			}
			else
			{
				Write-Log "User selected '$($result)'." -Source ${CmdletName};
			}
			
			if (![string]::IsNullOrEmpty($ResultVariable))
			{
				Write-Log "Result variable is '$($ResultVariable)'." -Source ${CmdletName};
				Set-PdVar -Name $ResultVariable -Value $result;
			}
			else
			{
				Write-Log "No result variable specified." -Source ${CmdletName};
			}
		}
		catch
		{
			$failed = "Failed to display the message box";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-LocalGroup {
	<#
	.SYNOPSIS
		Existence of a local group
	.DESCRIPTION
		Checks if a specific local group exists.
	.PARAMETER Name
		The name of the local group
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-LocalGroup -Name TestGroup
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-LocalGroup.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$exists = $false;
			try { $exists = ((Microsoft.PowerShell.LocalAccounts\Get-LocalGroup -SID (Get-LocalAccountSid $Name) -ErrorAction Ignore) -ne $null); } catch { $exists = $false; }
			Write-Log -Message "Local group '$($Name)' exists: $($exists)." -Source ${CmdletName};
			return $exists;
		}
		catch
		{
			$failed = "Failed to test existence of local group [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-LocalGroup {
	<#
	.SYNOPSIS
		Deletes a local group on the system
	.DESCRIPTION
		Use this command to delete a local group on the system where it is executed.
	.PARAMETER Name
		The name of the local group
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-LocalGroup -Name TestGroup
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-LocalGroup.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Write-Log -Message "Deleting the local group '$($Name)'." -Source ${CmdletName};
			Microsoft.PowerShell.LocalAccounts\Remove-LocalGroup -SID (Get-LocalAccountSid $Name);
		}
		catch
		{
			$failed = "Failed to delete local group [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-LocalUser {
	<#
	.SYNOPSIS
		Existence of a local user
	.DESCRIPTION
		Checks if a specific local user exists.
	.PARAMETER Name
		The User name to check 
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-LocalUser -Name TestUser
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-LocalUser.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$exists = $false;
			try { $exists = ((Microsoft.PowerShell.LocalAccounts\Get-LocalUser -SID (Get-LocalAccountSid $Name) -ErrorAction Ignore) -ne $null); } catch { $exists = $false; }
			Write-Log -Message "Local user '$($Name)' exists: $($exists)." -Source ${CmdletName};
			return $exists;
		}
		catch
		{
			$failed = "Failed to test existence of local user [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Remove-LocalUser {
	<#
	.SYNOPSIS
		Deletes a local user on the system
	.DESCRIPTION
		Use this command to delete a local user on the system where it is running.
	.PARAMETER Name
		The name of the local group
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-LocalUser -Name TestUser
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-LocalUser.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			Write-Log -Message "Deleting the local user '$($Name)'." -Source ${CmdletName};
			Microsoft.PowerShell.LocalAccounts\Remove-LocalUser -SID (Get-LocalAccountSid $Name);
		}
		catch
		{
			$failed = "Failed to delete local user [$($Name)]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Show-MultipleChoiceDialog {
	<#
	.SYNOPSIS
		This command shows a dialog box with different options. This can be used to provide the user with a list of mutually exclusive options to choose from. Each option is represented by a radio button.
	.DESCRIPTION
		This command shows a dialog box with different options.
	.PARAMETER Text
		Text displayed in the message window describing the options. 
	.PARAMETER Options
		Options is a list of strings that represent available options. The specification is made in the syntax <variable value>=text. 
	.PARAMETER ResultVariable
		Name of the variable where the dialog result should be stored.
	.PARAMETER Caption
		The title of the dialog
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Show-MultipleChoiceDialog -Text "Would you like to reboot now?" -ResultVariable _answer -Options @('REBOOT=Reboot','ABORT=Abort')
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Show-MultipleChoiceDialog.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Text = $null,
		[Parameter(Mandatory=$true)][AllowEmptyString()][string[]]$Options = $null,
		[Parameter(Mandatory=$true)][string]$ResultVariable = $null,
		[Parameter(Mandatory=$false)][string]$Caption = $null,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$pdc = Get-PdContext;
			If ($pdc.DeployMode -eq "NonInteractive") { Write-Log -Message "Bypassing Show-MultipleChoiceDialog [Mode: $($pdc.DeployMode)]. Text:$($Text)" -Source ${CmdletName}; return; }

			Import-Module "$($scriptDirectory)\SupportFiles\ShowMessage.dll";
			
			if ($Text -eq $null) { $Text = ""; }
			
			$message = Get-LocalizedString $Text;
			
			$title = $Caption;
			if ([string]::IsNullOrEmpty($title)) { $title = $pdc.PackageName; }
			$title = Get-LocalizedString $title;

			$buttons = @("Ok", "Cancel")
			
			$radioButtons = @();
			$radioButtonValueLookup = @{};
			[int]$id = 100;
			foreach ($option in $Options)
			{
				if ([string]::IsNullOrEmpty($option)) { continue; }

				$id++;
				$value, $name = $option.Split("=:", 2);
				if ([string]::IsNullOrEmpty($name)) { $name = $value; $value = $id.ToString(); }
				$localized = Get-LocalizedString $name;
				$radioButtons += "$($id)=$($localized)";
				$radioButtonValueLookup[$id] = $value;
			}
			
			$parameters = @{
				Content = $message;
				Title = $title;
				Buttons = $buttons;
				RadioButtons = $radioButtons;
			}

			Write-Log "Calling ShowMessage\Show-TaskDialog." -Source ${CmdletName};
			$selectedRadioButton = $null;
			$result = ShowMessage\Show-TaskDialog @parameters -RadioButtonVariable selectedRadioButton -Wait;
			
			$radioButtonResult = $null;
			if ($result -ne "Ok")
			{
				Write-Log "Show-TaskDialog returned: '$($result)' - ignoring selected radio button." -Source ${CmdletName};
			}
			elseif ($selectedRadioButton -eq $null)
			{
				Write-Log "Show-TaskDialog returned: '$($result)' - no radio button selected." -Source ${CmdletName};
			}
			elseif (!$radioButtonValueLookup.ContainsKey([int]$selectedRadioButton.ID))
			{
				Write-Log "Show-TaskDialog returned: '$($result)' - unknown selected radio button ID '$($selectedRadioButton.ID)'." -Source ${CmdletName};
			}
			else
			{
				$radioButtonResult = $radioButtonValueLookup[[int]$selectedRadioButton.ID];
				Write-Log "Show-TaskDialog returned: '$($result)', selected radio button is '$($radioButtonResult)' (ID $($selectedRadioButton.ID), Caption '$($selectedRadioButton.Caption)')." -Source ${CmdletName};
			}
			
			Write-Log "Result variable is '$($ResultVariable)'." -Source ${CmdletName};
			Set-PdVar -Name $ResultVariable -Value $radioButtonResult;
		}
		catch
		{
			$failed = "Failed to display the multiple choice box";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-SystemDefaultLanguage {
	<#
	.SYNOPSIS
		Reads the ID of the standard Windows system language and stores the result in a variable.
	.DESCRIPTION
		Reads the ID of the standard Windows system language and stores the result in a variable. The format in which the ID is read can be specified.
	.PARAMETER ResultVariable
		Name of the variable where the read ID should be stored. Specify only the name of the variable.
	.PARAMETER Format
		The format for the ID of the user language read. The following options are available:
		OneLetterDSMLangId
			German (D), English (E), French (F), Spanish (S) and Portuguese (P) are output as system languages. Language variants are not taken into account. All other languages are output as (E).
		IetfLanguageTag
			For example, for "German (Germany)" the value "de-DE" or for "English (United States)" the value "en-US" is returned.
		WindowsLCID
			The Locale ID as returned by Windows, e.g. "1031" for "German (Germany)" or "1033" for "English (United States)".
		ThreeLetterISO
			Three-letter code according to ISO 639-1, for example "deu" for "German" or "eng" for "English".
		ThreeLetterWindows
			Returns the language code as defined in the Windows API, for example "DEU" for "German" or "ENU" for "English".
		TwoLetterISO
			Three-letter code according to ISO 639-1, for example "de" for "German" or "en" for "English".
		EnglishName
			The official English language name of the user language, for example, "German (Germany)" or "English (Untited States)".
		DisplayName
			The localized display name of the user language, for example, "German (Germany)" or "English (USA)".
		NativeName
			The native name corresponds to the display name in the respective language.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-SystemDefaultLanguage -ResultVariable languageID -Format OneLetterDSMLangId
	.EXAMPLE
		Read-SystemDefaultLanguage -ResultVariable languageID -Format IetfLanguageTag
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-SystemDefaultLanguage.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][ValidateSet("OneLetterDSMLangId" ,"IetfLanguageTag" ,"WindowsLCID" ,"ThreeLetterISO" ,"ThreeLetterWindows" ,"TwoLetterISO" ,"EnglishName" ,"DisplayName" ,"NativeName")][string]$Format = "OneLetterDSMLangId",
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$culture = Get-Culture;
			Write-Log -Message "Get-Culture returned: '$($culture.Name)' ($culture.DisplayName)." -Source ${CmdletName};
			
			switch ($format)
			{
				"OneLetterDSMLangId" {
					switch ($culture.TwoLetterISOLanguageName)
					{
						"de" { $result = "D"; break; }
						"en" { $result = "E"; break; }
						"es" { $result = "S"; break; }
						"fr" { $result = "F"; break; }
						"pt" { $result = "P"; break; }
						default { $result = "E"; break; }
					}
					break;
				}
				"IetfLanguageTag" { $result = $culture.Name; break; }
				"WindowsLCID" { $result = $culture.LCID; break; }
				"ThreeLetterISO" { $result = $culture.ThreeLetterISOLanguageName; break; }
				"ThreeLetterWindows" { $result = $culture.ThreeLetterWindowsLanguageName; break; }
				"TwoLetterISO" { $result = $culture.TwoLetterISOLanguageName; break; }
				"EnglishName" { $result = $culture.EnglishName; break; }
				"DisplayName" { $result = $culture.DisplayName; break; }
				"NativeName" { $result = $culture.NativeName; break; }
				default { $result = $culture.Name; break; }
			}
			
			Write-Log -Message "System language ID: '$($result)'." -Source ${CmdletName};
			Set-PdVar -Name $ResultVariable -Value $result;
		}
		catch
		{
			$failed = "Failed to get system language ID";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Read-UserDefaultLanguage {
	<#
	.SYNOPSIS
		Reads the ID of the current user's Windows language and stores the result in a variable.
	.DESCRIPTION
		Reads the ID of the current user's Windows language and stores the result in a variable. The format in which the ID is read can be specified.
	.PARAMETER ResultVariable
		Name of the variable where the read ID should be stored. Specify only the name of the variable.
	.PARAMETER Format
		The format for the ID of the user language read. The following options are available:
		OneLetterDSMLangId
			German (D), English (E), French (F), Spanish (S) and Portuguese (P) are output as system languages. Language variants are not taken into account. All other languages are output as (E).
		IetfLanguageTag
			For example, for "German (Germany)" the value "de-DE" or for "English (United States)" the value "en-US" is returned.
		WindowsLCID
			The Locale ID as returned by Windows, e.g. "1031" for "German (Germany)" or "1033" for "English (United States)".
		ThreeLetterISO
			Three-letter code according to ISO 639-1, for example "deu" for "German" or "eng" for "English".
		ThreeLetterWindows
			Returns the language code as defined in the Windows API, for example "DEU" for "German" or "ENU" for "English".
		TwoLetterISO
			Three-letter code according to ISO 639-1, for example "de" for "German" or "en" for "English".
		EnglishName
			The official English language name of the user language, for example, "German (Germany)" or "English (Untited States)".
		DisplayName
			The localized display name of the user language, for example, "German (Germany)" or "English (USA)".
		NativeName
			The native name corresponds to the display name in the respective language.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Read-UserDefaultLanguage -ResultVariable languageID -Format OneLetterDSMLangId
	.EXAMPLE
		Read-UserDefaultLanguage -ResultVariable languageID -Format IetfLanguageTag
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-UserDefaultLanguage.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ResultVariable,
		[Parameter(Mandatory=$false)][ValidateSet("OneLetterDSMLangId" ,"IetfLanguageTag" ,"WindowsLCID" ,"ThreeLetterISO" ,"ThreeLetterWindows" ,"TwoLetterISO" ,"EnglishName" ,"DisplayName" ,"NativeName")][string]$Format = "OneLetterDSMLangId",
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$culture = Get-UiCulture;
			Write-Log -Message "Get-UiCulture returned: '$($culture.Name)' ($culture.DisplayName)." -Source ${CmdletName};
			
			switch ($format)
			{
				"OneLetterDSMLangId" {
					switch ($culture.TwoLetterISOLanguageName)
					{
						"de" { $result = "D"; break; }
						"en" { $result = "E"; break; }
						"es" { $result = "S"; break; }
						"fr" { $result = "F"; break; }
						"pt" { $result = "P"; break; }
						default { $result = "E"; break; }
					}
					break;
				}
				"IetfLanguageTag" { $result = $culture.Name; break; }
				"WindowsLCID" { $result = $culture.LCID; break; }
				"ThreeLetterISO" { $result = $culture.ThreeLetterISOLanguageName; break; }
				"ThreeLetterWindows" { $result = $culture.ThreeLetterWindowsLanguageName; break; }
				"TwoLetterISO" { $result = $culture.TwoLetterISOLanguageName; break; }
				"EnglishName" { $result = $culture.EnglishName; break; }
				"DisplayName" { $result = $culture.DisplayName; break; }
				"NativeName" { $result = $culture.NativeName; break; }
				default { $result = $culture.Name; break; }
			}
			
			Write-Log -Message "User language ID: '$($result)'." -Source ${CmdletName};
			Set-PdVar -Name $ResultVariable -Value $result;
		}
		catch
		{
			$failed = "Failed to get user language ID";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-TextFile([string]$extension)
{
	if ([string]::ISNullOrEmpty($extension))
	{
		Write-Log -DebugMessage -Message "No extension specified." -Source ${CmdletName};
		return $false;
	}
	
	$extension = ".$($extension.TrimStart('.'))";
	
	$wellKnownTextFileExtensions = @(".txt", ".cmd", ".bat", ".inf");
	
	$path = "Microsoft.PowerShell.Core\Registry::HKEY_CLASSES_ROOT\$($extension)";
	if (!(Test-Path $path))
	{
		$result = ($wellKnownTextFileExtensions -contains $extension);
		Write-Log -DebugMessage -Message "Extension '$($extension)' is not registered - is well known text file extension: $($result)." -Source ${CmdletName};
		return $result;
	}
	
	$key = Get-Item $path;
	$typeName = $key.GetValue(""); if ([string]::IsNullOrEmpty($typeName)) { $typeName = "$($extension.TrimStart('.').ToUpper())-file"; }
	$contentType = $key.GetValue("Content Type");
	$perceivedType = $key.GetValue("PerceivedType");
	
	$textMimeTypes = @("text/plain", "text/xml", "text/html", "text/css");
	if ($textMimeTypes -contains $contentType)
	{
		Write-Log -DebugMessage -Message "Extension '$($extension)' identified as text type '$($typeName)' (content type '$($contentType)')." -Source ${CmdletName};
		return $true;
	}
	elseif ([string]::IsNullOrEmpty($contentType) -and ($perceivedType -eq "text"))
	{
		Write-Log -DebugMessage -Message "Extension '$($extension)' identified as text type '$($typeName)' (perceived type '$($perceivedType)')." -Source ${CmdletName};
		return $true;
	}
	else
	{
		Write-Log -DebugMessage -Message "Extension '$($extension)' identified as '$($typeName)' (content type '$($contentType)', perceived type '$($perceivedType)') - no text file." -Source ${CmdletName};
		return $false;
	}
}

function Set-FileReplaceText {
	<#
	.SYNOPSIS
		Replaces strings in arbitrary files.
	.DESCRIPTION
		Replaces strings in arbitrary files. The command does not work line-oriented like the Edit-OemLine command, but replaces each string to be searched by the changed string. Therefore, no wildcards are allowed in the Find and Replace with text fields.
	.PARAMETER FileName
		File(s) to be changed. The use of variables and placeholders is possible.
	.PARAMETER Path
		Path to the file(s) to be changed.
	.PARAMETER Find
		The actual string to be replaced must be specified, wildcards are not allowed.
	.PARAMETER Replace
		The string that replaces the string to be searched for. Again, no wildcards are allowed.
	.PARAMETER Recurse
		Applies the command to all subdirectories of the directory specified in path.
	.PARAMETER TextFilesOnly
		Allows to restrict the application to text files. The mime type is used as a recognition feature. It is recommended to use this option.
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-FileReplaceText -FileName '*.*' -Path '.\Files' -Find Bla -Replace Blupp -Recurse -TextFilesOnly -Wow64 -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-FileReplaceText.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$FileName = $null,
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$true)][string]$Find,
		[Parameter(Mandatory=$true)][AllowEmptyString()][string]$Replace,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][switch]$TextFilesOnly = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			$Path = Expand-Path $Path -Wow64:$Wow64;
			Write-Log -Message "Replacing text in files of directory '$($Path)'." -Source ${CmdletName};
			if (![System.IO.Directory]::Exists($Path)) { throw "Directory '$($Path)' does not exist."; }
			
			$fileSpecifications = @();
			$pattern = "(^|;+| +)(`"(?<spec>[^`"]+)`"|'(?<spec>[^']+)'|(?<spec>[^ ;]+))"; # '
			$fileSpecifications = @([regex]::Matches($FileName, $pattern) | % { $_.Groups["spec"].Value; });
			Write-Log -Message "File specifications: '$([string]::Join(''', ''', $fileSpecifications))'." -Source ${CmdletName};

			$matchPattern = [regex]::Escape($Find);
			$replacePattern = $Replace;
			$regexOptions = [System.Text.RegularExpressions.RegexOptions]::IgnoreCase;
			Write-Log -Message $(if ($matchPattern -eq $Find) { "Replacing '$($matchPattern)' by '$($replacePattern)'." } else { "Replacing '$($Find) [regular expression: '$($matchPattern)'] by '$($replacePattern)'." }) -Source ${CmdletName};
			
			$isTextFileLookup = @{};
			$doReplace = {param($directory)

				$filePaths = @($fileSpecifications | % { [System.IO.Directory]::GetFiles($directory, $_) } | sort -Unique);
				foreach ($filePath in $filePaths)
				{
					if ($TextFilesOnly)
					{
						$extension = [System.IO.Path]::GetExtension($filePath);
						if (!$isTextFileLookup.ContainsKey($extension)) { $isTextFileLookup[$extension] = Test-TextFile -extension $extension; }
						$isTextFile = $isTextFileLookup[$extension];
						if (!$isTextFile) { Write-Log -Message "Skipping '$($filePath)': no text file." -Source ${CmdletName}; continue; }
					}

					Write-Log -Message "Replacing text in '$($filePath)'." -Source ${CmdletName};
					$backupIndex = 0;
					$backupFilePath = "$($filePath).bak";
					while ([System.IO.File]::Exists($backupFilePath))
					{
						$backupIndex++;
						$backupFilePath = "$($filePath).$($backupIndex).bak";
					}
					
					Write-Log -Message "Creating backup file '$($backupFilePath)'." -Source ${CmdletName};
					[System.IO.File]::Copy($filePath, $backupFilePath);

					$fileInfo = Get-TextWithEncoding -path $filePath;
					
					$text = [regex]::Replace($fileInfo.Text, $matchPattern, $replacePattern, $regexOptions);
					
					[System.IO.File]::WriteAllText($filePath, $text, $fileInfo.Encoding);
				}

				if ($Recurse)
				{
					foreach ($subDirectory in [System.IO.Directory]::GetDirectories($directory))
					{
						& $doReplace -directory $subDirectory;
					}
				}
			}

			& $doReplace -directory $Path;
		}
		catch
		{
			$failed = "Failed to replace text in files of [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Set-FileAttribute {
	<#
	.SYNOPSIS
		This command changes the file attributes for the specified file(s) and directory(s).
	.DESCRIPTION
		This command changes the file attributes for the specified file(s) and directory(s). The attributes "Read-only", "Archive", "System" and "Hidden" can be enabled or removed. The mode of operation as well as the usable syntax follow the DOS command ATTRIB, whereby the use of placeholders and variables is permitted.
	.PARAMETER FileName
		File(s) to be changed. The use of variables and placeholders is possible.
	.PARAMETER Path
		Directory in which the files are searched or for which the attributes are to be changed. The use of variables is possible.
	.PARAMETER Modify
		Specifies the changes to the file or directory. Use + to Add attributes and - to remove attributes. S = System, A = Archive, R = ReadOnly, H = Hidden.
	.PARAMETER OfFiles
		Enable this option if you want to change attributes for files in the specified directory and, if enabled, its subdirectories..
	.PARAMETER OfDirectories
		Enable this option if you want to change the attributes for the specified directory and, if specified, its subdirectories.
	.PARAMETER Recurse
		Includes subdirectories
	.PARAMETER Wow64
		If the script is executed on a 64-bit operating system, the counterpart for 32-bit applications SysWOW64 can be used automatically when referring to the SYSTEM32 directory.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Set-FileAttribute -FileName '*.*' -Path '.\Files' -Modify '-SHRA' -OfFiles
	.EXAMPLE
		Set-FileAttribute -FileName '*.exe' -Path '.\Files' -Modify '+RA' -OfFiles -Recurse -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Set-FileAttribute.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][string]$FileName = $null,
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$true)][string]$Modify,
		[Parameter(Mandatory=$false)][switch]$OfFiles = $false,
		[Parameter(Mandatory=$false)][switch]$OfDirectories = $false,
		[Parameter(Mandatory=$false)][switch]$Recurse = $false,
		[Parameter(Mandatory=$false)][switch]$Wow64 = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context) { return; }

			if ($OfFiles -and [string]::IsNullOrEmpty($FileName)) { throw "-OfFiles is set, but -FileName is empty."; }
			
			$Path = Expand-Path $Path -Wow64:$Wow64;
			$options = @();
			if ($OfFiles) { $options += "of files"; }
			if ($OfDirectories) { $options += "of directories"; }
			if ($Recurse) { $options += "including subdirectories"; }
			Write-Log -Message "Modifying attributes in directory '$($Path)': $([string]::Join(', ', $options))." -Source ${CmdletName};
			if (![System.IO.Directory]::Exists($Path)) { throw "Directory '$($Path)' does not exist."; }
			
			# format: +SRHA -SRHA
			$add = @();
			$remove = @();
			$prefix = "+";
			$addFileAttributes = 0;
			$removeFileAttributes = 0;
			$addDirectoryAttributes = 0;
			$removeDirectoryAttributes = 0;
			foreach ($c in $Modify.ToCharArray())
			{
				$attribute = $null;

				if (($c -eq '+') -or ($c -eq '-')) { $prefix = $c; continue; }
				elseif ($c -eq 'S') { $attribute = [System.IO.FileAttributes]::System; }
				elseif ($c -eq 'R') { $attribute = [System.IO.FileAttributes]::ReadOnly; }
				elseif ($c -eq 'H') { $attribute = [System.IO.FileAttributes]::Hidden; }
				elseif ($c -eq 'A') { $attribute = [System.IO.FileAttributes]::Archive; }
				else { continue; }
				
				if ([string]::IsNullOrEmpty($attribute)) { continue; }
				elseif ($prefix -eq '+') { $add += $attribute; }
				elseif ($prefix -eq '-') { $remove += $attribute; }
			}

			if ($add.Count -gt 0)
			{
				if ($OfFiles)
				{
					$addFileAttributes = [System.IO.FileAttributes]$add;
					Write-Log -Message "Adding file attributes: $($addFileAttributes)." -Source ${CmdletName};
				}
				if ($OfDirectories)
				{
					$addDirectoryAttributes = [System.IO.FileAttributes]@($add | where { $_ -ne [System.IO.FileAttributes]::System });
					Write-Log -Message "Adding directory attributes: $($addDirectoryAttributes)." -Source ${CmdletName};
				}
			}

			if ($remove.Count -gt 0)
			{
				if ($OfFiles)
				{
					$removeFileAttributes = [System.IO.FileAttributes]$remove;
					Write-Log -Message "Removing file attributes: $($removeFileAttributes)." -Source ${CmdletName};
				}
				if ($OfDirectories)
				{
					$removeDirectoryAttributes = [System.IO.FileAttributes]@($remove | where { $_ -ne [System.IO.FileAttributes]::System });
					Write-Log -Message "Removing directory attributes: $($removeDirectoryAttributes)." -Source ${CmdletName};
				}
			}
			
			$fileSpecifications = @();
			if ($OfFiles)
			{
				$pattern = "(^|;+| +)(`"(?<spec>[^`"]+)`"|'(?<spec>[^']+)'|(?<spec>[^ ;]+))"; # '
				$fileSpecifications = @([regex]::Matches($FileName, $pattern) | % { $_.Groups["spec"].Value; });
				Write-Log -Message "File specifications: '$([string]::Join(''', ''', $fileSpecifications))'." -Source ${CmdletName};
			}
			
			$doModify = {param($directory)

				if ($OfDirectories)
				{
					Write-Log -Message "Modifying attributes of '$($directory)'." -Source ${CmdletName};
					$info = New-Object System.IO.DirectoryInfo $directory;
					if ($addDirectoryAttributes -ne 0) { $info.Attributes = ($info.Attributes -bor $addDirectoryAttributes); }
					if ($removeDirectoryAttributes -ne 0) { $info.Attributes = ($info.Attributes -band -bnot $removeDirectoryAttributes); }
				}

				if ($OfFiles)
				{
					$filePaths = @($fileSpecifications | % { [System.IO.Directory]::GetFiles($directory, $_) } | sort -Unique);
					foreach ($filePath in $filePaths)
					{
						Write-Log -Message "Modifying attributes of '$($filePath)'." -Source ${CmdletName};
						$info = New-Object System.IO.FileInfo $filePath;
						if ($addFileAttributes -ne 0) { $info.Attributes = ($info.Attributes -bor $addFileAttributes); }
						if ($removeFileAttributes -ne 0) { $info.Attributes = ($info.Attributes -band -bnot $removeFileAttributes); }
					}
				}

				if ($Recurse)
				{
					foreach ($subDirectory in [System.IO.Directory]::GetDirectories($directory))
					{
						& $doModify -directory $subDirectory;
					}
				}
			}

			& $doModify -directory $Path;
		}
		catch
		{
			$failed = "Failed to modify attributes in [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Get-WindowsVersion {
	<#
	.SYNOPSIS
		Reads out the installed Windows version
	.DESCRIPTION
		Reads out the installed Windows version
	.PARAMETER UseComponents
		Specifies how detailed the version number is to be read out. A component only reads out the major version number. 2 additionally reads out the minor version number. 3 and 4 also read out the build and update version number.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Get-WindowsVersion
	.EXAMPLE
		Get-WindowsVersion -UseComponents 3
	.EXAMPLE
		Get-WindowsVersion -UseComponents 4 -ContinueOnError
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Get-WindowsVersion.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][int]$UseComponents = 2,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$result = $null;
			
			$version = [System.Environment]::OSVersion.Version;
			if (($version.Major -gt 0) -or ($version.Minor -gt 0) -or ($version.Build -gt 0) -or ($version.Revision -gt 0))
			{
				Write-Log -Message "Detected Windows version: $($version)." -Source ${CmdletName};

				$result = $version;

				if ($UseComponents -gt 0)
				{
					$partCount = $UseComponents;
					if ($partCount -le 0) { $partCount = 2; }
					
					$parts = @();
					$parts += $(if ($partCount -ge 1) { [Math]::Max($version.Major, 0) } else { 0 });
					$parts += $(if ($partCount -ge 2) { [Math]::Max($version.Minor, 0) } else { 0 });
					if ($partCount -ge 3) { $parts += [Math]::Max($version.Build, 0) }
					if ($partCount -ge 4)
					{
						$ubr = Get-RegistryValue -KeyPath "HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows NT\CurrentVersion" -ValueName "UBR";
						Write-Log -Message "Detected Update Build Number (UBR): $($ubr)." -Source ${CmdletName};
						
						$parts += [Math]::Max($ubr, 0)
					}
					
					$result = New-Object System.Version $parts;
					Write-Log -Message "Using version number: $($result)." -Source ${CmdletName};
				}
			}
			else
			{
				Write-Log -Message "No Windows version detected." -Source ${CmdletName};
			}
			
			return $result;
		}
		catch
		{
			$failed = "Failed to get Windows version";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Copy-PbFile {
	<#
	.SYNOPSIS
		Copys a file
	.DESCRIPTION
		This command copies a file from the source directory to a destination directory. This command does not override the destination file is source file has the same timestamp. This is a helper function. Please use Copy-File instead
	.PARAMETER From
		The Path parameter is a string containing the source file.
	.PARAMETER To
		The Path parameter is a string containing the destination path.
	.PARAMETER CreateBackup
		CreateBackup creates backup file
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Copy-PbFile -From "C:\temp\test.txt" -To "C:\target"
	.EXAMPLE
		Copy-PbFile -From "C:\temp\test.txt" -To "C:\target" -ContinueOnError -CreateBackup
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Copy-PbFile.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$From,
		[Parameter(Mandatory=$true)][string]$To,
		[Parameter(Mandatory=$false)][Alias("B")][switch]$CreateBackup = $false,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if ($CreateBackup) {
				
				if (Test-Path $To) {
					$lastWriteTimeTarget = (Get-Item $To).LastWriteTime;
					$lastWriteTimeSource = (Get-Item $From).LastWriteTime;

					if ($lastWriteTimeSource -ne $lastWriteTimeTarget) {
						$fileExtension = (Get-Item $To).Extension;

						for ($num = 1 ; $num -le 5 ; $num++) {
							$BackupFileName = $To.Replace($fileExtension, "NI$($num)")
							if (!Test-Path $BackupFileName -Or ($num -eq 5)) {
								Write-Log -Message "Backup of old File enabled. Creating Backupfile: $($BackupFileName)" -Source ${CmdletName};
								Copy-Item -Path $To -Destination $BackupFileName -Force;
								break;
							}
						}
					}
					else {
						Write-Log -Message "Backup of old File enabled but both files has the same write timestamp. Igonoring this file." -Source ${CmdletName};
					}
				}
			}
			Write-Log -Message "Copying file from $($From) to $($To):" -Source ${CmdletName};
			Copy-Item -Path $From -Destination $To -Force
			
		}
		catch
		{
			$failed = "Failed to copy file $From to $To";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}
	

function RegisterFont([string]$Font, [string]$FontPath, [bool]$UserContext = $false) {
		try
		{
			$fileName = (Get-Item $Font).Name;
			$fontExtension = (Get-Item $Font).Extension;
			$fontName = $fileName.Replace($fontExtension, "")

			if ($fontExtension.ToLower() -eq ".ttf") {
				$fontName = $fontName + " (TrueType)"
			}
			else { #OTF
				$fontName = $fontName + " (OpenType)"
			}

			# Register Font in Registry
			$registerFontRegistryPath = "HKLM:\Software\Microsoft\Windows NT\CurrentVersion\Fonts"
			if ($UserContext) {
				$registerFontRegistryPath = "HKCU:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Fonts"
			}
			
			$CurrentItemProperty = $null
			try { $CurrentItemProperty = Get-ItemProperty -Path $registerFontRegistryPath -Name $fontName }
			catch { }
		
			if ($FontPath.StartsWith("$($env:windir)\Fonts\")) {
				$FontPath = $FontPath.Replace("$($env:windir)\Fonts\","")
			}

			if ($null -ne $CurrentItemProperty) { 
				Set-ItemProperty -Name $fontName -Path $registerFontRegistryPath -Value $FontPath
				Write-Log -Message "Successfully updated font: $($fontName)" -Source ${CmdletName};
			}
			else {
				New-ItemProperty -Name $fontName -Path $registerFontRegistryPath -PropertyType string -Value $FontPath
				Write-Log -Message "Successfully registered font: $($fontName)" -Source ${CmdletName};
			}
		}
		catch
		{
			$failed = "Failed to register font $fileName";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			throw "$($failed): $($_.Exception.Message)"
		}
}


function UnregisterFont([string]$FontFile, [bool]$userContext) {
		try
		{
			# Register Font in Registry
			$registerFontRegistryPath = "HKLM:\Software\Microsoft\Windows NT\CurrentVersion\Fonts"
			if ($UserContext) {
				$registerFontRegistryPath = "HKCU:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Fonts"
			}
			Write-Log -Message "Using registry path: $($registerFontRegistryPath)"

			$key = Get-PdRegistryKey -Path $registerFontRegistryPath -AcceptNull -Writable;
			if ($null -ne $key) {
				#$values = @($key.GetValueNames() | % { $key.GetValue($_) } | Where-Object { $_ -like "*$($FontFile.ToLower())"});
				$valueNames = $key.GetValueNames()
				foreach($valueName in $valueNames) {
					if ($null -eq $valueName) { continue }

					$value = $key.GetValue($value)
					if ($null -eq $value) { continue }

					if ($value.ToLower() -like "*$($FontFile.ToLower())") {
						Write-Log -Message "Removing property $($valueName)"
						Remove-ItemProperty -Path $registerFontRegistryPath -Name $valueName
					}
				}
			}
			Write-Log -Message "Successfully unregistered Font $($FontFile)"
		}
		catch
		{
			$failed = "Failed to unregister font $FontFile";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			throw "$($failed): $($_.Exception.Message)"
		}
}
		

function Install-TTF {
	<#
	.SYNOPSIS
		Installs fonts
	.DESCRIPTION
		This command lets you install fonts. The following file extensions are supported: TTF, OTF, TTC, FON
	.PARAMETER Path
		The Path parameter is a string containing the file path to the fonts you want to install.
	.PARAMETER CreateBackup
		CreateBackup creates backup files for each font
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.PARAMETER PreventUninstall
	    If set, Uninstall-TTF will not be processed on 'uninstallation Mode'
	.EXAMPLE
		Uninstall-TTF -FileList @('ALGER.TTF','ARLRDBD.TTF','BROADW.TTF') -Context Computer
	.EXAMPLE
		Uninstall-TTF -FileList @('ALGER.TTF') -Context Computer -ContinueOnError
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Uninstall-TTF.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Path,
		[Parameter(Mandatory=$false)][Alias("B")][switch]$CreateBackup = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }

			$Path = Expand-Path $Path;


			$fontFiles = @();
			if ([System.IO.File]::Exists($Path))
			{
				$fontFiles = @( $Path );
				Write-Log -Message "Installing single font file '$($Path)'." -Source ${CmdletName};
			}
			elseif ([System.IO.Directory]::Exists($Path))
			{
				$fontFiles = @(Get-ChildItem -Path $Path | Where-Object { $_.Name -match '\.(ttf|otf|fon|ttc)$' } | % { $_.FullName });
				Write-Log -Message "Installing font files from directory '$($Path)' (found: $($fontFiles.Count))." -Source ${CmdletName};
			}
			else
			{
				throw "The specified path '$($Path)' does not exist."
			}

			if ($fontFiles.Count -lt 1) {
				Write-Log -Message "No font files found at the given directory ($($path))." -Source ${CmdletName};
				throw "No font files found at the given directory ($($path))."
			}

			
			if (Test-ReverseMode)
			{
				Write-Log -Message "Uninstall Mode. Running Uninstall-TTF for all Files in: $($Path)" -Source ${CmdletName};

				foreach($font in $fontFiles) {
					$fileName = (Get-Item $font).Name;
					Uninstall-TTF -FileList @($fileName) -Context $Context -ContinueOnError:$ContinueOnError
				}

				return; # exit from reverse mode
			}

			[bool]$userContext = ($Context -eq "User" -or $Context -eq "UserPerService")
			# Checking Target Directory to put the fonts in
			if ($userContext) {
				$fontTargetPath = "$($env:localappdata)\Microsoft\Windows\Fonts"
			}
			else {
				# if fontTargetPath can not identified or is somehow empty, i set the target directory to the default windows font directory
				$fontTargetPath = "$($env:windir)\Fonts"
			}
			
			Write-Log -Message "Using $($fontTargetPath) as Font Directory." -Source ${CmdletName};

			if (![System.IO.Directory]::Exists($fontTargetPath)) {
				Write-Log -Message "Target Directory does not exists" -Source ${CmdletName};
				throw "Target Directory does not exists"
			} 
			
			foreach($font in $fontFiles) {
				$fileName = (Get-Item $font).Name;
				
				$fontFileTargetPath = "";
				if ($fontTargetPath.EndsWith('\')) { $fontFileTargetPath = $fontTargetPath + $fileName; }
				else {  $fontFileTargetPath = $fontTargetPath + '\' + $fileName; }

				Copy-PbFile -From $font -To $fontFileTargetPath -CreateBackup:$CreateBackup -ContinueOnError:$ContinueOnError
				RegisterFont -Font $font -FontPath $fontFileTargetPath -UserContext $userContext
			}
		}
		catch
		{
			$failed = "Failed to install TrueType-Font [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function RemoveBackupFiles([string]$FilePath) {
		try
		{
			if ($FilePath.Length -eq 0) {
				Write-Log -Message "No file defined" -Source ${CmdletName};
				return
			} 

			#i am not using the (Get-Item $FileName).Extension because this returns an error if the file does not exist. 
			# I'm just needing the extension name
			$extension = [IO.Path]::GetExtension($FilePath) 
			
			for ($i = 1; $i -lt 6; $i++) {
				$backupFilePath = $FilePath.Replace($extension, ".NI$($i)")
				if (Test-Path $backupFilePath) {
					Remove-item $backupFilePath
					Write-Log -Message "Deleted file $backupFilePath" -Source ${CmdletName};
				}
			}
		}
		catch
		{
			$failed = "Failed to delete backup files [$Path]";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			throw "$($failed): $($_.Exception.Message)"
		}
}

function Uninstall-TTF {
	<#
	.SYNOPSIS
		Uninstalls fonts
	.DESCRIPTION
		This command lets you uninstall fonts. The following file extensions are supported: TTF, OTF, TTC, FON
	.PARAMETER FileList
		The FileList Parameter represents a string list that contains file names of the fonts to be uninstalled
	.PARAMETER DeleteBackups
		DeleteBackups deletes backup files if they exist for the given font
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Uninstall-TTF -FileList @('ALGER.TTF','ARLRDBD.TTF','BROADW.TTF') -Context Computer
	.EXAMPLE
		Uninstall-TTF -FileList @('ALGER.TTF') -Context Computer -ContinueOnError
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Uninstall-TTF.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string[]]$FileList,
		[Parameter(Mandatory=$false)][switch]$DeleteBackups = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -PreventUninstall:$PreventUninstall) { return; }

			if ($FileList.Count -eq 0) {
				Write-Log -Message "No font files to delete" -Source ${CmdletName};
				return
			} 

			[bool]$userContext = ($Context -eq "User" -or $Context -eq "UserPerService")
			# Checking Target Directory to put the fonts in
			if ($userContext) {
				$fontTargetPath = "$($env:localappdata)\Microsoft\Windows\Fonts"
			}
			else {
				# if fontTargetPath can not identified or is somehow empty, i set the target directory to the default windows font directory
				$fontTargetPath = "$($env:windir)\Fonts"
			}

			Write-Log -Message "Using $($fontTargetPath) as Font Directory." -Source ${CmdletName};

			if (![System.IO.Directory]::Exists($fontTargetPath)) {
				Write-Log -Message "Target Directory does not exists" -Source ${CmdletName};
				throw "Target Directory does not exists"
			} 
			
			foreach($fileName in $FileList) {
				$fontFileTargetPath = "";

				if ($fontTargetPath.EndsWith('\')) { $fontFileTargetPath = $fontTargetPath + $fileName; }
				else {  $fontFileTargetPath = $fontTargetPath + '\' + $fileName; }

				if (Test-Path $fontFileTargetPath) {
					if ($DeleteBackups) {
						Write-Log -Message "Delete Backup Flag is set. Deleting backup files." -Source ${CmdletName};
						RemoveBackupFiles -FilePath $fontFileTargetPath
					}

					Write-Log -Message "Unregister Font $($fileName)"
					UnregisterFont -FontFile $fileName -UserContext $userContext

					if (Test-Path $fontFileTargetPath) {
						Write-Log -Message "Deleting file $($fontFileTargetPath)"

						try	{ Remove-Item $fontFileTargetPath -Force }
						catch { 
							Write-Log -Message "Failed to delete $($fontFileTargetPath). The font might be in access. This file will be deleted after next reboot."
							[PSPD.API]::DeleteFileAfterReboot($Path);
						}
						
					}
				}
				else {
					Write-Log -Message "Target File not found $($fileName)" -Source ${CmdletName};
				}
			}
		}
		catch
		{
			$failed = "Failed to uninstall TrueType-Font";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}
	


function New-Share {
	<#
	.SYNOPSIS
		Creates a new SMB share
	.DESCRIPTION
		Creates a new SMB share. The commandlet can create shares on the local system as well as on a remote device. On install mode 'uninstall' this command
		will call "Remove-Share"
	.PARAMETER ServerName
		ServerName takes a string as value an represents the netbios or FQDN name of the target machine. Use localhost for current machine.
	.PARAMETER ShareName
		ShareName takes a string as value and defines the name of the share to be created. If the share name is already existent on the target machine, this command will fail
	.PARAMETER SharePath
		SharePath takes a string as value and defines the directory of the share.
	.PARAMETER Comment
		Comment takes a string as value and defines a description for the share.
	.PARAMETER UserLimit
	    UserLimit takes a string as value and defines the limit of concurrent connections. Default is unlimited.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		New-Share -ServerName localhost -ShareName TestShare -SharePath "$($env:SystemDrive)\Temp"
	.EXAMPLE
		New-Share -ServerName localhost -ShareName TestShare -SharePath "C:\Temp" -ContinueOnError -Context ComputerPerService
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-Share.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ServerName,
		[Parameter(Mandatory=$true)][string]$ShareName,
		[Parameter(Mandatory=$true)][string]$SharePath,
		[Parameter(Mandatory=$false)][string]$Comment,
		[Parameter(Mandatory=$true)][string]$UserLimit,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }

			if (Test-ReverseMode)
			{
				Write-Log -Message "Uninstall Mode. Running Remove-Share for Machine: $($ServerName) and Share $($ShareName)" -Source ${CmdletName};
				Remove-Share -ServerName $ServerName -ShareName $ShareName -Context $Context
				return;
			}

			[int]$userLimitInt = -1
			if (![int]::TryParse($UserLimit, [ref]$userLimitInt)) {
				Write-Log -Message "Not a valid UserLimit value $($UserLimit) - $($userLimitInt)" -Source ${CmdletName};
				throw "Not a valid UserLimit value $($UserLimit)"
			} 

			if ($userLimitInt -lt -1) {
				Write-Log -Message "$($userLimitInt) is not a valid value for UserLimit" -Source ${CmdletName};
				throw "$($userLimitInt) is not a valid value for UserLimit"
			}

			if($userLimitInt -eq -1) { 
				Write-Log -Message "UserLimit is set to max user" -Source ${CmdletName};
				$userLimitInt = 0; 
			}
			else {
				Write-Log -Message "UserLimit is set to $($userLimitInt)" -Source ${CmdletName};
			}


			Write-Log -Message "Using $($ServerName)." -Source ${CmdletName};

			if ($ServerName -eq "localhost" -or $Servername -eq "127.0.0.1" -or $ServerName -eq $env:computername) {
				Write-Log -Message "Using current machine for creating a share"

				if (![System.IO.Directory]::Exists($SharePath)) {
					Write-Log -Message "Target Directory does not exists" -Source ${CmdletName};
					throw "Target Directory does not exists"
				}
				Write-Log -Message "Target Directory exists and is set to $($SharePath)" -Source ${CmdletName};

				$share = GetShareIfExists($ShareName)
				if ($null -ne $share) {
					Write-Log -Message "A share with the same name already exists $($ShareName)" -Source ${CmdletName};
					throw "A share with the same name already exists $($ShareName)"
				}

				$SID ='S-1-1-0' #Workaround. Using SID to any trying to get Name of the Everyone Group. This depends on OS Language
				$objSID = New-Object System.Security.Principal.SecurityIdentifier($SID)
				$objUser = $objSID.Translate([System.Security.Principal.NTAccount])
		
				$everyoneGroup = $objUser.value
				if ($everyoneGroup -eq "") {
					$everyoneGroup = "Everyone"
				}

				$share = New-SmbShare -Description $Comment -ConcurrentUserLimit $userLimitInt -FullAccess $everyoneGroup -Path $SharePath -Name $ShareName
				Write-Log -Message "Successfully created share $($share.Name) on $($share.Path)" -Source ${CmdletName};
			}
			else {
				Write-Log -Message "Using remote machine for creating a share"

				if (Test-Connection $ServerName) {
					Write-Log -Message "Ping to server $ServerName was successful"
				}
				else {
					Write-Log -Message "Warning: Not able to ping Server $($ServerName)" -Source ${CmdletName};
					Write-Log -Message "Maybe ICMP is blocked."
				}

				Invoke-Command -ComputerName $ServerName -scriptblock {
					if (![System.IO.Directory]::Exists($Using:SharePath)) {
						Write-Log -Message "Target Directory does not exists" -Source ${CmdletName};
						throw "Target Directory does not exists"
					}
					Write-Log -Message "Target Directory exists and is set to $($Using:SharePath)" -Source ${CmdletName};
	
					$share = GetShareIfExists($Using:ShareName)
					if ($null -ne $share) {
						Write-Log -Message "A share with the same name already exists $($Using:ShareName)" -Source ${CmdletName};
						throw "A share with the same name already exists $($Using:ShareName)"
					}

					$SID ='S-1-1-0' #Workaround. Using SID to any trying to get Name of the Everyone Group. This depends on OS Language
					$objSID = New-Object System.Security.Principal.SecurityIdentifier($SID)
					$objUser = $objSID.Translate([System.Security.Principal.NTAccount])
			
					$everyoneGroup = $objUser.value
					if ($everyoneGroup -eq "") {
						$everyoneGroup = "Everyone"
					}
			
					$share = New-SmbShare -Description $Using:Comment -ConcurrentUserLimit $Using:userLimitInt -FullAccess $everyoneGroup -Path $Using:SharePath -Name $Using:ShareName
					Write-Log -Message "Successfully created share $($share.Name) on $($share.Path)" -Source ${CmdletName};
				}
			}
		}
		catch
		{
			$failed = "Failed to create share $($ShareName) on path $($SharePath)";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function GetShareIfExists($ShareName) {
	$shares = Get-SmbShare
	$share = $null
	foreach ($s in $shares) { if ($s.Name -ieq $ShareName) { $share = $s; break; } }
	return $share
}

function Remove-Share {
	<#
	.SYNOPSIS
		Removes an SMB share
	.DESCRIPTION
		Removes an SMB share. The commandlet can remove shares on the local system as well as on a remote device.
	.PARAMETER ServerName
		ServerName takes a string as value an represents the netbios or FQDN name of the target machine. Use localhost for current machine.
	.PARAMETER ShareName
		ShareName takes a string as value and defines the name of the share to be removed.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Remove-Share -ServerName localhost -ShareName test
	.EXAMPLE
		Remove-Share -ServerName localhost -ShareName test -ContinueOnError
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Remove-Share.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$ServerName,
		[Parameter(Mandatory=$true)][string]$ShareName,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -PreventUninstall:$PreventUninstall) { return; }

			Write-Log -Message "Removing share on machine: $($ServerName)." -Source ${CmdletName};

			if ($ServerName -eq "localhost" -or $Servername -eq "127.0.0.1" -or $ServerName -eq $env:computername) {
				Write-Log -Message "$($ServerName) is the current machine.";

				$share = GetShareIfExists($ShareName)
				
				if ($null -eq $share) {
					throw "Cannot remove share $($ShareName). Share does not exist.";
				}

				try { 
					Remove-SmbShare -Name $ShareName -Force;

					$share = GetShareIfExists($ShareName)
					
					if ($null -eq $share) { Write-Log -Message "Sucessfully removed share $($ShareName)"; }
					else { 
						throw "Not able to remove share $($ShareName). Share still exists"; 
					}
				}
				catch { 
					throw "Failed to remove share $($ShareName)";
				}
			}
			else {
				Write-Log -Message "Using remote machine $($ServerName) for removing the share $($ShareName)";

				if (Test-Connection $ServerName) {
					Write-Log -Message "Ping to server $ServerName was successful";
				}
				else {
					Write-Log -Message "Warning: Not able to ping Server $($ServerName)" -Source ${CmdletName};
					Write-Log -Message "Maybe ICMP is blocked.";
				}

				Invoke-Command -ComputerName $ServerName -scriptblock {	
					$share = GetShareIfExists($Using:ShareName)
				
					if ($null -eq $share) {
						throw "Cannot remove share $Using:ShareName because it does not exists.";
					}

					try { 
						Remove-SmbShare -Name $Using:ShareName -Force;
	
						$share = GetShareIfExists($Using:ShareName)
						
						if ($null -eq $share) { Write-Log -Message "Sucessfully removed share $($Using:ShareName)"; }
						else { 
							throw "Not able to remove share $($Using:ShareNamee). Share still exists"; 
						}
					}
					catch { 
						throw "Failed to remove share $($Using:ShareName)";
					}
				}
			}
		}
		catch
		{
			$failed = "Failed to remove share $($ShareName)";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}


function Test-Laptop {
	<#
	.SYNOPSIS
		Checks whether the computer is a laptop computer
	.DESCRIPTION
		Checks whether the computer is a laptop computer. 
		Returns true if the chassis type is a laptop and has a battery.
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-Laptop
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-Laptop.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$result = $false
			if(Get-WmiObject -Class win32_systemenclosure | Where-Object { $_.chassistypes -eq 9 -or $_.chassistypes -eq 10 -or $_.chassistypes -eq 14}) 
			{ 
				$result = $true 
				Write-Log  -Message "Computer identified as laptop based on the chassis type." -Source ${CmdletName};
			}
			if(Get-WmiObject -Class win32_battery) 
			{ 
				$result = $true 
				Write-Log  -Message "Computer identified as laptop because it has an battery." -Source ${CmdletName};
			}

			if (-not $result) { Write-Log  -Message "Computer is not a laptop." -Source ${CmdletName}; }
			Write-Log -Message "Returning result: $($result)." -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to test if computer is a laptop.";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}


function Test-Service {
	<#
	.SYNOPSIS
		Checks whether a service is in the desired state 
	.DESCRIPTION
		Checks whether a service is in the desired state . Returns
		false if the service is not in the desired state or does not exist.
	.PARAMETER Name
		The name of the service
	.PARAMETER Property
		The property to check. Possible values: 'State' and 'StartMode'
	.PARAMETER Value
		Possible values for StartMode: Automatic, Disabled, Manual
		Possible values for State:  ContinuePending, Paused, PausePending, Running, StartPending, Stopped, StopPending
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-Service
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-Service.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Name,
		[Parameter(Mandatory=$false)][string]$Property,
		[Parameter(Mandatory=$false)][string]$Value,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -PreventUninstall:$PreventUninstall) { return; }
			#$services = Get-WmiObject win32_Service | Select-Object Name, State, Startmode
			$services = Get-Service

			$serviceToCheck = $null;
			foreach ($service in $services) {
				if ($service.Name -ieq $Name) {
					$serviceToCheck = $service
					break;
				}
			}

			if ($null -eq $serviceToCheck) {
				Write-Log -Message "Service with name $($Name) not found." -Source ${CmdletName};
				return $false;
			}
			
			Write-Log -Message "Service $($Name) found." -Source ${CmdletName};

			if (-not $PSBoundParameters.ContainsKey('Property')) { 
				if ($PSBoundParameters.ContainsKey('Value')) { 
					Write-Log -Message "No Properties to validate. Value Parameter is set to $($Value)." -Source ${CmdletName};
					Write-Log -Message "Value Parameter is invalid without Property Parameter. Value will be ignored. Returning True." -Source ${CmdletName};
					}
				else { 
					Write-Log -Message "No Properties to validate. Returning True." -Source ${CmdletName};
				}
				return $true;  
			}

			if (-not $PSBoundParameters.ContainsKey('Value')) { 
				Write-Log -Message "Property is set to $($Property) but no Value is set. Stopping execution" -Source ${CmdletName};
				throw "Invalid combination of Parameters: Property is definied but no value parameter is set. Remove Property of define a Value Parameter"
			}

			if ($Property -eq "State") {
				Write-Log -Message "Checking state of service" -Source ${CmdletName};
				Write-Log -Message "Service State is $($service.State) and should be $($Value)." -Source ${CmdletName};
				return ($service.Status.ToString() -ieq $Value);
			}
			ElseIf ($Property -eq "StartMode"){
				Write-Log -Message "Checking Startmode of service" -Source ${CmdletName};
				Write-Log -Message "Service Startmode is $($service.Startmode) and should be $($Value)." -Source ${CmdletName};
				return ($service.StartType.ToString() -ieq $Value);
			}
			else {
				throw "Invalid Property set: $($Property). Property unknown. Please use 'State' or 'StartMode'"
			}
		}
		catch
		{
			$failed = "Failed to Test Service";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}
function Test-OnBattery {
	<#
	.SYNOPSIS
		Checks whether the current computer is running on battery
	.DESCRIPTION
		Checks whether the current computer is running on battery. Returns
		false if computer has no battery or is on power
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-LocalAdmin
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-LocalAdmin.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			if(Get-WmiObject -Class win32_battery) 
			{ 
				$onBattery = !(Get-WmiObject -Class BatteryStatus -Namespace root\wmi).PowerOnLine
				if ($onBattery) { Write-Log  -Message "Computer is running on Battery" -Source ${CmdletName}; }
				else { Write-Log  -Message "Computer is NOT running on Battery" -Source ${CmdletName};}
				return $onBattery;
			}
			else {
				Write-Log  -Message "Computer has no Battery. Possibly the current machine is not an laptop." -Source ${CmdletName};
				return $false;
			}
		}
		catch
		{
			$failed = "Failed to test if computer is on battery";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-InternetAccess {
	<#
	.SYNOPSIS
		Checks if it is possible to send web request the test web address with IPv4 or IPv6
	.DESCRIPTION
		Checks if it is possible to send web request the test web address with IPv4 or IPv6 using
		the registry key for ActiveWebProbe: HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet
		Uses Web Proxy if configured.
	.EXAMPLE
		Test-InternetAccess
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-InternetAccess.html
	#>
	[CmdletBinding()]
	param (
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{	
		try {
			$webhost = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveWebProbeHost;
			$path = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveWebProbePath;
			$content = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveWebProbeContent

			Write-Log -Message "Testing internet access with IPv4" -Source ${CmdletName};
			Write-Log -Message "Active Web Probe Host: $($webhost)" -Source ${CmdletName};
			Write-Log -Message "Active Web Probe Path: $($path)" -Source ${CmdletName};
			Write-Log -Message "Active Web Probe Content: $($content)" -Source ${CmdletName};


			$url = ("http://{0}/{1}" -f $webhost, $path)
			if ((Invoke-WebRequest $url -UseBasicParsing).Content -eq $content) { 
				Write-Log -Message 'IPv4 web probe succeeded.' -Source ${CmdletName};
				return $true 
			}

			Write-Log -Message 'IPv4 web probe failed. Trying again using system web proxy' -Source ${CmdletName};
			$proxyurl = ([System.Net.WebRequest]::GetSystemWebproxy()).GetProxy($url)
			if ((Invoke-WebRequest -Proxy $proxyurl -ProxyUseDefaultCredentials $url -UseBasicParsing).Content -eq $content) { 
				Write-Log -Message 'IPv4 web probe succeeded.' -Source ${CmdletName};
				return $true 
			}
		} catch {
			Write-Log -Message 'IPv4 web probe failed.' -Source ${CmdletName};
		}
		
		# Web request test with IP6 
		try {
			$webhost = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveWebProbeHostV6;
			$path = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveWebProbePathV6;
			$content = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveWebProbeContentV6;

			Write-Log -Message "Testing internet access with IPv6" -Source ${CmdletName};
			Write-Log -Message "Active Web Probe Host: $($webhost)" -Source ${CmdletName};
			Write-Log -Message "Active Web Probe Path: $($path)" -Source ${CmdletName};
			Write-Log -Message "Active Web Probe Content: $($content)" -Source ${CmdletName};

			$url = ("http://{0}/{1}" -f $webhost, $path)
			if ((Invoke-WebRequest $url -UseBasicParsing).Content -eq $content) { 
				Write-Log -Message 'IPv6 web probe succeeded.' -Source ${CmdletName};
				return $true 
			}

			Write-Log -Message 'IPv6 web probe failed. Trying again using system web proxy' -Source ${CmdletName};
			$proxyurl = ([System.Net.WebRequest]::GetSystemWebproxy()).GetProxy($url)
			if ((Invoke-WebRequest -Proxy $proxyurl -ProxyUseDefaultCredentials $url -UseBasicParsing).Content -eq $content) { 
				Write-Log -Message 'IPv6 web probe succeeded.' -Source ${CmdletName};
				return $true 
			}
		} catch {
			Write-Log -Message 'IPv6 web probe failed.' -Source ${CmdletName};
		}
		
		# DNS resolution test with IPv4
		$ipaddress = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveDnsProbeHost;
		$content = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveDnsProbeContent;

		Write-Log -Message "Testing internet by Domain Name resolution with IPv4" -Source ${CmdletName};
		Write-Log -Message "IP Address: $($ipaddress)" -Source ${CmdletName};
		Write-Log -Message "Content: $($content)" -Source ${CmdletName};

		$resolved = Resolve-DnsName -Type A -ErrorAction SilentlyContinue $ipaddress
		Write-Log -Message "Resolved IPv4: ${$resolved.IPAddress}" -Source ${CmdletName};

		if ($resolved.IPAddress -eq $content) {
			Write-Log -Message 'Resolved IPv4 Adress matches registry value. Name resolution succeeded.' -Source ${CmdletName};
			return $true
		}

		Write-Log -Message 'IPv4 Address does not match expected value. Proceeding with IPv6' -Source ${CmdletName};

		# DNS resolution test with IPv6
		$ipaddress = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveDnsProbeHostV6;
		$content = (Get-ItemProperty HKLM:\SYSTEM\CurrentControlSet\Services\NlaSvc\Parameters\Internet).ActiveDnsProbeContentV6;

		Write-Log -Message "Testing internet by Domain Name resolution with IPv6" -Source ${CmdletName};
		Write-Log -Message "IP Address: $($ipaddress)" -Source ${CmdletName};
		Write-Log -Message "Content: $($content)" -Source ${CmdletName};

		$resolved = Resolve-DnsName -Type AAAA -ErrorAction SilentlyContinue $ipaddress
		Write-Log -Message "Resolved IPv6: ${$resolved.IPAddress}" -Source ${CmdletName};

		if ($resolved.IPAddress -eq $content) {
			Write-Log -Message 'Resolved IPv6 Adress matches registry value. Name resolution succeeded.' -Source ${CmdletName};
			return $true
		}
		Write-Log -Message 'IPv6 Address does not match expected value.' -Source ${CmdletName};

		# Everything failed
		Write-Log -Message 'All tests failed. Probably there is no internet connection' -Source ${CmdletName};
		return $false
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}


function Test-LocalAdmin {
	<#
	.SYNOPSIS
		Checks if the current user is in local admin group
	.DESCRIPTION
		Checks if the current user is in local admin group
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-LocalAdmin
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-LocalAdmin.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	
	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$isAdmin = (new-object System.Security.Principal.WindowsPrincipal([System.Security.Principal.WindowsIdentity]::GetCurrent())).IsInRole(([System.Security.Principal.SecurityIdentifier]"S-1-5-32-544"))
			If ($isAdmin)
			{
				Write-Log -Message "Current User ${env:USERNAME} is in local admin group. Returning TRUE" -Source ${CmdletName};
				return $true;
			}
			else  
			{
				Write-Log -Message "Current User ${env:USERNAME} is NOT in local admin group. Returning FALSE" -Source ${CmdletName};
				return $false;
			}
		}
		catch
		{
			$failed = "Failed to test if current user is in local admin group";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-RunningAsService {
	<#
	.SYNOPSIS
		Checks if the script is executed in session 0
	.DESCRIPTION
		Checks if the script is executed in session 0
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-RunningAsService
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-RunningAsService.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	

	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$sessionID = (Get-Process -PID $pid).SessionID
			If ($sessionID -eq 0)
			{
				Write-Log -Message "Current process ${$pid} is running in system session (Session ID 0). Returning TRUE" -Source ${CmdletName};
				return $true;
			}
			else  
			{
				Write-Log -Message "Current process ${$pid} is NOT running in system session (Session ID 0). Returning FALSE" -Source ${CmdletName};
				return $false;
			}
		}
		catch
		{
			$failed = "Failed to test current session of process ${$pid}";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-VirtualMachine {
	<#
	.SYNOPSIS
		Checks whether the current computer is a virtual machine
	.DESCRIPTION
		Checks whether the current computer is a virtual machine
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-VirtualMachine
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-VirtualMachine.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	

	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$platform = Get-HardwarePlatform -ContinueOnError $ContinueOnError
			Write-Log -Message "Hardware Platform: $platform" -Source ${CmdletName};

			if ($platform -ieq "Physical") {
				Write-Log -Message "Hardware Platform is NOT virtual: Returing False" -Source ${CmdletName};
				return $false
			}
			
			Write-Log -Message "Hardware Platform is virtual: Returing True" -Source ${CmdletName};
			return $true
		}
		catch
		{
			$failed = "Failed to test current hardware platform";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-MicrosoftUpdate {
	<#
	.SYNOPSIS
		Checks whether a update is installed (KB Article)
	.DESCRIPTION
		Checks whether a update is installed (KB Article)
		Takes string parameter that represents an kb (knowlege base) number.
	.PARAMETER KBNumber
		The KB Number to check
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-MicrosoftUpdat
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-MicrosoftUpdate.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$KBNumber,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	

	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$exists = Test-MSUpdates -KBNumber $KBNumber -ContinueOnError $ContinueOnError
			if ($exists) {
				Write-Log -Message "KBNumber $KBNumber found. Returing True" -Source ${CmdletName};
				return $true
			}
			Write-Log -Message "KBNumber NOT $KBNumber found. Returing False" -Source ${CmdletName};
			return $false
		}
		catch
		{
			$failed = "Failed to check $KBNumber";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}


function Test-UserLoggedOn {
	<#
	.SYNOPSIS
		Checks whether terminal service login is disabled or restricted (Drain mode)
	.DESCRIPTION
		Checks whether terminal service login is disabled or restricted (Drain mode)
		Returns True if 'drain mode' or 'drain until restart' is active
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-UserLoggedOn
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-UserLoggedOn.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory=$true)][string]$Username,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)
	

	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
	
	process
	{
		try
		{
			$sessions = Get-CimInstance -ClassName Win32_ComputerSystem | Select-Object -Property UserName

			foreach($session in $sessions) {
				if ($session.UserName -like $Username) {
					Write-Log -Message "Found active Session that matches username: $Username. Returning True" -Source ${CmdletName};
					return $true
				}
				else {
					Write-Log -Message "Active Session that matches username: $Username NOT found. Returning False" -Source ${CmdletName};
					return $false
				}
			}
			return $false
		}
		catch
		{
			$failed = "Failed to query for username: $Username";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	
	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-RemoteLoginDisabled {
	<#
	.SYNOPSIS
		Checks whether terminal service login is disabled or restricted (Drain mode)
	.DESCRIPTION
		Checks whether terminal service login is disabled or restricted (Drain mode)
		Returns True if 'drain mode' or 'drain until restart' is active
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-RemoteLoginDisabled
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-RemoteLoginDisabled.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $false)][Alias("X")][switch]$ContinueOnError = $false
	)
		
	
	begin {
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
		
	process {
		try {
			Get-WmiObject win32_terminalservicesetting -N "root\cimv2\terminalservices" | % {
				if ($_.logons -eq 1) {
					Write-Log -Message "Logons are disabled. Returning TRUE" -Source ${CmdletName};
					return $true
				}
				Else {
					switch ($_.sessionbrokerdrainmode) {
						0 {
							Write-Log -Message "Logons are not disabled. Everyone can access. Returning FALSE" -Source ${CmdletName};
							return $false
						}
						1 {
							Write-Log -Message "Logons are disabled until restart (DRAIN UNTIL RESTART MODE). Returning TRUE" -Source ${CmdletName};
							return $true
						}
						2 {
							Write-Log -Message "Only already logged in users can login (DRAIN MODE). Returning TRUE" -Source ${CmdletName};
							return $true
						}
						default {
							Write-Log -Message "Unknown Drain Mode Value $($_sessionbrokerdrainmode)" -Source ${CmdletName};
							return $true
						}
					}
				}
			}
		}
		catch {
			$failed = "Failed to query for username: $Username";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
		
	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-ComputerLocked {
	<#
	.SYNOPSIS
		Checks whether users are still logged in and active. 
	.DESCRIPTION
		Checks whether users are still logged in and active.
		Returns True if no users are logged in and login is visible
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-ComputerLocked
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-ComputerLocked.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $false)][Alias("X")][switch]$ContinueOnError = $false
	)
			
		
	begin {
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
			
	process {

		try { 
			$user = Get-WmiObject -Class win32_computersystem | select -ExpandProperty username -ErrorAction Stop 
		} 
		catch { 
			Write-Log -Message "No User logged on" -Source ${CmdletName};
			return $false 
		}

		Write-Log -Message "Current User: $($user)" -Source ${CmdletName};

		try {
			if ((Get-Process logonui -ErrorAction Stop) -and ($user)) {
				Write-Log -Message "Logon UI active. User is logged off" -Source ${CmdletName};
				return $true
			}
			else {
				Write-Log -Message "Logon UI inactive. User is logged on" -Source ${CmdletName};
				return $false
			}
		}
		catch { 
			if ($user) {
				Write-Log -Message "User is logged on." -Source ${CmdletName};
			}
			else {
				$failed = "Unexpected System Status. Can't get user processes.";
				Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
				If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
			}
			return $false 
		}
	}
			
	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Test-WindowsFeature {
	<#
	.SYNOPSIS
		Checks whether a windows feature is installed
	.DESCRIPTION
		Checks whether a windows feature is installed
        Takes an string for the featurename. Wildcards (*) are allowed.
		If wildcards are used, all features that fit pattern must be installed to return 'true'
	.PARAMETER Name
		The name of the windows feature to be checked
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Test-WindowsFeature -Name TelnetClient
	.EXAMPLE
		Test-WindowsFeature -Name Microsoft-Hyper-V* -ContinueOnError
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-WindowsFeature.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $true)][string]$Name,
		[Parameter(Mandatory = $false)][Alias("X")][switch]$ContinueOnError = $false
	)
			
		
	begin {
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
			
	process {

		try { 
			if ($Name -eq "") {
				Write-Log -Message "Parameter 'Name' not set" -Source ${CmdletName};
				throw "Parameter 'Name' not set"
			}

			$features = Get-WmiObject -Class Win32_OptionalFeature |
			ForEach-Object {
				$properties = @{
					Installed   = ($_.InstallState -eq 1);
					Name        = $_.Name;
					DisplayName = $_.Caption;
				}
				New-Object PsObject -Property $properties
			}

			Write-Log -Message "Found $($features.Count) optional features on this computer." -Source ${CmdletName};
					
			$multipleFeaturesInstalled = $false
			foreach ($feature in $features) {
				if ($feature.Name -like $Name) {
					#Removed displayname check because of different behavior on multilingual machines. Name is unique between multiple languages
					Write-Log -Message "Feature with Name: '$($feature.Name)' and DisplayName '$($feature.DisplayName)' matches parameter: '$Name'" -Source ${CmdletName};
					$multipleFeaturesInstalled = $true
					if (!$feature.Installed) {
						Write-Log -Message "Feature: '$($feature.Name)' is NOT installed. Returning FALSE" -Source ${CmdletName};
						return $false
					}
				}
			}
			return $multipleFeaturesInstalled
		} 
		catch { 
			$failed = "Failed to query WMI for feature $($Name)";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
			
	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}	

function Enable-WindowsFeature {
	<#
	.SYNOPSIS
		Installs and activates one or many Windows features.
	.DESCRIPTION
		Installs and activates one or many Windows features.
        Takes an array of strings for the featurenames. Wildcards (*) are allowed
	.PARAMETER FeatureNames
		A list of feature names to be installed
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.PARAMETER PreventUninstall
		If switch is used and installation mode is 'uninstall' then uninstallation will be ignored
	.EXAMPLE
		Enable-WindowsFeature -FeatureNames TelnetClient
	.EXAMPLE
		Enable-WindowsFeature -FeatureNames @('WorkFolders-Client','SmbDirect','TelnetClient','WAS-NetFxEnvironment','Microsoft-Windows-Subsystem-Linux') -Context Computer
	.EXAMPLE
		Enable-WindowsFeature -FeatureNames @('Microsoft-Hyper-V*') -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Enable-WindowsFeature.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $true)][string[]]$FeatureNames,
		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory = $false)][Alias("X")][switch]$ContinueOnError = $false,
		[Parameter(Mandatory = $false)][switch]$PreventUninstall = $false
	)
		
	
	begin {
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
		
	process {
		if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }

		if (Test-ReverseMode) {
			Write-Log -Message "Uninstall Mode. Running Disable-WindowsFeature for Features: $($FeatureNames)" -Source ${CmdletName};
			Disable-WindowsFeature -FeatureNames $FeatureNames -Context $Context
			return; # exit from reverse mode
		}

		try { 
			$features = Get-WindowsOptionalFeature -Online
			Write-Log -Message "Found: $($features.Count) features." -Source ${CmdletName};

			foreach ($Name in $FeatureNames) {			
				$featureFoundOnAvailable = $false
				foreach ($feature in $features) { # Search for Feature
					if ($feature.FeatureName -like $Name) { # Found Feature that matches condition
						$featureFoundOnAvailable = $true
						Write-Log -Message "Feature '$($feature.FeatureName)' matches Condition '$Name'. Checking Status of Feature: '$($feature.FeatureName)'" -Source ${CmdletName};
						$alreadyInstalled = Test-WindowsFeature -Name $feature.FeatureName
						if ($alreadyInstalled) {
							Write-Log -Message "Windows Feature $($feature.FeatureName) already installed." -Source ${CmdletName};
							continue
						}
		
						try {
							Write-Log -Message "Trying to install Feature '$($feature.FeatureName)'"  -Source ${CmdletName};	
							Enable-WindowsOptionalFeature -Online -FeatureName $feature.FeatureName -NoRestart -All
							Write-Log -Message "Successfully installed Feature '$($feature.FeatureName)'"  -Source ${CmdletName};	
							break;
						}
						catch {
							$failed = "Failed to install Feature '$($feature.FeatureName)'";
							Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
							If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
						}
					}
				}

				if ($featureFoundOnAvailable -eq $false) {
					# Was not found on feature list
					$failed = "Feature $($Name) is not available for this computer.";
					Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
					If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
				}
			}
		} 
		catch { 
			$failed = "Unable to install all features.";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
		
	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}	
	
function Disable-WindowsFeature {
	<#
	.SYNOPSIS
		Disables one or many Windows features.
	.DESCRIPTION
		Disables one or many Windows features.
        Takes an array of strings for the featurenames. Wildcards (*) are allowed
	.PARAMETER FeatureNames
		A list of feature names to be disabled
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.PARAMETER PreventUninstall
		If switch is used and installation mode is 'uninstall' then uninstallation will be ignored
	.EXAMPLE
		Disable-WindowsFeature -FeatureNames TelnetClient
	.EXAMPLE
		Disable-WindowsFeature -FeatureNames @('WorkFolders-Client','SmbDirect','TelnetClient','WAS-NetFxEnvironment','Microsoft-Windows-Subsystem-Linux') -Context Computer
	.EXAMPLE
		Disable-WindowsFeature -FeatureNames @('Microsoft-Hyper-V*') -Context Computer
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Disable-WindowsFeature.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $true)][string[]]$FeatureNames,
		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory = $false)][Alias("X")][switch]$ContinueOnError = $false
	)
		
	
	begin {
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
		
	process {

		try { 
			$features = Get-WindowsOptionalFeature -Online
			Write-Log -Message "Found: $($features.Count) features." -Source ${CmdletName};

			foreach ($Name in $FeatureNames) {
				$featureFoundOnAvailable = $false
				foreach ($feature in $features) {
					if ($feature.FeatureName -like $Name) {
						$featureFoundOnAvailable = $true
						Write-Log -Message "Feature '$($feature.FeatureName)' matches Condition '$Name'. Checking Status of Feature: '$($feature.FeatureName)'" -Source ${CmdletName};
						$isInstalled = Test-WindowsFeature -Name $feature.FeatureName
						if (!$isInstalled) {
							Write-Log -Message "Windows Feature $($feature.FeatureName) not installed. Ignoring" -Source ${CmdletName};
							continue
						}
		
						try {
							Write-Log -Message "Trying to disable Feature '$($feature.FeatureName)'"  -Source ${CmdletName};	
							Disable-WindowsOptionalFeature -Online -FeatureName $feature.FeatureName -NoRestart
							Write-Log -Message "Successfully disabled Feature '$($feature.FeatureName)'"  -Source ${CmdletName};	
							break;
						}
						catch {
							$failed = "Failed to disable Feature '$($feature.FeatureName)'";
							Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
							If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
						}
					}
				}

				if ($featureFoundOnAvailable -eq $false) {
					# Was not found on feature list 
					Write-Log -Message "Feature $($Name) is not available on this computer. Ignoring..."; -Source ${CmdletName};
				}
			}
				
		} 
		catch { 
			$failed = "Unable to disable all features";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
		
	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}	

function Suspend-PdBitlocker {
	<#
	.SYNOPSIS
		Suspends Bitlocker Pin entry
	.DESCRIPTION
		Suspends Bitlocker Pin entry
	.PARAMETER RebootCount
		Defines how often the input of the Bitlocker pin is inhibited
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Suspend-PdBitlocker -RebootCount 5
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Suspend-PdBitlocker.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $true)][int]$RebootCount,
		[Parameter(Mandatory = $false)][Alias("X")][switch]$ContinueOnError = $false
	)

	begin {
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
		
	process {

		try { 
			$drives = Get-BitLockerVolume

			Write-Log "Found $($drives.Count) Bitlocker mounpoints." 
			
			foreach ($mountpoint in $drives) {
				Write-Log -Message "Trying to suspend Bitlocker for mountpoint $($mountpoint.MountPoint)."  -Source ${CmdletName};
				Suspend-BitLocker -MountPoint $mountpoint.MountPoint -RebootCount $RebootCount
				Write-Log -Message "Successfull. Suspending Bitlocker PIN entry for $($RebootCount) reboots on mountpoint $($mountpoint.MountPoint)"
			}
		} 
		catch { 
			$failed = "Unable to suspend Bitlocker Volumes";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
		
	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Invoke-PowerShellCode
{
	<#
	.SYNOPSIS
		Executes PowerShell Code
	.DESCRIPTION
		Executes PowerShell Code and stores the result in a variable.
	.PARAMETER Script
		The Code to run
	.PARAMETER LogAs
		Description for logging
	.PARAMETER ReverseScript
		The Code to run on uninstall mode.
	.PARAMETER LogReverseAs
		Description for logging reverse code on uninstall mode
	.PARAMETER ResultVariable
		Name of a variable that contains the result.
	.PARAMETER PreventUninstall	
		Prevents script from executing 'Uninstall-SingleFile" on 'uninstall mode'
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Invoke-PowerShellCode -Script { Write-Host "Hello World" } -LogAs Invoke-HelloWorld -ResultVariable _return
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Invoke-PowerShellCode.html
	#>

	[CmdletBinding()]
	param(
		[Parameter(Mandatory=$true)][scriptblock]$Script,
		[Parameter(Mandatory=$false)][string]$LogAs = $null,
		[Parameter(Mandatory=$false)][scriptblock]$ReverseScript = $null,
		[Parameter(Mandatory=$false)][string]$LogReverseAs = $null,
		[Parameter(Mandatory=$false)][string]$ResultVariable = $null,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)

	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}

	process
	{
		try
		{
			$supportReverse = ($ReverseScript -ne $null);
			
			if (Test-SkipCommand -Context $Context -SupportReverse:$supportReverse -PreventUninstall:$PreventUninstall) { return; }

			function limit([string]$text, [int]$maxLength = 35) { return $(if ($text.Length -gt $maxLength) { ($text.Substring(0, ($maxLength - 3)) + "...") } else { $text }); }
			
			$code = $Script;
			$displayName = $LogAs;

			if (Test-ReverseMode)
			{
				$code = $ReverseScript;
				$displayName = $LogReverseAs;

				if (([string]$code).Trim("{}<#> ") -eq ".")
				{
					Write-Log -Message "Reverse script is [$($code)] -> using installation-script." -Source ${CmdletName};
					$code = $Script;
					if ([string]::IsNullOrEmpty($displayName)) { $displayName = $LogAs; }
				}
			}

			if ([string]::IsNullOrEmpty($displayName)) { $displayName = limit $code.ToString().Trim(); }
				
			Write-Log -Message "Invoking PowerShell code '$($displayName)'." -Source ${CmdletName};
			Write-Log -Message "Executing script block [ $($code.ToString().Trim()) ] with no parameters." -Source ${CmdletName};
			$result = . $code;

			if ($result -ne $null)
			{
				$displayResult = ([string](limit ($result | Out-String) 1027)).TrimEnd();
				Write-Log -Message "Result is: [ $($displayResult) ]." -Source ${CmdletName};
			}

			if (![string]::IsNullOrEmpty($ResultVariable)) { Set-PdVar -Name $ResultVariable -Value $result; }
		}
		catch
		{
			$failed = "Failed to invoke PowerShell code";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}

	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Connect-SmbShare
{
	<#
	.SYNOPSIS
		Connects an Smb Share (net use)
	.DESCRIPTION
		Connects a Smb Share with or without user credentials
	.PARAMETER ShareName
		The Ressource to access
	.PARAMETER Username
		Optional: Username for share access
	.PARAMETER Password
		Optional: Password for share access
	.PARAMETER Drive	
		The drive letter to assign
	.PARAMETER PreventUninstall	
		Prevents script from executing 'Uninstall-SingleFile" on 'uninstall mode'
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Connect-SmbShare -ShareName '\\myserver\\share$' -Drive 'X:' -Persistent
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Connect-SmbShare.html
	#>

	[CmdletBinding()]
	param(
		[Parameter(Mandatory=$true)][string]$ShareName,
		[Parameter(Mandatory=$false)][string]$Username = $null,
		[Parameter(Mandatory=$false)][string]$Password = $null,
		[Parameter(Mandatory=$false)][string]$Drive = $null,
		[Parameter(Mandatory=$false)][switch]$Persistent = $false,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)

	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		[hashtable]$displayParameters = $PSBoundParameters;
		@("Password") | where { $displayParameters.ContainsKey($_) } | % { $displayParameters[$_] = "-hidden-" }
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $displayParameters -Header
	}

	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -SupportReverse -PreventUninstall:$PreventUninstall) { return; }

			if (Test-ReverseMode)
			{
				Write-Log -Message "Reverse Mode: Calling Disconnect-SmbShare" -Source ${CmdletName};
				Disconnect-SmbShare -Drive $Drive
				return; 
			}

			$parameters = @{ RemotePath = $ShareName; Persistent = $Persistent; }

			if (-Not [String]::IsNullOrEmpty($Drive)) { 
				Write-Log -Message "Connecting SMB share '$($ShareName)' using '$($Drive)' as localpath." -Source ${CmdletName};

				if (Test-Path $Drive) {
					throw "Localpath '$($Drive)' already in use."
				}

				$parameters.LocalPath = $Drive;
			}
			else { Write-Log -Message "Connecting SMB share '$($ShareName)'." -Source ${CmdletName}; }

			$hasUsername = (-Not ([String]::IsNullOrEmpty($Username)));
			$hasPassword = (-Not ([String]::IsNullOrEmpty($Password)));

			if ($hasUsername) { Write-Log -Message "Using '$($Username)' as username to map share $($ShareName)" -Source ${CmdletName}; }
			elseif ($hasPassword) { Write-Log -Message "Password specified without username. Trying share-level security to connect to ressource '$($ShareName)'." -Source ${CmdletName}; }

			if ($hasUsername) { $parameters.Username = $Username; }
			if ($hasPassword) { $parameters.Password = (ConvertTo-PlainText $Password); }

			$connection = New-SmbMapping @parameters -ErrorAction Stop
		}
		catch
		{
			$failed = "Failed to connect SMB share";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}

	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}


function Disconnect-SmbShare
{
	<#
	.SYNOPSIS
		Disconnects an Smb Share (net use)
	.DESCRIPTION
		Disconnects a Smb Share
	.PARAMETER Drive	
		The drive letter to assign
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Disconnect-SmbShare -Drive 'X:'
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Disconnect-SmbShare.html
	#>

	[CmdletBinding()]
	param(
		[Parameter(Mandatory=$true)][string]$Drive = $null,
		[Parameter(Mandatory=$false)][switch]$SafeOnly = $true,
		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null,
		[Parameter(Mandatory=$false)][switch]$PreventUninstall = $false,
		[Parameter(Mandatory=$false)][Alias("X")][switch]$ContinueOnError = $false
	)

	begin
	{
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}

	process
	{
		try
		{
			if (Test-SkipCommand -Context $Context -PreventUninstall:$PreventUninstall) { return; }

			if ([String]::IsNullOrEmpty($Drive)) { throw "Missing drive parameter." }

			if (Test-Path $Drive) {
				Write-Log -Message "Disconnecting smb share on driveletter $($Drive)." -Source ${CmdletName};

				if ($SafeOnly -eq $false) {
					Write-Log -Message "Force parameter available. Forcing disconnecting even if files are open." -Source ${CmdletName};
					Remove-SmbMapping -LocalPath $Drive -Force -Confirm:$false -ErrorAction Stop
				}
				else {
					Remove-SmbMapping -LocalPath $Drive -Confirm:$false -ErrorAction Stop
				}
				
				if (Test-Path $Drive) {
					Write-Log -Message "Successfully disconnected smb share on driveletter $($Drive)" -Source ${CmdletName};
				}
			}
			else {
				Write-Log -Message "Driveletter: $($Drive) is not in use." -Source ${CmdletName};
			}
		}
		catch
		{
			$failed = "Failed to disconnect smb share";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}

	end
	{
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}


function Show-PdToastMessage 
{
	<#
	.SYNOPSIS
		Displays a Toast Notification
	.DESCRIPTION
		Displays a Toast Notification
	.PARAMETER Xml	
		The advanced Toast Notification content
	.PARAMETER Title
		The title of the Toast Notification
	.PARAMETER Message
		The Message to be displayed
	.PARAMETER Caption
		The caption of the Toast Notification
	.PARAMETER Button1
		The text of the first Button
	.PARAMETER Button2
		The text of the second Button
	.PARAMETER Button3
		The text of the third Button
	.PARAMETER ImagePlacement
		Defines the placement of the image being used. Possible values are: Hero, AppLogoOverride
	.PARAMETER Image
		The path to the image to be displayed
	.PARAMETER Textbox
		The placeholder text to be displayed in the textbox
	.PARAMETER Selection
		The choices in the dropdown to be displayed
	.PARAMETER ResultVariablePrefix
		The prefix for the result variable
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.EXAMPLE
		Show-PdToastMessage -Title 'Toast Notification' -ResultVariablePrefix result -Caption 'Toast Notification' -Message 'Hello World' -Textbox 'please enter text' -Button1 OK -Button2 Cancel
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Show-PdToastMessage.html
	#>

	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $false)][String[]]$Xml = $null,
		[Parameter(Mandatory = $false)][String]$Title = $null,
		[Parameter(Mandatory = $false)][String]$Message = $null,
		[Parameter(Mandatory = $false)][String]$Caption = $null,
		[Parameter(Mandatory = $false)][String]$Button1 = $null,
		[Parameter(Mandatory = $false)][String]$Button2 = $null,
		[Parameter(Mandatory = $false)][String]$Button3 = $null,
		[Parameter(Mandatory = $false)][ValidateSet("Hero", "AppLogoOverride")][String]$ImagePlacement = $null,
		[Parameter(Mandatory = $false)][String]$Image = $null,
		[Parameter(Mandatory = $false)][String]$Textbox = $null,
		[Parameter(Mandatory = $false)][String[]]$Selection = $null,
		[Parameter(Mandatory = $false)][String]$ResultVariablePrefix = $null,

		[Parameter(Mandatory=$false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
	)

	begin {
		## Get the name of this function and write header
		[String]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}

	process {

		if (Test-SkipCommand -Context $Context) { return; }

		function toAbsoluteUri([string]$path)
		{
			try 
			{
				if ([String]::IsNullOrEmpty($path)) { return $null; }

				$uri = [uri]$path;
				if ($uri.IsAbsoluteUri) { return $uri; }

				return [uri](Expand-Path $path);
			}
			catch 
			{
				return $null;
			}
		}

		function toTempFileName([string]$path)
		{
			return ([string]::Join("", @([System.Security.Cryptography.MD5]::Create().ComputeHash([System.Text.Encoding]::Unicode.GetBytes($path)) | % { $_.ToString("x2") })) + [System.IO.Path]::GetExtension($path));
		}

		function copyImageLocallyIfUnc([string]$path)
		{
			try {
				$uri = toAbsoluteUri $path;
				if ($uri -eq $null) { return $path; }

				$tmpName = toTempFileName $uri.AbsoluteUri;
				$tmpPath = "$($Env:TEMP)\PPB_$($tmpName)";

				if ($uri.IsUnc) {
					Copy-Item -Path $uri.LocalPath -Destination $tmpPath -Force;
					Write-Log -Message "Using temporary path '$($tmpPath)' for '$($uri.LocalPath)'. ($($path))" -Source ${CmdletName};
				}
				elseif ($uri.IsFile) {
					 return $uri.LocalPath;
				}
				elseif (($uri.Scheme -eq [uri]::UriSchemeHttps) -or ($uri.Scheme -eq [uri]::UriSchemeHttp)) {
					$void = Invoke-WebRequest -Uri $uri -OutFile $tmpPath;
					Write-Log -Message "Using temporary path '$($tmpPath)' for '$($uri.AbsoluteUri)'. ($($path))" -Source ${CmdletName};
				}
				else {
					return $path;
				}

				return $tmpPath;
			}
			catch {
				Write-Log -Message "Failed to make '$($path)' available: $($_.Exception.Message)" -Source ${CmdletName};
				return $path;
			}
		}

		function parseStringElement([System.Xml.XmlElement]$element, [string]$properties, [string[]]$unnamedPropertyNames, [string]$defaultProperties)
		{
			 if($element -eq $null){
				return $element;
			 }

			[string] $PartSeperator = "|";
			[string] $ValueSeperator = ":";

			if($properties -eq $null){
				$properties = [String]::Empty;
			}

			if(($unnamedPropertyNames -eq $null) -or ($unnamedPropertyNames.Length -eq 0)){
				$unnamedPropertyNames = [string[]]@( [String]::Empty );
			}

			if (![String]::IsNullOrEmpty($defaultProperties)){
				$properties = [String]::Concat($defaultProperties, $PartSeperator, $properties);
			}

            [string] $EscapedPartSeperator = "||";
            [string] $InterimPlaceholder = "\t";
			[string[]] $parts = @($properties.Replace($EscapedPartSeperator, $InterimPlaceholder).Split([string[]] @( $PartSeperator ), [StringSplitOptions]::RemoveEmptyEntries) | % { $_.Replace($InterimPlaceholder, $PartSeperator) });

			$unnamedPropertyValues = New-Object System.Collections.Generic.List[String];
            [string] $UnnamedPropertyPseudoName = "-";

			foreach ($part in $parts)
			{
				[int] $sep = $part.IndexOf($ValueSeperator);

				[string] $name = $(if ($sep -ge 0) { $part.Substring(0, $sep).Trim() } else { "" });
				[string] $value = $(if ($sep -ge 0) { $part.Substring($sep + 1).Trim() } else { $part.Trim() });

				if (($sep -lt 0) -or ($name -eq $UnnamedPropertyPseudoName))
				{
					$name = $unnamedPropertyNames[$unnamedPropertyValues.Count % $unnamedPropertyNames.Length];
					$unnamedPropertyValues.Add($value);
				}
                elseif ($name -ne [System.Xml.XmlConvert]::EncodeName($name))
                {
					$name = $unnamedPropertyNames[$unnamedPropertyValues.Count % $unnamedPropertyNames.Length];
                    $value = $part.Trim();
					$unnamedPropertyValues.Add($value);
                }

				if (![String]::IsNullOrEmpty($name)){
					$element.SetAttribute($name, $value);
				}
				else {
					$element.InnerText = $value;
				}
			}

			while (($unnamedPropertyValues.Count -gt 0) -and ($unnamedPropertyValues.Count -lt $unnamedPropertyNames.Length))
			{
				[string] $name = $unnamedPropertyNames[$unnamedPropertyValues.Count % $unnamedPropertyNames.Length];
				[string] $value = $unnamedPropertyValues[0];
				$unnamedPropertyValues.Add($value);
				if (![String]::IsNullOrEmpty($name)){
					$element.SetAttribute($name, $value);
				}
				else{
					$element.InnerText = $value;
				} 
			}

			return $element;
		}
		
		$xmlString = $(if ($Xml -is [Array]) { [String]::Join("", $Xml) } else { $null });

		if ([String]::IsNullOrEmpty($XmlString)){
			$toastXml = New-Object System.Xml.XmlDocument

			$toastElement = $toastXml.AppendChild($toastXml.CreateElement("toast"));

			$visualElement = $toastElement.AppendChild($toastXml.CreateElement("visual"));

			$actionsElement = $toastElement.AppendChild($toastXml.CreateElement("actions"));

			$bindingElement = $visualElement.AppendChild($toastXml.CreateElement("binding"));
			$bindingElement.SetAttribute("template", "ToastGeneric");

			$captionElement = $bindingElement.AppendChild($toastXml.CreateElement("text"));
			$captionElement.InnerText = $Caption;

			if (![String]::IsNullOrEmpty($Message))
			{
				$textElement = $bindingElement.AppendChild($toastXml.CreateElement("text"));
				$textElement.InnerText = $Message;
			}

			if (![String]::IsNullOrEmpty($Image))
			{
				$lookup = @{ "Hero" = "hero"; "AppLogoOverride" = "appLogoOverride"; }
				$value = $(if ($lookup.ContainsKey($ImagePlacement)) { $lookup[$ImagePlacement] } else { $ImagePlacement });

				$newImage = copyImageLocallyIfUnc $Image;
				$imageElement = $bindingElement.AppendChild($toastXml.CreateElement("image"));
				$imageElement.SetAttribute("src", $newImage);
				$imageElement.SetAttribute("placement", $value);
			}

			if (![String]::IsNullOrEmpty($Textbox))
			{
				$textboxElement = $actionsElement.AppendChild($toastXml.CreateElement("input"));
				$textboxElement.SetAttribute("id", "textbox");
				$textboxElement.SetAttribute("type", "text");
				$textboxElement.SetAttribute("placeHolderContent", $Textbox);
			}
			if (($Selection -is [Array]) -and ($Selection.Length -gt 0))
			{
				$selectionId = "selection"
				$defaultInput = $null
				$selectionInputElement = $actionsElement.AppendChild($toastXml.CreateElement("input"));
				$selectionInputElement.SetAttribute("id", $selectionId)
				$selectionInputElement.SetAttribute("type", "selection")

				for ($i = 0; $i -lt $Selection.Length; $i++)
				{
					$selectionElement = $selectionInputElement.AppendChild($toastXml.CreateElement("selection"));
					$selectionElement = parseStringElement -element $selectionElement -properties $Selection[$i] -unnamedPropertyNames @("id","content") -defaultProperties ""

				}
				if ([String]::IsNullOrEmpty($defaultInput)){
					$defaultInput = $selectionInputElement.FirstChild.Attributes["id"].Value;
				}

				$selectionInputElement.SetAttribute("defaultInput", $defaultInput);
			}

			if (![String]::IsNullOrEmpty($Button1))
			{
				$button1Element = $actionsElement.AppendChild($toastXml.CreateElement("action"));
				$button1Element = parseStringElement -element $button1Element -properties $Button1 -unnamedPropertyNames @( "content", "arguments" ) -defaultProperties "activationType:background"
			}

			if (![String]::IsNullOrEmpty($Button2))
			{
				$button2Element = $actionsElement.AppendChild($toastXml.CreateElement("action"));
				$button2Element = parseStringElement -element $button2Element -properties $Button2 -unnamedPropertyNames @( "content", "arguments" ) -defaultProperties "activationType:background"
			}

			if (![String]::IsNullOrEmpty($Button3))
			{
				$button3Element = $actionsElement.AppendChild($toastXml.CreateElement("action"));
				$button3Element = parseStringElement -element $button3Element -properties $Button3 -unnamedPropertyNames @( "content", "arguments" ) -defaultProperties "activationType:background"
			}

			$XmlString = $toastXml.OuterXml
		}

		$appID = "PSPD-Toast";

		$toastAppKeyPath = "Registry::HKEY_CURRENT_USER\Software\Classes\AppUserModelId\$($appID)";
		$key = $(if (![String]::IsNullOrEmpty($Title)) { Get-PdRegistryKey -Path $toastAppKeyPath -Writable -Create -AcceptNull } else { $null });
		if ($key -ne $null)
		{
			$key.SetValue("DisplayName", $Title);

			$iconUri = $null;
			$packageRoot = (Get-PdContext).PackageDirectory;
			$list = @("SupportFiles\Package.png","SupportFiles\Package.ico","SupportFiles\Package.jpg","SupportFiles\Package.jpeg","SupportFiles\Package.gif","AppDeployToolkit\AppDeployToolkitLogo.png","AppDeployToolkit\AppDeployToolkitLogo.ico");
			foreach ($icon in $list){ if ([System.IO.File]::Exists([System.IO.Path]::Combine($packageRoot, $icon))) { $iconUri = [System.IO.Path]::Combine($packageRoot, $icon); break; } }
			$key.SetValue("IconUri", $iconUri);
		}
		else
		{
			if (![String]::IsNullOrEmpty($Title)) { Write-Log -Message "Cannot set the specified title [$($Title)]." -Source ${CmdletName} }
			$appID = "Microsoft.Windows.Computer"; # title => My Computer (Dieser PC)
		}

		$xmlDoc = New-Object System.Xml.XmlDocument;
		$xmlDoc.LoadXml($XmlString);
		$imageElements = $xmlDoc.DocumentElement.SelectNodes("visual/binding//image");

		foreach ($img in $imageElements)
		{
			$imagePath = $img.GetAttribute("src");
			$imageSource = copyImageLocallyIfUnc $imagePath;
			$img.SetAttribute("src", $imageSource);
		}

		$xmlMessageText = "<...>";
		if ($configToolkitLogDebugMessage) { # PSADT
			$writer = New-Object System.IO.StringWriter;
			$xmlDoc.Save($writer);
			$xmlMessageText = "`"$($writer.ToString())`"";
			$writer.Close();
		}
		Write-Log -Message "Calling ShowMessage\Show-ToastMessage -Wait -ActivationResultVariable _ActivationResult -AppID $($appID) -ReturnEventArgs -Xml $($xmlMessageText)." -Source ${CmdletName};
		
		$void = Show-ToastMessage -Wait -Xml $xmlDoc.OuterXml -ActivationResultVariable _ActivationResult -AppID $appID -ReturnEventArgs 

		if (![string]::IsNullOrEmpty($ResultVariablePrefix)) 
		{ 

			if (![string]::IsNullOrEmpty($_ActivationResult.Arguments))
			{
				Set-PdVar -Name ($ResultVariablePrefix + ".Button") -Value $_ActivationResult.Arguments;
			}

			if ($_ActivationResult.UserInput -ne $null)
			{
				if (![string]::IsNullOrEmpty($_ActivationResult.UserInput["textbox"]))
				{
					Set-PdVar -Name ($ResultVariablePrefix + ".Text") -Value $_ActivationResult.UserInput["textbox"];
				}

				if (![string]::IsNullOrEmpty($_ActivationResult.UserInput["selection"]))
				{
					Set-PdVar -Name ($ResultVariablePrefix + ".SelectedItem") -Value $_ActivationResult.UserInput["selection"];
				}
			}

			Set-PdVar -Name ($ResultVariablePrefix + ".Raw") -Value $_ActivationResult; 
		}
	}

	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Expand-PdArchive {
	<#
	.SYNOPSIS
		Extracts files from a specified archive (zipped) file.
	.DESCRIPTION
		The `Expand-PdArchive` cmdlet extracts files from a specified zipped archive file to a specified destination folder.
		An archive file allows multiple files to be packaged, and optionally compressed, into a single zipped file for
		easier distribution and storage.
	.PARAMETER Path
		Specifies the path to the archive file.
	.PARAMETER DestinationPath
		By default, Expand-PdArchive creates a folder in the current location that's the same name as the ZIP file. The parameter allows you to specify the path to a different folder. The target folder is created if it doesn't exist.
	.PARAMETER Force
		Use this parameter to overwrite existing files. By default, Expand-PdArchive doesn't overwrite.
	.PARAMETER Confirm
		Prompts you for confirmation before running the cmdlet.
	.PARAMETER WhatIf
		Shows what would happen if the cmdlet runs. The cmdlet isn't run.
	.PARAMETER Context
		Defines the context to be used for the script execution. Possible values are: Any, User, UserPerService, Computer, ComputerPerService
	.PARAMETER ContinueOnError
		If switch is used. The script wont abort if an error occurs
	.EXAMPLE
		Expand-PdArchive -Path Draftv2.zip -DestinationPath C:\Reference
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Expand-PdArchive.html
	#>
	[CmdletBinding(DefaultParametersetName = "Path")]
	param (
		[Parameter(Mandatory = $true, ParameterSetName = "Path")][string]$Path,
		[Parameter(Mandatory = $true, ParameterSetName = "LiteralPath")][string]$LiteralPath,
		[Parameter(Mandatory = $false)][string]$DestinationPath = $null,
		[Parameter(Mandatory = $false)][switch]$Force = $false,
		[Parameter(Mandatory = $false)][switch]$Confirm = $false,
		[Parameter(Mandatory = $false)][switch]$WhatIf = $false,

		[Parameter(Mandatory = $false)][Alias("X")][switch]$ContinueOnError = $false,
		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
	)

	begin {
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}
		
	process {
		if (Test-SkipCommand -Context $Context) { return; }

		$pdParameterNames = @( "Context", "ContinueOnError" );
		$parameters = @{};
		$PSBoundParameters.GetEnumerator() | where { $pdParameterNames -notcontains $_.Key } | % { $parameters[$_.Key] = $_.Value; }

		try {
			if ($PsCmdlet.ParameterSetName -eq 'LiteralPath') { Write-Log -Message "Archive: '$($LiteralPath)' (literal)" -Source ${CmdletName}; }
			else { Write-Log -Message "Archive: '$($Path)'" -Source ${CmdletName}; }

			if ($PSBoundParameters.ContainsKey('DestinationPath')) { Write-Log -Message "Destination path: '$($DestinationPath)'" -Source ${CmdletName}; }
			else { Write-Log -Message "Destination path: (default destination path)" -Source ${CmdletName}; }

			Expand-Archive @parameters;
		}
		catch {
			$failed = "Expand-Archive failed";
			Write-Log -Message "$($failed): $($_.Exception.Message)"  -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}
	end {
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
    }
}

function Test-UserGroupMembership {
	<#
	.SYNOPSIS
		Tests whether the current user is in the defined group.
	.DESCRIPTION
		Tests whether the current user is in the defined group.
	.PARAMETER Group
		Specifies the group in which the current user should be.
	.RETURNS
		Returns true if the current user is the defined group, else it will return false.
	.EXAMPLE
		Test-UserGroupMembership -Group "Users"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-UserGroupMembership.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $true)][string]$Group
	)

	begin {
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}

	process {
		try
		{
			$result = $false;
			$sid = $null;
			try
			{
				$sid = Get-LocalAccountSid -identifier $Group;
			}
			catch
			{
				$failed = "Failed to get the SID of '$Group' - returning $result.";
				Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
				return $result;
			}
			$currentUser = [System.Security.Principal.WindowsIdentity]::GetCurrent();
			# When testing for newly created role information, such as a new user or a new group,
			# it is important to log out and log in to force the propagation of role information within the domain.
			# Not doing so can cause the IsInRole test to return false. [https://learn.microsoft.com/dotnet/api/system.security.principal.windowsprincipal.isinrole]
			$result = (New-Object System.Security.Principal.WindowsPrincipal($currentUser)).IsInRole($sid);
			Write-Log -Message "Checking if $($currentUser.Name) is part of $($Group) - returning $result" -Source ${CmdletName};
			return $result;
		}
		catch
		{
			$failed = "Failed to check membership of the current user for '$Group'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}

	end {
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
    }
}

function Test-WorkstationGroupMembership {
	<#
	.SYNOPSIS
		Tests whether the current workstation is in the defined group.
	.DESCRIPTION
		Tests whether the current workstation is in the defined group.
	.PARAMETER Group
		Specifies the group in which the current workstation should be.
	.RETURNS
		Returns true if the current workstation is the defined group, else it will return false.
	.EXAMPLE
		Test-WorkstationGroupMembership -Group "Workstations"
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Test-WorkstationGroupMembership.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $true)][string]$Group
	)

	begin {
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}

	process {
		try 
		{
			$result = $false;
			$sid = $null;
			try 
			{
				$sid = Get-LocalAccountSid -identifier $Group;
			}
			catch
			{
				$failed = "Failed to get the SID of '$Group' - returning $result.";
				Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
				return $result;
			}
			if($IsMachinePartOfDomain){
				# When testing for newly created role information, such as a new user or a new group,
				# it is important to log out and log in to force the propagation of role information within the domain.
				# Not doing so can cause the IsInRole test to return false. [https://learn.microsoft.com/dotnet/api/system.security.principal.windowsprincipal.isinrole]
				$result = (New-Object System.Security.Principal.WindowsPrincipal("$($Env:COMPUTERNAME)`$")).IsInRole($sid);
				Write-Log -Message "Checking if $($Env:COMPUTERNAME)`$ is part of $($Group) - returning $result" -Source ${CmdletName};
			}
			else {
				Write-Log -Message "Workstation is not part of a domain - returning $result" -Source ${CmdletName};
			}
			return $result;
		}
		catch
		{
			$failed = "Failed to check membership of the current computer for '$Group'";
			Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 3 -Source ${CmdletName};
			If (!$ContinueOnError) { throw "$($failed): $($_.Exception.Message)" }
		}
	}

	end {
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
    }
}


#region PSADT wrapper functions 

Function Read-ADTInstalledApplication {
    <#
	.SYNOPSIS
		Retrieves information about installed applications.
	.DESCRIPTION
		Retrieves information about installed applications by querying the registry. You can specify an application name, a product code, or both.
		Returns information about application publisher, name & version, product code, uninstall string, install source, location, date, and application architecture.
	.PARAMETER Name
		The name of the application to retrieve information for. Performs a contains match on the application display name by default.
	.PARAMETER Exact
		Specifies that the named application must be matched using the exact name.
	.PARAMETER WildCard
		Specifies that the named application must be matched using a wildcard search.
	.PARAMETER RegEx
		Specifies that the named application must be matched using a regular expression search.
	.PARAMETER ProductCode
		The product code of the application to retrieve information for.
	.PARAMETER IncludeUpdatesAndHotfixes
		Include matches against updates and hotfixes in results.
	.INPUTS
		None
		You cannot pipe objects to this function.
	.OUTPUTS
		PSObject
			Returns a PSObject with information about an installed application
			- Publisher
			- DisplayName
			- DisplayVersion
			- ProductCode
			- UninstallString
			- InstallSource
			- InstallLocation
			- InstallDate
			- Architecture
	.EXAMPLE
		Get-InstalledApplication -Name 'Adobe Flash'
	.EXAMPLE
		Get-InstalledApplication -ProductCode '{1AD147D0-BE0E-3D6C-AC11-64F6DC4163F1}'
	.Outputs
		For every detected matching Application the Function puts out a custom Object containing the following Properties:
		DisplayName, DisplayVersion, InstallDate, Publisher, Is64BitApplication, ProductCode, InstallLocation, UninstallSubkey, UninstallString, InstallSource.
	.NOTES
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Read-ADTInstalledApplication.html
	#>
    [CmdletBinding()]
    param (
        [Parameter(Mandatory = $false)]
        [ValidateNotNullorEmpty()]
        [String[]]$Name,
        [Parameter(Mandatory = $false)]
        [Switch]$Exact = $false,
        [Parameter(Mandatory = $false)]
        [Switch]$WildCard = $false,
        [Parameter(Mandatory = $false)]
        [Switch]$RegEx = $false,
        [Parameter(Mandatory = $false)]
        [ValidateNotNullorEmpty()]
        [String]$ProductCode,
        [Parameter(Mandatory = $false)]
        [Switch]$IncludeUpdatesAndHotfixes,

		[Parameter(Mandatory = $false)][string]$ResultVariablePrefix = "",
		[Parameter(Mandatory = $false)][switch]$PassThru = $false,
		[Parameter(Mandatory = $false)][string[]]$ReturnProperties = @(),

		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
    )

    begin {
        ## Get the name of this function and write header
        [String]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
    }

    process {

		$pdParameterNames = @( "ResultVariablePrefix", "PassThru", "ReturnProperties", "Context" );
		$adtParameters = @{};
		$PSBoundParameters.GetEnumerator() | where { $pdParameterNames -notcontains $_.Key } | % { $adtParameters[$_.Key] = $_.Value; }
		
		$list = @(Get-InstalledApplication @adtParameters);
		Write-Log -Message "Matching applications found: $($list.Count). $([string]::Join(', ', @($list | % { $_.DisplayName })))" -Source ${CmdletName};
		
		$app = $(if ($list.Count -gt 0) { $list[0] } else { $null });
		if ($app -ne $null) { Write-Log -Message "first item: name '$($app.DisplayName)', version '$($app.DisplayVersion)'." -Source ${CmdletName}; }
		
		if (![string]::IsNullOrEmpty($ResultVariablePrefix))
		{
			$properties = @($ReturnProperties | % { $_.Trim() });
			foreach ($property in $properties) {
				$value = $(if ($app -ne $null) { $app.$property } else { $null });
				if ($property -eq "DisplayVersion") {
					[version]$version = "0.0";
					if ([version]::TryParse($value, [ref]$version)) { $value = $version }
				}
				elseif ($property -eq "Count") {
					$value = $list.Count;
				}

				Set-PdVar -Name "$($ResultVariablePrefix).$($property)" -Value $value;
			}
		}
		
		if ($PassThru) { return $list; }
    }

    end {
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
    }
}

Function Get-ADTInstalledApplicationProperty {
    <#
	.SYNOPSIS
		Retrieves information about installed applications.
	.DESCRIPTION
		Retrieves information about installed applications by querying the registry. You can specify an application name, a product code, or both.
		Returns information about application publisher, name & version, product code, uninstall string, install source, location, date, and application architecture.
	.PARAMETER Name
		The name of the application to retrieve information for. Performs a contains match on the application display name by default.
	.PARAMETER Exact
		Specifies that the named application must be matched using the exact name.
	.PARAMETER WildCard
		Specifies that the named application must be matched using a wildcard search.
	.PARAMETER RegEx
		Specifies that the named application must be matched using a regular expression search.
	.PARAMETER ProductCode
		The product code of the application to retrieve information for.
	.PARAMETER IncludeUpdatesAndHotfixes
		Include matches against updates and hotfixes in results.
	.INPUTS
		None
		You cannot pipe objects to this function.
	.OUTPUTS
		PSObject
			Returns a PSObject with information about an installed application
			- Publisher
			- DisplayName
			- DisplayVersion
			- ProductCode
			- UninstallString
			- InstallSource
			- InstallLocation
			- InstallDate
			- Architecture
	.EXAMPLE
		Get-InstalledApplication -Name 'Adobe Flash'
	.EXAMPLE
		Get-InstalledApplication -ProductCode '{1AD147D0-BE0E-3D6C-AC11-64F6DC4163F1}'
	.Outputs
		For every detected matching Application the Function puts out a custom Object containing the following Properties:
		DisplayName, DisplayVersion, InstallDate, Publisher, Is64BitApplication, ProductCode, InstallLocation, UninstallSubkey, UninstallString, InstallSource.
	.NOTES
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Get-ADTInstalledApplicationProperty.html
	#>
    [CmdletBinding()]
    param (
        [Parameter(Mandatory = $false)]
        [ValidateNotNullorEmpty()]
        [String[]]$Name,
        [Parameter(Mandatory = $false)]
        [Switch]$Exact = $false,
        [Parameter(Mandatory = $false)]
        [Switch]$WildCard = $false,
        [Parameter(Mandatory = $false)]
        [Switch]$RegEx = $false,
        [Parameter(Mandatory = $false)]
        [ValidateNotNullorEmpty()]
        [String]$ProductCode,
        [Parameter(Mandatory = $false)]
        [Switch]$IncludeUpdatesAndHotfixes,

		[Parameter(Mandatory = $false)][string]$Property = "DisplayName",

		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
    )

    begin {
        ## Get the name of this function and write header
        [String]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
    }

    process {

		$pdParameterNames = @( "Property" );
		$adtParameters = @{};
		$PSBoundParameters.GetEnumerator() | where { $pdParameterNames -notcontains $_.Key } | % { $adtParameters[$_.Key] = $_.Value; }

		$list = @(Get-InstalledApplication @adtParameters);
		# [version]$version = "0.0"; $list = @($list | sort -Descending { if ([version]::TryParse($_.DisplayVersion, [ref]$version)) { $version; } else { $_.DisplayVersion } });
		Write-Log -Message "Matching applications found: $($list.Count). $([string]::Join(', ', @($list | % { $_.DisplayName })))" -Source ${CmdletName};

		$returnValue = $(if ($list.Count -eq 0) { $null } elseif ([string]::IsNullOrEmpty($Property)) { $list[0].DisplayName } else { $list[0].$Property });

		if ($Property -eq "Count") { $returnValue = $list.Count; }
		elseif ($Property -eq "DisplayVersion") { [version]$version = "0.0"; if ([version]::TryParse($returnValue, [ref]$version)) { $returnValue = $version; } }

		Write-Log -Message "Returning '$($Property)': '$($returnValue)'." -Source ${CmdletName};
		return $returnValue;
	}

    end {
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
    }
}

function Show-ADTInstallationWelcome {
    <#
	.SYNOPSIS
		Show a welcome dialog prompting the user with information about the installation and actions to be performed before the installation can begin.
	.DESCRIPTION
		The following prompts can be included in the welcome dialog:
			a) Close the specified running applications, or optionally close the applications without showing a prompt (using the -Silent switch).
			b) Defer the installation a certain number of times, for a certain number of days or until a deadline is reached.
			c) Countdown until applications are automatically closed.
			d) Prevent users from launching the specified applications while the installation is in progress.
		Notes:
			The process descriptions are retrieved from WMI, with a fall back on the process name if no description is available. Alternatively, you can specify the description yourself with a '=' symbol - see examples.
			The dialog box will timeout after the timeout specified in the XML configuration file (default 1 hour and 55 minutes) to prevent SCCM installations from timing out and returning a failure code to SCCM. When the dialog times out, the script will exit and return a 1618 code (SCCM fast retry code).
	.PARAMETER CloseApps
		Name of the process to stop (do not include the .exe). Specify multiple processes separated by a comma. Specify custom descriptions like this: "winword=Microsoft Office Word,excel=Microsoft Office Excel"
	.PARAMETER Silent
		Stop processes without prompting the user.
	.PARAMETER CloseAppsCountdown
		Option to provide a countdown in seconds until the specified applications are automatically closed. This only takes effect if deferral is not allowed or has expired.
	.PARAMETER ForceCloseAppsCountdown
		Option to provide a countdown in seconds until the specified applications are automatically closed regardless of whether deferral is allowed.
	.PARAMETER PromptToSave
		Specify whether to prompt to save working documents when the user chooses to close applications by selecting the "Close Programs" button. Option does not work in SYSTEM context unless toolkit launched with "psexec.exe -s -i" to run it as an interactive process under the SYSTEM account.
	.PARAMETER PersistPrompt
		Specify whether to make the Show-InstallationWelcome prompt persist in the center of the screen every couple of seconds, specified in the AppDeployToolkitConfig.xml. The user will have no option but to respond to the prompt. This only takes effect if deferral is not allowed or has expired.
	.PARAMETER BlockExecution
		Option to prevent the user from launching processes/applications, specified in -CloseApps, during the installation.
	.PARAMETER AllowDefer
		Enables an optional defer button to allow the user to defer the installation.
	.PARAMETER AllowDeferCloseApps
		Enables an optional defer button to allow the user to defer the installation only if there are running applications that need to be closed. This parameter automatically enables -AllowDefer
	.PARAMETER DeferTimes
		Specify the number of times the installation can be deferred.
	.PARAMETER DeferDays
		Specify the number of days since first run that the installation can be deferred. This is converted to a deadline.
	.PARAMETER DeferDeadline
		Specify the deadline date until which the installation can be deferred.
		Specify the date in the local culture if the script is intended for that same culture.
		If the script is intended to run on EN-US machines, specify the date in the format: "08/25/2013" or "08-25-2013" or "08-25-2013 18:00:00"
		If the script is intended for multiple cultures, specify the date in the universal sortable date/time format: "2013-08-22 11:51:52Z"
		The deadline date will be displayed to the user in the format of their culture.
	.PARAMETER CheckDiskSpace
		Specify whether to check if there is enough disk space for the installation to proceed.
		If this parameter is specified without the RequiredDiskSpace parameter, the required disk space is calculated automatically based on the size of the script source and associated files.
	.PARAMETER RequiredDiskSpace
		Specify required disk space in MB, used in combination with CheckDiskSpace.
	.PARAMETER MinimizeWindows
		Specifies whether to minimize other windows when displaying prompt. Default: $true.
	.PARAMETER TopMost
		Specifies whether the windows is the topmost window. Default: $true.
	.PARAMETER ForceCountdown
		Specify a countdown to display before automatically proceeding with the installation when a deferral is enabled.
	.PARAMETER CustomText
		Specify whether to display a custom message specified in the XML file. Custom message must be populated for each language section in the XML.
	.INPUTS
		None
		You cannot pipe objects to this function.
	.OUTPUTS
		None
		This function does not return objects.
	.EXAMPLE
		Show-ADTInstallationWelcome -CloseApps 'iexplore,winword,excel'
		Prompt the user to close Internet Explorer, Word and Excel.
	.EXAMPLE
		Show-ADTInstallationWelcome -CloseApps 'winword,excel' -Silent
		Close Word and Excel without prompting the user.
	.EXAMPLE
		Show-ADTInstallationWelcome -CloseApps 'winword,excel' -BlockExecution
		Close Word and Excel and prevent the user from launching the applications while the installation is in progress.
	.EXAMPLE
		Show-ADTInstallationWelcome -CloseApps 'winword=Microsoft Office Word,excel=Microsoft Office Excel' -CloseAppsCountdown 600
		Prompt the user to close Word and Excel, with customized descriptions for the applications and automatically close the applications after 10 minutes.
	.EXAMPLE
		Show-ADTInstallationWelcome -CloseApps 'winword,msaccess,excel' -PersistPrompt
		Prompt the user to close Word, MSAccess and Excel.
		By using the PersistPrompt switch, the dialog will return to the center of the screen every couple of seconds, specified in the AppDeployToolkitConfig.xml, so the user cannot ignore it by dragging it aside.
	.EXAMPLE
		Show-ADTInstallationWelcome -AllowDefer -DeferDeadline '25/08/2013'
		Allow the user to defer the installation until the deadline is reached.
	.EXAMPLE
		Show-ADTInstallationWelcome -CloseApps 'winword,excel' -BlockExecution -AllowDefer -DeferTimes 10 -DeferDeadline '25/08/2013' -CloseAppsCountdown 600
		Close Word and Excel and prevent the user from launching the applications while the installation is in progress.
		Allow the user to defer the installation a maximum of 10 times or until the deadline is reached, whichever happens first.
		When deferral expires, prompt the user to close the applications and automatically close them after 10 minutes.
	.NOTES
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Show-ADTInstallationWelcome.html
	#>
    [CmdletBinding(DefaultParametersetName = 'None')]
    param (
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][String]$CloseApps,
        [Parameter(Mandatory = $false)][Switch]$Silent = $false,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$CloseAppsCountdown = 0,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$ForceCloseAppsCountdown = 0,
        [Parameter(Mandatory = $false)][Switch]$PromptToSave = $false,
        [Parameter(Mandatory = $false)][Switch]$PersistPrompt = $false,
        [Parameter(Mandatory = $false)][Switch]$BlockExecution = $false,
        [Parameter(Mandatory = $false)][Switch]$AllowDefer = $false,
        [Parameter(Mandatory = $false)][Switch]$AllowDeferCloseApps = $false,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$DeferTimes = 0,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$DeferDays = 0,
        [Parameter(Mandatory = $false)][String]$DeferDeadline = '',
        [Parameter(ParameterSetName = 'CheckDiskSpaceParameterSet', Mandatory = $true)][ValidateScript({ $_.IsPresent -eq ($true -or $false) })][Switch]$CheckDiskSpace,
        [Parameter(ParameterSetName = 'CheckDiskSpaceParameterSet', Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$RequiredDiskSpace = 0,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Boolean]$MinimizeWindows = $true,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Boolean]$TopMost = $true,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$ForceCountdown = 0,
        [Parameter(Mandatory = $false)][Switch]$CustomText = $false,

		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
    )

    begin {
        ## Get the name of this function and write header
        [String]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
    }

    process {
		if (Test-SkipCommand -Context $Context) { return; }

		$pdParameterNames = @( "Context" );
		$adtParameters = @{};
		$PSBoundParameters.GetEnumerator() | where { $pdParameterNames -notcontains $_.Key } | % { $adtParameters[$_.Key] = $_.Value; }

		if (![string]::IsNullOrEmpty($adtParameters.CloseApps))
		{
			# normalize -CloseApps (remove blanks): <process name>[=<process description>][,<process name>[=<process description>]]...
			$adtParameters.CloseApps = [string]::Join(",", @($adtParameters.CloseApps.Split(",") | where { ![string]::IsNullOrWhitespace($_) } | % { [string]::Join("=", @($_.Split("=") | where { ![string]::IsNullOrWhitespace($_) } | % { $_.Trim() })); }));
		}

		# Show-InstallationWelcome -CloseApps $CloseApps -CloseAppsCountdown $CloseAppsCountdown -DeferTimes $DeferTimes -DeferDays $DeferDays -DeferDeadline $DeferDeadline -Silent:$Silent -PromptToSave:$PromptToSave -BlockExecution:$BlockExecution -AllowDefer:$AllowDefer -AllowDeferCloseApps:$AllowDeferCloseApps -CheckDiskSpace:$CheckDiskSpace -RequiredDiskSpace $RequiredDiskSpace -MinimizeWindows $MinimizeWindows -TopMost $TopMost -CustomText:$CustomText  -ForceCloseAppsCountdown $ForceCloseAppsCountdown -ForceCountdown $ForceCountdown -PersistPrompt:$PersistPrompt
		Show-InstallationWelcome @adtParameters;
	}

	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Show-ADTBalloonTip{
	<#
	.SYNOPSIS
		Displays a balloon tip notification in the system tray.
	.DESCRIPTION
		Displays a balloon tip notification in the system tray.
	.PARAMETER BalloonTipText
		Text of the balloon tip.
	.PARAMETER BalloonTipTitle
		Title of the balloon tip.
	.PARAMETER BalloonTipIcon
		Icon to be used. Options: 'Error', 'Info', 'None', 'Warning'. Default is: Info.
	.PARAMETER BalloonTipTime
		Time in milliseconds to display the balloon tip. Default: 10000.
	.PARAMETER NoWait
		Create the balloontip asynchronously. Default: $false
	.EXAMPLE
		Show-ADTBalloonTip -BalloonTipText 'Installation Started' -BalloonTipTitle 'Application Name'
	.EXAMPLE
		Show-ADTBalloonTip -BalloonTipIcon 'Info' -BalloonTipText 'Installation Started' -BalloonTipTitle 'Application Name' -BalloonTipTime 1000
	.NOTES
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Show-ADTBalloonTip.html
	#>
	[CmdletBinding()]
	param (
		[Parameter(Mandatory = $true,Position=0)][ValidateNotNullOrEmpty()][string]$BalloonTipText,
		[Parameter(Mandatory = $false,Position=1)][ValidateNotNullorEmpty()][string]$BalloonTipTitle = $installTitle,
		[Parameter(Mandatory = $false,Position=2)][ValidateSet('Error','Info','None','Warning')][Windows.Forms.ToolTipIcon]$BalloonTipIcon = 'Info',
		[Parameter(Mandatory = $false,Position=3)][ValidateNotNullorEmpty()][int32]$BalloonTipTime = 10000,
		[Parameter(Mandatory = $false,Position=4)][switch]$NoWait = $false,

		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
	)

	begin {
		## Get the name of this function and write header
		[string]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
	}

	process {
		if (Test-SkipCommand -Context $Context) { return; }

		$pdParameterNames = @( "Context" );
		$adtParameters = @{};
		$PSBoundParameters.GetEnumerator() | where { $pdParameterNames -notcontains $_.Key } | % { $adtParameters[$_.Key] = $_.Value; }

		# Show-BalloonTip -BalloonTipText $BalloonTipText -BalloonTipTitle $BalloonTipTitle -BalloonTipIcon $BalloonTipIcon -BalloonTipTime $BalloonTipTime -NoWait:$NoWait
		Show-BalloonTip @adtParameters;
	}

	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}

}

function Show-ADTInstallationRestartPrompt {
    <#
	.SYNOPSIS
		Displays a restart prompt with a countdown to a forced restart.
	.DESCRIPTION
		Displays a restart prompt with a countdown to a forced restart.
	.PARAMETER CountdownSeconds
		Specifies the number of seconds to countdown before the system restart. Default: 60
	.PARAMETER CountdownNoHideSeconds
		Specifies the number of seconds to display the restart prompt without allowing the window to be hidden. Default: 30
	.PARAMETER NoSilentRestart
		Specifies whether the restart should be triggered when Deploy mode is silent or very silent. Default: $true
	.PARAMETER NoCountdown
		Specifies not to show a countdown.
		The UI will restore/reposition itself persistently based on the interval value specified in the config file.
	.PARAMETER SilentCountdownSeconds
		Specifies number of seconds to countdown for the restart when the toolkit is running in silent mode and NoSilentRestart is $false. Default: 5
	.PARAMETER TopMost
		Specifies whether the windows is the topmost window. Default: $true.
	.INPUTS
		None
		You cannot pipe objects to this function.
	.OUTPUTS
		System.String
		Returns the version of the specified file.
	.EXAMPLE
		Show-ADTInstallationRestartPrompt -Countdownseconds 600 -CountdownNoHideSeconds 60
	.EXAMPLE
		Show-ADTInstallationRestartPrompt -NoCountdown
	.EXAMPLE
		Show-ADTInstallationRestartPrompt -Countdownseconds 300 -NoSilentRestart $false -SilentCountdownSeconds 10
	.NOTES
		Be mindful of the countdown you specify for the reboot as code directly after this function might NOT be able to execute - that includes logging.
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Show-ADTInstallationRestartPrompt.html
	#>
    [CmdletBinding()]
    param (
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$CountdownSeconds = 60,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$CountdownNoHideSeconds = 30,
        [Parameter(Mandatory = $false)][Boolean]$NoSilentRestart = $true, 
		[Parameter(Mandatory = $false)][Switch]$NoCountdown = $false,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$SilentCountdownSeconds = 5,
		[Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Boolean]$TopMost = $true,

		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
    )

    begin {
        ## Get the name of this function and write header
        [String]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
    }

    process {
		if (Test-SkipCommand -Context $Context) { return; }

		$pdParameterNames = @( "Context" );
		$adtParameters = @{};
		$PSBoundParameters.GetEnumerator() | where { $pdParameterNames -notcontains $_.Key } | % { $adtParameters[$_.Key] = $_.Value; }
     
		# Show-InstallationRestartPrompt -CountdownSeconds $CountdownSeconds -CountdownNoHideSeconds $CountdownNoHideSeconds -NoSilentRestart $NoSilentRestart -NoCountdown:$NoCountdown -SilentCountdownSeconds $SilentCountdownSeconds -TopMost $TopMost
		Show-InstallationRestartPrompt @adtParameters;
	}

	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Show-ADTInstallationProgress {
    <#
	.SYNOPSIS
		Displays a progress dialog in a separate thread with an updateable custom message.
	.DESCRIPTION
		Create a WPF window in a separate thread to display a marquee style progress ellipse with a custom message that can be updated.
		The status message supports line breaks.
		The first time this function is called in a script, it will display a balloon tip notification to indicate that the installation has started (provided balloon tips are enabled in the configuration).
	.PARAMETER StatusMessage
		The status message to be displayed. The default status message is taken from the XML configuration file.
	.PARAMETER WindowLocation
		The location of the progress window. Default: center of the screen.
	.PARAMETER TopMost
		Specifies whether the progress window should be topmost. Default: $true.
	.INPUTS
		None
		You cannot pipe objects to this function.
	.OUTPUTS
		None
		This function does not generate any output.
	.EXAMPLE
		Show-ADTInstallationProgress
		Uses the default status message from the XML configuration file.
	.EXAMPLE
		Show-ADTInstallationProgress -StatusMessage 'Installation in Progress...'
	.EXAMPLE
		Show-ADTInstallationProgress -StatusMessage "Installation in Progress...`r`nThe installation may take 20 minutes to complete."
	.EXAMPLE
		Show-ADTInstallationProgress -StatusMessage 'Installation in Progress...' -WindowLocation 'BottomRight' -TopMost $false
	.NOTES
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Show-ADTInstallationProgress.html
	#>
    [CmdletBinding()]
    param (
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][String]$StatusMessage = $configProgressMessageInstall,
        [Parameter(Mandatory = $false)][ValidateSet('Default', 'BottomRight', 'TopCenter')][String]$WindowLocation = 'Default',
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Boolean]$TopMost = $true,

		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
    )

    begin {
        ## Get the name of this function and write header
        [String]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
    }

    process {
		if (Test-SkipCommand -Context $Context) { return; }

		$pdParameterNames = @( "Context" );
		$adtParameters = @{};
		$PSBoundParameters.GetEnumerator() | where { $pdParameterNames -notcontains $_.Key } | % { $adtParameters[$_.Key] = $_.Value; }

		Show-InstallationProgress @adtParameters;
	}

	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Close-ADTInstallationProgress {
	<#
	.SYNOPSIS
		Closes the dialog created by Show-InstallationProgress.
	.DESCRIPTION
		Closes the dialog created by Show-InstallationProgress.
		This function is called by the Exit-Script function to close a running instance of the progress dialog if found.
	.PARAMETER WaitingTime
		How many seconds to wait, at most, for the InstallationProgress window to be initialized, before the function returns, without closing anything. Range: 1 - 60  Default: 5
	.INPUTS
		None
		You cannot pipe objects to this function.
	.OUTPUTS
		System.String
		Returns the version of the specified file.
	.EXAMPLE
		Close-InstallationProgress
	.NOTES
		This is an internal script function and should typically not be called directly.
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Close-ADTInstallationProgress.html
	#>
    [CmdletBinding()]
    param (
        [Parameter(Mandatory = $false)][ValidateRange(1, 60)][Int32]$WaitingTime = 5,

		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
    )

    begin {
        ## Get the name of this function and write header
        [String]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
    }

    process {
		if (Test-SkipCommand -Context $Context) { return; }
		
		$pdParameterNames = @( "Context" );
		$adtParameters = @{};
		$PSBoundParameters.GetEnumerator() | where { $pdParameterNames -notcontains $_.Key } | % { $adtParameters[$_.Key] = $_.Value; }

		Close-InstallationProgress @adtParameters;
	}

	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

function Show-ADTInstallationPrompt {
    <#
	.SYNOPSIS
		Displays a custom installation prompt with the toolkit branding and optional buttons.
	.DESCRIPTION
		Any combination of Left, Middle or Right buttons can be displayed. The return value of the button clicked by the user is the button text specified.
	.PARAMETER Title
		Title of the prompt. Default: the application installation name.
	.PARAMETER Message
		Message text to be included in the prompt
	.PARAMETER MessageAlignment
		Alignment of the message text. Options: Left, Center, Right. Default: Center.
	.PARAMETER ButtonLeftText
		Show a button on the left of the prompt with the specified text
	.PARAMETER ButtonRightText
		Show a button on the right of the prompt with the specified text
	.PARAMETER ButtonMiddleText
		Show a button in the middle of the prompt with the specified text
	.PARAMETER Icon
		Show a system icon in the prompt. Options: Application, Asterisk, Error, Exclamation, Hand, Information, None, Question, Shield, Warning, WinLogo. Default: None
	.PARAMETER NoWait
		Specifies whether to show the prompt asynchronously (i.e. allow the script to continue without waiting for a response). Default: $false.
	.PARAMETER PersistPrompt
		Specify whether to make the prompt persist in the center of the screen every couple of seconds, specified in the AppDeployToolkitConfig.xml. The user will have no option but to respond to the prompt - resistance is futile!
	.PARAMETER MinimizeWindows
		Specifies whether to minimize other windows when displaying prompt. Default: $false.
	.PARAMETER Timeout
		Specifies the time period in seconds after which the prompt should timeout. Default: UI timeout value set in the config XML file.
	.PARAMETER ExitOnTimeout
		Specifies whether to exit the script if the UI times out. Default: $true.
	.PARAMETER TopMost
		Specifies whether the progress window should be topmost. Default: $true.
	.INPUTS
		None
		You cannot pipe objects to this function.
	.OUTPUTS
		None
		This function does not generate any output.
	.EXAMPLE
		Show-ADTInstallationPrompt -Message 'Do you want to proceed with the installation?' -ButtonRightText 'Yes' -ButtonLeftText 'No'
	.EXAMPLE
		Show-ADTInstallationPrompt -Title 'Funny Prompt' -Message 'How are you feeling today?' -ButtonRightText 'Good' -ButtonLeftText 'Bad' -ButtonMiddleText 'Indifferent'
	.EXAMPLE
		Show-ADTInstallationPrompt -Message 'You can customize text to appear at the end of an install, or remove it completely for unattended installations.' -Icon Information -NoWait
	.NOTES
	.LINK
		https://webhelp.nwc-services.de/ppb/latest/en-US/Show-ADTInstallationPrompt.html
	#>
    [CmdletBinding()]
    param (
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][String]$Title = $installTitle,
        [Parameter(Mandatory = $false)][String]$Message = '',
        [Parameter(Mandatory = $false)][ValidateSet('Left', 'Center', 'Right')][String]$MessageAlignment = 'Center',
        [Parameter(Mandatory = $false)][String]$ButtonRightText = '',
        [Parameter(Mandatory = $false)][String]$ButtonLeftText = '',
        [Parameter(Mandatory = $false)][String]$ButtonMiddleText = '',
        [Parameter(Mandatory = $false)][ValidateSet('Application', 'Asterisk', 'Error', 'Exclamation', 'Hand', 'Information', 'None', 'Question', 'Shield', 'Warning', 'WinLogo')][String]$Icon = 'None',
        [Parameter(Mandatory = $false)][Switch]$NoWait = $false,
        [Parameter(Mandatory = $false)][Switch]$PersistPrompt = $false,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Boolean]$MinimizeWindows = $false,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Int32]$Timeout = $configInstallationUITimeout,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Boolean]$ExitOnTimeout = $false,
        [Parameter(Mandatory = $false)][ValidateNotNullorEmpty()][Boolean]$TopMost = $true,
		
		[Parameter(Mandatory = $false)][String]$ResultVariable = '',
		[Parameter(Mandatory = $false)][ValidateSet("Any", "User", "UserPerService", "Computer", "ComputerPerService")][string]$Context = $null
    )

    begin {
        ## Get the name of this function and write header
        [String]${CmdletName} = $PSCmdlet.MyInvocation.MyCommand.Name
        Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -CmdletBoundParameters $PSBoundParameters -Header
    }

    process {
		if (Test-SkipCommand -Context $Context) { return; }

		$pdParameterNames = @( "Context", "ResultVariable" );
		$adtParameters = @{};
		$PSBoundParameters.GetEnumerator() | where { $pdParameterNames -notcontains $_.Key } | % { $adtParameters[$_.Key] = $_.Value; }
     
		# $result = Show-InstallationPrompt -Title $Title -Message $Message -MessageAlignment $MessageAlignment -ButtonRightText $ButtonRightText -ButtonMiddleText $ButtonMiddleText -ButtonLeftText $ButtonLeftText -Icon $Icon -NoWait:$NoWait -PersistPrompt:$PersistPrompt -MinimizeWindows $MinimizeWindows -Timeout $Timeout -ExitOnTimeout $ExitOnTimeout -TopMost $TopMost
		$result = Show-InstallationPrompt @adtParameters;

		Write-Log "Show-InstallationPrompt returned: $($result)." -Source ${CmdletName};

		if (![string]::IsNullOrEmpty($ResultVariable)) { Set-PdVar -Name $ResultVariable -Value $result };
	}

	end {
		Write-FunctionHeaderOrFooter -CmdletName ${CmdletName} -Footer
	}
}

#endregion

#region workaround function Resolve-Error in AppDeployToolkitMain.ps1

# Resolve-Error of PSADT 3.8.4 does not return meaningful error information
# using Resolve-Error of PSADT 3.8.3 as workaround

function Resolve-PdError {
	[CmdletBinding()]
	Param (
		[Parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true,ValueFromPipelineByPropertyName=$true)]
		[AllowEmptyCollection()]
		[array]$ErrorRecord,
		[Parameter(Mandatory=$false,Position=1)]
		[ValidateNotNullorEmpty()]
		[string[]]$Property = ('Message','InnerException','FullyQualifiedErrorId','ScriptStackTrace','PositionMessage'),
		[Parameter(Mandatory=$false,Position=2)]
		[switch]$GetErrorRecord = $true,
		[Parameter(Mandatory=$false,Position=3)]
		[switch]$GetErrorInvocation = $true,
		[Parameter(Mandatory=$false,Position=4)]
		[switch]$GetErrorException = $true,
		[Parameter(Mandatory=$false,Position=5)]
		[switch]$GetErrorInnerException = $true
	)

	Begin {
		## If function was called without specifying an error record, then choose the latest error that occurred
		If (-not $ErrorRecord) {
			If ($global:Error.Count -eq 0) {
				#Write-Warning -Message "The `$Error collection is empty"
				Return
			}
			Else {
				[array]$ErrorRecord = $global:Error[0]
			}
		}

		## Allows selecting and filtering the properties on the error object if they exist
		[scriptblock]$SelectProperty = {
			Param (
				[Parameter(Mandatory=$true)]
				[ValidateNotNullorEmpty()]
				$InputObject,
				[Parameter(Mandatory=$true)]
				[ValidateNotNullorEmpty()]
				[string[]]$Property
			)

			[string[]]$ObjectProperty = $InputObject | Get-Member -MemberType '*Property' | Select-Object -ExpandProperty 'Name'
			ForEach ($Prop in $Property) {
				If ($Prop -eq '*') {
					[string[]]$PropertySelection = $ObjectProperty
					Break
				}
				ElseIf ($ObjectProperty -contains $Prop) {
					[string[]]$PropertySelection += $Prop
				}
			}
			Write-Output -InputObject $PropertySelection
		}

		#  Initialize variables to avoid error if 'Set-StrictMode' is set
		$LogErrorRecordMsg = $null
		$LogErrorInvocationMsg = $null
		$LogErrorExceptionMsg = $null
		$LogErrorMessageTmp = $null
		$LogInnerMessage = $null
	}
	Process {
		If (-not $ErrorRecord) { Return }
		ForEach ($ErrRecord in $ErrorRecord) {
			## Capture Error Record
			If ($GetErrorRecord) {
				[string[]]$SelectedProperties = & $SelectProperty -InputObject $ErrRecord -Property $Property
				$LogErrorRecordMsg = $ErrRecord | Select-Object -Property $SelectedProperties
			}

			## Error Invocation Information
			If ($GetErrorInvocation) {
				If ($ErrRecord.InvocationInfo) {
					[string[]]$SelectedProperties = & $SelectProperty -InputObject $ErrRecord.InvocationInfo -Property $Property
					$LogErrorInvocationMsg = $ErrRecord.InvocationInfo | Select-Object -Property $SelectedProperties
				}
			}

			## Capture Error Exception
			If ($GetErrorException) {
				If ($ErrRecord.Exception) {
					[string[]]$SelectedProperties = & $SelectProperty -InputObject $ErrRecord.Exception -Property $Property
					$LogErrorExceptionMsg = $ErrRecord.Exception | Select-Object -Property $SelectedProperties
				}
			}

			## Display properties in the correct order
			If ($Property -eq '*') {
				#  If all properties were chosen for display, then arrange them in the order the error object displays them by default.
				If ($LogErrorRecordMsg) { [array]$LogErrorMessageTmp += $LogErrorRecordMsg }
				If ($LogErrorInvocationMsg) { [array]$LogErrorMessageTmp += $LogErrorInvocationMsg }
				If ($LogErrorExceptionMsg) { [array]$LogErrorMessageTmp += $LogErrorExceptionMsg }
			}
			Else {
				#  Display selected properties in our custom order
				If ($LogErrorExceptionMsg) { [array]$LogErrorMessageTmp += $LogErrorExceptionMsg }
				If ($LogErrorRecordMsg) { [array]$LogErrorMessageTmp += $LogErrorRecordMsg }
				If ($LogErrorInvocationMsg) { [array]$LogErrorMessageTmp += $LogErrorInvocationMsg }
			}

			If ($LogErrorMessageTmp) {
				$LogErrorMessage = 'Error Record:'
				$LogErrorMessage += "`n-------------"
				$LogErrorMsg = $LogErrorMessageTmp | Format-List | Out-String
				$LogErrorMessage += $LogErrorMsg
			}

			## Capture Error Inner Exception(s)
			If ($GetErrorInnerException) {
				If ($ErrRecord.Exception -and $ErrRecord.Exception.InnerException) {
					$LogInnerMessage = 'Error Inner Exception(s):'
					$LogInnerMessage += "`n-------------------------"

					$ErrorInnerException = $ErrRecord.Exception.InnerException
					$Count = 0

					While ($ErrorInnerException) {
						[string]$InnerExceptionSeperator = '~' * 40

						[string[]]$SelectedProperties = & $SelectProperty -InputObject $ErrorInnerException -Property $Property
						$LogErrorInnerExceptionMsg = $ErrorInnerException | Select-Object -Property $SelectedProperties | Format-List | Out-String

						If ($Count -gt 0) { $LogInnerMessage += $InnerExceptionSeperator }
						$LogInnerMessage += $LogErrorInnerExceptionMsg

						$Count++
						$ErrorInnerException = $ErrorInnerException.InnerException
					}
				}
			}

			If ($LogErrorMessage) { $Output = $LogErrorMessage }
			If ($LogInnerMessage) { $Output += $LogInnerMessage }

			Write-Output -InputObject $Output

			If (Test-Path -LiteralPath 'variable:Output') { Clear-Variable -Name 'Output' }
			If (Test-Path -LiteralPath 'variable:LogErrorMessage') { Clear-Variable -Name 'LogErrorMessage' }
			If (Test-Path -LiteralPath 'variable:LogInnerMessage') { Clear-Variable -Name 'LogInnerMessage' }
			If (Test-Path -LiteralPath 'variable:LogErrorMessageTmp') { Clear-Variable -Name 'LogErrorMessageTmp' }
		}
	}
	End {
	}
}

if ($appDeployMainScriptVersion -eq "3.8.4")
{
	Write-Log "Using Resolve-Error 3.8.4 workaround.";
	New-Alias -Force -Name Resolve-Error -Value Resolve-PdError;
	Export-ModuleMember -Alias Resolve-Error -Function Resolve-PdError;
}


#endregion

## Automatic refrence variable '.\' contains the path to the package directory
${script:.\} = "$($(if (![string]::IsNullOrEmpty($CurrentPackageDirectory)) { $CurrentPackageDirectory } else { $scriptDirectory }).TrimEnd('\/'))\";
Export-ModuleMember -Variable ".\"

function Export-CustomVariable([string]$Path)
{
	$variables = @{};

	try
	{
		$variablesDir = Expand-Path $Path;
		$variablesScripts = $(if ([System.IO.Directory]::Exists($variablesDir)) { [System.IO.Directory]::GetFiles($variablesDir, "*.Variables.ps1") + [System.IO.Directory]::GetFiles($variablesDir, "*.Variables.psd1") } else { @() });
		Write-Log -Message "Export variable files in '$($variablesDir)' - scripts: $($variablesScripts.Count)." -Source $moduleName;
		foreach ($variablesScript in $variablesScripts)
		{
			try
			{
				$result = @();
				if ([System.IO.Path]::GetExtension($variablesScript) -eq ".ps1") { $result = & $variablesScript; }
				elseif ([System.IO.Path]::GetExtension($variablesScript) -eq ".psd1") { $result = Import-LocalizedData -FileName ([System.IO.Path]::GetFileName($variablesScript)) -BaseDirectory ([System.IO.Path]::GetDirectoryName($variablesScript)); }

				$items = @($result | % { if ($_ -is [psvariable]) { $_ } elseif ($_-is [hashtable]) { $_.GetEnumerator() | % { New-Object PSObject -Property @{ Name = $_.Key; Value = $_.Value; } } } });

				foreach ($item in $items)
				{
					$parameters = @{ Name = $item.Name; Value = $item.Value; }
					if (![string]::IsNullOrEmpty($varible.Description)) { $parameters.Description = $item.Description; }
					if ($item.Option -ne $null) { $parameters.Option = $item.Option; }
					$variables[$item.Name] = New-Variable -PassThru -Force -Scope Script -Visibility Public @parameters;
				}

				Write-Log -Message "Items from '$($variablesScript)': $($items.Count)." -Source $moduleName;
			}
			catch
			{
				$failed = "Failed to read export variables from '$($variablesScript)'";
				Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 2 -Source $moduleName;
			}
		}
	}
	catch
	{
		$failed = "Failed to initialize export variables";
		Write-Log -Message "$($failed).`r`n$(Resolve-Error)" -Severity 2 -Source $moduleName;
	}

	Write-Log -Message "Initialized export variables: $($variables.Count)." -Source $moduleName;
	Export-ModuleMember -Variable @($variables.GetEnumerator() | % { $_.Value.Name });
}

## export variables in .\Variables subfolder
Export-CustomVariable -Path ([System.IO.Path]::Combine($scriptDirectory, "Variables"));

## Exports from AppDeployToolkitMain.ps1
Export-ModuleMember -Function Write-Log, Exit-Script, Show-DialogBox, Resolve-Error
Export-ModuleMember -Variable InstallPhase, InstallName, AppDeployToolkitName, DeploymentType, LogName, AppName, DisableLogging, configShowBalloonNotifications
# Export-ModuleMember -Function Exit-Script, Show-DialogBox, Resolve-Error

## Exports from AppDeployToolkitMain.ps1 defined in $InitialContext
Export-ModuleMember -Function $AdtExportFunctions -Variable $AdtExportVariables

## Exports from AppDeployToolkitMain.ps1 as Commands
## Export-ModuleMember -Function ...

Export-ModuleMember -Function Get-PdContext, Set-BeginUninstallScript, Add-DeleteAtEndOfScript, Invoke-DeleteAtEndOfScript, Clear-DeleteAtEndOfScript
Export-ModuleMember -Function Set-PdVar, Install-FileList, Install-File, Copy-File, Remove-FileList, Remove-File, Install-Assembly, Uninstall-GacAssembly, New-Directory, Remove-Directory
Export-ModuleMember -Function Install-Win32Service, Start-Win32Service, Stop-Win32Service, Uninstall-Win32Service, Start-Program, Start-ProgramAs, Test-InstallMode
Export-ModuleMember -Function Copy-RegistryKey, Remove-RegistryKey, Mount-Registry, Dismount-Registry, Import-Registry, Save-Registry, Read-RegistryValue, Write-RegistryValue, Write-RegistryDWord, Write-RegistryMultiString, Write-RegistryQWord, Test-RegistryKey, Test-RegistryValue, Get-RegistryValue
Export-ModuleMember -Function Exit-Package, Stop-PdProcess, Install-PnpDevice, Install-MsiProduct, Test-MsiProduct, Uninstall-MsiProduct, Install-MsiPatch, Repair-MsiProduct, Install-MsiFeature
Export-ModuleMember -Function Add-PrinterConnection, Remove-PrinterConnection, New-Link, Remove-Link, New-ShellFolder, Remove-ShellFolder, Merge-IniFile, Read-IniFileValue, Write-IniFileValue, Get-DriveFreeKB
Export-ModuleMember -Function Test-Platform, Test-FileExists, Get-FileDate, Get-FileVersion, Get-ProductVersion, Test-SystemRestart, Unregister-SystemRestart, Register-SystemRestart, Register-SystemShutdown, Start-SystemShutdown, Stop-SystemShutdown
Export-ModuleMember -Function Suspend-ScriptExecution, Set-PdVarIncrement, Set-PdVarDecrement, Test-RunningOnX64, Test-RunningOnServerOS
Export-ModuleMember -Function Write-InstalledAppsRegistry, Test-PackageInstalled, Get-InstalledAppsRegistry, Remove-InstalledAppsRegistry, Test-PackageRevisionInstalled
Export-ModuleMember -Function Set-Label, Invoke-Goto, Test-ProcessRunning, Write-RegistryKey, Write-ActiveSetupRegistry, Remove-ActiveSetupRegistry, Invoke-ActiveSetupUserPart
Export-ModuleMember -Function Read-StringElement, Read-LeftString, Read-MidString, Read-ReplacePattern, Read-ReplaceString, Read-RightString, Add-EnvironmentPath, Edit-OemLine, Edit-OemText, Add-OemEnvironmentPath
Export-ModuleMember -Function Invoke-Script, Invoke-ScriptAs, Read-FileVersion, Test-FileInUse, Test-ComputerName, Read-WmiObject, Read-WmiObjectCount, Read-IndexedWmiObject
Export-ModuleMember -Function Add-LocalGroupMember, New-LocalGroup, Remove-LocalGroupMember, Add-LocalUserMembership, New-LocalUser, Rename-LocalUser, Set-LocalUserFlags, Set-LocalUserPassword, Set-LocalUserProfile
Export-ModuleMember -Function Test-PendingReboot, Search-RegistryKey, Send-SmtpMail, Test-Line, Set-NtfsSecurity, Set-XmlNode, Read-XmlNode, Test-XmlNode, Request-EndInstallerSession, New-InternetLink
Export-ModuleMember -Function New-PackageCacheEntry, Remove-PackageCacheEntry, Set-ApplicationCompatibilityFlags, Show-PdMessageBox, Test-LocalGroup, Remove-LocalGroup, Test-LocalUser, Remove-LocalUser, Show-MultipleChoiceDialog, Read-SystemDefaultLanguage, Read-UserDefaultLanguage, Set-FileReplaceText, Set-FileAttribute
Export-ModuleMember -Function Get-WindowsVersion, Install-TTF, Uninstall-TTF, New-Share, Remove-Share, Test-Laptop, Test-Service, Test-OnBattery, Test-InternetAccess, Test-LocalAdmin, Test-RunningAsService
Export-ModuleMember -Function Test-VirtualMachine, Test-MicrosoftUpdate, Test-UserLoggedOn, Test-ComputerLocked, Test-RemoteLoginDisabled, Test-WindowsFeature, Enable-WindowsFeature, Disable-WindowsFeature, Invoke-PowerShellCode, Suspend-PdBitlocker
Export-ModuleMember -Function Connect-SmbShare, Disconnect-SmbShare, Set-RegistrySecurity, Show-PdToastMessage, Expand-PdArchive, Test-UserGroupMembership, Test-WorkstationGroupMembership
Export-ModuleMember -Function Show-ADTInstallationWelcome, Show-ADTBalloonTip, Show-ADTInstallationRestartPrompt, Show-ADTInstallationProgress, Close-ADTInstallationProgress, Show-ADTInstallationPrompt, Read-ADTInstalledApplication, Get-ADTInstalledApplicationProperty

function Write-DsmLog
{
	$writeLog = Get-Command -Name Write-Log -CommandType Function;
	# & $writeLog -Message "HUHU...($([string]::join(' ', $args)))"
	& $writeLog @args
}
# Write-Log -Message "BEGIN: SETTING ALIAS Write-Log to Write-DsmLog"
# New-Alias -Name Write-Log -Value Write-DsmLog
# Write-Log -Message "DONE: SETTING ALIAS Write-Log to Write-DsmLog"
# Export-ModuleMember -Alias Write-Log


# SIG # Begin signature block
# MIIrvQYJKoZIhvcNAQcCoIIrrjCCK6oCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCa9//QMCDwm9K5
# o1oMSgKMGdiLFaS8WYCuqN9DO1xDd6CCJNMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYUMIID/KADAgECAhB6I67a
# U2mWD5HIPlz0x+M/MA0GCSqGSIb3DQEBDAUAMFcxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIFRp
# bWUgU3RhbXBpbmcgUm9vdCBSNDYwHhcNMjEwMzIyMDAwMDAwWhcNMzYwMzIxMjM1
# OTU5WjBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSww
# KgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNjCCAaIw
# DQYJKoZIhvcNAQEBBQADggGPADCCAYoCggGBAM2Y2ENBq26CK+z2M34mNOSJjNPv
# IhKAVD7vJq+MDoGD46IiM+b83+3ecLvBhStSVjeYXIjfa3ajoW3cS3ElcJzkyZlB
# nwDEJuHlzpbN4kMH2qRBVrjrGJgSlzzUqcGQBaCxpectRGhhnOSwcjPMI3G0hedv
# 2eNmGiUbD12OeORN0ADzdpsQ4dDi6M4YhoGE9cbY11XxM2AVZn0GiOUC9+XE0wI7
# CQKfOUfigLDn7i/WeyxZ43XLj5GVo7LDBExSLnh+va8WxTlA+uBvq1KO8RSHUQLg
# zb1gbL9Ihgzxmkdp2ZWNuLc+XyEmJNbD2OIIq/fWlwBp6KNL19zpHsODLIsgZ+WZ
# 1AzCs1HEK6VWrxmnKyJJg2Lv23DlEdZlQSGdF+z+Gyn9/CRezKe7WNyxRf4e4bwU
# trYE2F5Q+05yDD68clwnweckKtxRaF0VzN/w76kOLIaFVhf5sMM/caEZLtOYqYad
# tn034ykSFaZuIBU9uCSrKRKTPJhWvXk4CllgrwIDAQABo4IBXDCCAVgwHwYDVR0j
# BBgwFoAU9ndq3T/9ARP/FqFsggIv0Ao9FCUwHQYDVR0OBBYEFF9Y7UwxeqJhQo1S
# gLqzYZcZojKbMA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEAMBMG
# A1UdJQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgwBgYEVR0gADBMBgNVHR8ERTBD
# MEGgP6A9hjtodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1l
# U3RhbXBpbmdSb290UjQ2LmNybDB8BggrBgEFBQcBAQRwMG4wRwYIKwYBBQUHMAKG
# O2h0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGlu
# Z1Jvb3RSNDYucDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNv
# bTANBgkqhkiG9w0BAQwFAAOCAgEAEtd7IK0ONVgMnoEdJVj9TC1ndK/HYiYh9lVU
# acahRoZ2W2hfiEOyQExnHk1jkvpIJzAMxmEc6ZvIyHI5UkPCbXKspioYMdbOnBWQ
# Un733qMooBfIghpR/klUqNxx6/fDXqY0hSU1OSkkSivt51UlmJElUICZYBodzD3M
# /SFjeCP59anwxs6hwj1mfvzG+b1coYGnqsSz2wSKr+nDO+Db8qNcTbJZRAiSazr7
# KyUJGo1c+MScGfG5QHV+bps8BX5Oyv9Ct36Y4Il6ajTqV2ifikkVtB3RNBUgwu/m
# SiSUice/Jp/q8BMk/gN8+0rNIE+QqU63JoVMCMPY2752LmESsRVVoypJVt8/N3qQ
# 1c6FibbcRabo3azZkcIdWGVSAdoLgAIxEKBeNh9AQO1gQrnh1TA8ldXuJzPSuALO
# z1Ujb0PCyNVkWk7hkhVHfcvBfI8NtgWQupiaAeNHe0pWSGH2opXZYKYG4Lbukg7H
# pNi/KqJhue2Keak6qH9A8CeEOB7Eob0Zf+fU+CCQaL0cJqlmnx9HCDxF+3BLbUuf
# rV64EbTI40zqegPZdA+sXCmbcZy6okx/SjwsusWRItFA3DE8MORZeFb6BmzBtqKJ
# 7l939bbKBy2jvxcJI98Va95Q5JnlKor3m0E7xpMeYRriWklUPsetMSf2NvUQa/E5
# vVyefQIwggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0GCSqGSIb3DQEB
# DAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLTAr
# BgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290IFI0NjAeFw0y
# MTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIB
# gQCbK51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEeAEZGbEN4QMgC
# sJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGVoYW3haT29PST
# ahYkwmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk8XL/tfCK6cPu
# YHE215wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzhP06ShdnRqtZl
# V59+8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41aHU5pledi9lC
# BbH9JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+ac7IH60B+Ja7
# TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/2yGgW+ufflcZ
# /ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9gRvd+QOfdRrJZ
# b1sCAwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4Xm
# MB0GA1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8BAf8EBAMCAYYw
# EgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcDAzAbBgNVHSAE
# FDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyGOmh0dHA6Ly9j
# cmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5nUm9vdFI0Ni5j
# cmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8vY3J0LnNlY3Rp
# Z28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYucDdjMCMGCCsG
# AQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOC
# AgEABv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW4HL9hcpkOTz5
# jUug2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIhrCymlaS98+Qp
# oBCyKppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYpmlG+vd0xqlqd
# 099iChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7DcML4iTAWS+MVX
# eNLj1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yELg09Jlo8BMe8
# 0jO37PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRzNyE6bxuSKcut
# isqmKL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw4rg3sTCggkHS
# RqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6E52z1NZJ6ctu
# MFBQZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6xWls/GDnVNue
# KjWUH3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZtg0y9ShQoPzmC
# cn63Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2RwwggY/MIIEp6ADAgEC
# AhA2nIGBal4DSIUR8kAAOO1LMA0GCSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdC
# MRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVi
# bGljIENvZGUgU2lnbmluZyBDQSBSMzYwHhcNMjMxMjAxMDAwMDAwWhcNMjYxMTMw
# MjM1OTU5WjBWMQswCQYDVQQGEwJERTEbMBkGA1UECAwSQmFkZW4tV8O8cnR0ZW1i
# ZXJnMRQwEgYDVQQKDAtDQU5DT00gR21iSDEUMBIGA1UEAwwLQ0FOQ09NIEdtYkgw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDArSw9LTzTaw9/KvfNlE0S
# Fvf9qcOQ/lXAPYkhX3xVx2L1/MrBosbFLHvJJ6iWD2z4717VRgN/f4woMx1ZZ+k5
# so33kL6rTKliDv1C91rd1aI4NGxwVMg0wOLKA0UDN0oR8/WgKES3QBJYq//zi3+V
# vARTml1dpOcQje3dXPcRY1m8zo/ObmB8pPr8pQAYTgmZuYtj+IZe1G96HnEMaglZ
# T7Ko9CiuCT6MoUuB7P2w+JMpRRUSW7sEk1eex3+g6iL/NcthpF3YqWCgWwNHys0b
# oYNaeQuH2BC4o4Lv2yXdR8nK1SuVaYelCd4b5hICwL3as6+z8Y2J0HEY3focyhtZ
# sHtahMk8STDcjvgWeap79Qv5jktC7pPdKxv1aI/U0d0V/vzQv+aJ0ySIrHbGMrwG
# MlqSRULednSIxm0YqUnay3C+w+9g+cF9A3TcAGHJwiBZrAvYexcMz4yC/ZpEIVpd
# nfzJZQkzqUO0AuBg0qoXbTGb67ZZwHoJKiZjLXRYuWK3Swzei0wj1VWqvmMgEVC9
# zAI3pC0kYXq9OUvFhEYyQ5iCgbluVK1Q60BwsuP1L1z81PH+0gfPpIln1cg35Dp5
# xL6O+wOC1J5GmlqfAzbwVYblBjAo8J0oZgBGHA1akmFBEINDTHxdDgNh+f/2Jop9
# z/gm4uXWBzSSRVw6TbCfaQIDAQABo4IBiTCCAYUwHwYDVR0jBBgwFoAUDyrLIIco
# uOxvSK4rVKYpqhekzQwwHQYDVR0OBBYEFDtSj69rtLzxAlC2x8F+gn+1H8w/MA4G
# A1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
# MEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUwIwYIKwYBBQUHAgEWF2h0dHBz
# Oi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBJBgNVHR8EQjBAMD6gPKA6hjho
# dHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NB
# UjM2LmNybDB5BggrBgEFBQcBAQRtMGswRAYIKwYBBQUHMAKGOGh0dHA6Ly9jcnQu
# c2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5nQ0FSMzYuY3J0MCMG
# CCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwF
# AAOCAYEAJjsMqKSeh1xy940zL3vIr3V1dl2f7Qp49GXD+7pXXfHABvw/y7AZFLCT
# PoHmkV71A3VBbQ9xc2Avzd1lQOXolYC/SuO7CWPPLpcK7j15YxurHKhxufC50f6E
# Cmg9ziRMtKIvbjJRX5FDXFfHsEcNJ+3gX2jgaWxB0UgdqE/p4ionURoMze8yHHrQ
# F59br2lAFwr4IQzVjDsh/2Kki8QugddiA506lbDHiVaxKi29hVQ/EXXzWpUzJf8r
# gCaYjKSEXAYq264QDPnEdvq7oTjXG/25VCodN/H7EpHaHv10kSvUIL8QV4ZUKPab
# 631VFavfLaKHgQhYjzWdXe71qcvuLdaSObTbzWlwK9/l9G3cj5HYpnyaAba1nhIe
# MiWiiAc0tGEfS9DVfnbdr1zn3U/PiAVbsoaXx4RhlvIBnMOke2HIOcpVgWraHDN1
# j73vyOagdJBbD1alLFuFVoVhMDQ6OHQcSUBt/xuZ/SUsauguKfdmlPfYwGsdA/f1
# mAks8ZrRMIIGXTCCBMWgAwIBAgIQOlJqLITOVeYdZfzMEtjpiTANBgkqhkiG9w0B
# AQwFADBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSww
# KgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNjAeFw0y
# NDAxMTUwMDAwMDBaFw0zNTA0MTQyMzU5NTlaMG4xCzAJBgNVBAYTAkdCMRMwEQYD
# VQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxMDAuBgNV
# BAMTJ1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgU2lnbmVyIFIzNTCCAiIw
# DQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAI3RZ/TBSJu9/ThJOk1hgZvD2NxF
# pWEENo0GnuOYloD11BlbmKCGtcY0xiMrsN7LlEgcyoshtP3P2J/vneZhuiMmspY7
# hk/Q3l0FPZPBllo9vwT6GpoNnxXLZz7HU2ITBsTNOs9fhbdAWr/Mm8MNtYov32os
# vjYYlDNfefnBajrQqSV8Wf5ZvbaY5lZhKqQJUaXxpi4TXZKohLgxU7g9RrFd477j
# 7jxilCU2ptz+d1OCzNFAsXgyPEM+NEMPUz2q+ktNlxMZXPF9WLIhOhE3E8/oNSJk
# NTqhcBGsbDI/1qCU9fBhuSojZ0u5/1+IjMG6AINyI6XLxM8OAGQmaMB8gs2IZxUT
# OD7jTFR2HE1xoL7qvSO4+JHtvNceHu//dGeVm5Pdkay3Et+YTt9EwAXBsd0PPmC0
# cuqNJNcOI0XnwjE+2+Zk8bauVz5ir7YHz7mlj5Bmf7W8SJ8jQwO2IDoHHFC46ePg
# +eoNors0QrC0PWnOgDeMkW6gmLBtq3CEOSDU8iNicwNsNb7ABz0W1E3qlSw7jTmN
# oGCKCgVkLD2FaMs2qAVVOjuUxvmtWMn1pIFVUvZ1yrPIVbYt1aTld2nrmh544Auh
# 3tgggy/WluoLXlHtAJgvFwrVsKXj8ekFt0TmaPL0lHvQEe5jHbufhc05lvCtdwbf
# Bl/2ARSTuy1s8CgFAgMBAAGjggGOMIIBijAfBgNVHSMEGDAWgBRfWO1MMXqiYUKN
# UoC6s2GXGaIymzAdBgNVHQ4EFgQUaO+kMklptlI4HepDOSz0FGqeDIUwDgYDVR0P
# AQH/BAQDAgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgw
# SgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEFBQcCARYXaHR0cHM6
# Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEoGA1UdHwRDMEEwP6A9oDuGOWh0
# dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGluZ0NB
# UjM2LmNybDB6BggrBgEFBQcBAQRuMGwwRQYIKwYBBQUHMAKGOWh0dHA6Ly9jcnQu
# c2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGluZ0NBUjM2LmNydDAj
# BggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcNAQEM
# BQADggGBALDcLsn6TzZMii/2yU/V7xhPH58Oxr/+EnrZjpIyvYTz2u/zbL+fzB7l
# brPml8ERajOVbudan6x08J1RMXD9hByq+yEfpv1G+z2pmnln5XucfA9MfzLMrCAr
# NNMbUjVcRcsAr18eeZeloN5V4jwrovDeLOdZl0tB7fOX5F6N2rmXaNTuJR8yS2F+
# EWaL5VVg+RH8FelXtRvVDLJZ5uqSNIckdGa/eUFhtDKTTz9LtOUh46v2JD5Q3nt8
# mDhAjTKp2fo/KJ6FLWdKAvApGzjpPwDqFeJKf+kJdoBKd2zQuwzk5Wgph9uA46VY
# K8p/BTJJahKCuGdyKFIFfEfakC4NXa+vwY4IRp49lzQPLo7WticqMaaqb8hE2QmC
# FIyLOvWIg4837bd+60FcCGbHwmL/g1ObIf0rRS9ceK4DY9rfBnHFH2v1d4hRVvZX
# yCVlrL7ZQuVzjjkLMK9VJlXTVkHpuC8K5S4HHTv2AJx6mOdkMJwS4gLlJ7gXrIVp
# nxG+aIniGDCCBoIwggRqoAMCAQICEDbCsL18Gzrno7PdNsvJdWgwDQYJKoZIhvcN
# AQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYD
# VQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3Jr
# MS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5
# MB4XDTIxMDMyMjAwMDAwMFoXDTM4MDExODIzNTk1OVowVzELMAkGA1UEBhMCR0Ix
# GDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJs
# aWMgVGltZSBTdGFtcGluZyBSb290IFI0NjCCAiIwDQYJKoZIhvcNAQEBBQADggIP
# ADCCAgoCggIBAIid2LlFZ50d3ei5JoGaVFTAfEkFm8xaFQ/ZlBBEtEFAgXcUmanU
# 5HYsyAhTXiDQkiUvpVdYqZ1uYoZEMgtHES1l1Cc6HaqZzEbOOp6YiTx63ywTon43
# 4aXVydmhx7Dx4IBrAou7hNGsKioIBPy5GMN7KmgYmuu4f92sKKjbxqohUSfjk1mJ
# lAjthgF7Hjx4vvyVDQGsd5KarLW5d73E3ThobSkob2SL48LpUR/O627pDchxll+b
# TSv1gASn/hp6IuHJorEu6EopoB1CNFp/+HpTXeNARXUmdRMKbnXWflq+/g36NJXB
# 35ZvxQw6zid61qmrlD/IbKJA6COw/8lFSPQwBP1ityZdwuCysCKZ9ZjczMqbUcLF
# yq6KdOpuzVDR3ZUwxDKL1wCAxgL2Mpz7eZbrb/JWXiOcNzDpQsmwGQ6Stw8tTCqP
# umhLRPb7YkzM8/6NnWH3T9ClmcGSF22LEyJYNWCHrQqYubNeKolzqUbCqhSqmr/U
# dUeb49zYHr7ALL8bAJyPDmubNqMtuaobKASBqP84uhqcRY/pjnYd+V5/dcu9ieER
# jiRKKsxCG1t6tG9oj7liwPddXEcYGOUiWLm742st50jGwTzxbMpepmOP1mLnJskv
# ZaN5e45NuzAHteORlsSuDt5t4BBRCJL+5EZnnw0ezntk9R8QJyAkL6/bAgMBAAGj
# ggEWMIIBEjAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4E
# FgQU9ndq3T/9ARP/FqFsggIv0Ao9FCUwDgYDVR0PAQH/BAQDAgGGMA8GA1UdEwEB
# /wQFMAMBAf8wEwYDVR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAA
# MFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0LmNvbS9VU0VS
# VHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5LmNybDA1BggrBgEFBQcBAQQp
# MCcwJQYIKwYBBQUHMAGGGWh0dHA6Ly9vY3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZI
# hvcNAQEMBQADggIBAA6+ZUHtaES45aHF1BGH5Lc7JYzrftrIF5Ht2PFDxKKFOct/
# awAEWgHQMVHol9ZLSyd/pYMbaC0IZ+XBW9xhdkkmUV/KbUOiL7g98M/yzRyqUOZ1
# /IY7Ay0YbMniIibJrPcgFp73WDnRDKtVutShPSZQZAdtFwXnuiWl8eFARK3PmLqE
# m9UsVX+55DbVIz33Mbhba0HUTEYv3yJ1fwKGxPBsP/MgTECimh7eXomvMm0/GPxX
# 2uhwCcs/YLxDnBdVVlxvDjHjO1cuwbOpkiJGHmLXXVNbsdXUC2xBrq9fLrfe8IBs
# A4hopwsCj8hTuwKXJlSTrZcPRVSccP5i9U28gZ7OMzoJGlxZ5384OKm0r568Mo9T
# YrqzKeKZgFo0fj2/0iHbj55hc20jfxvK3mQi+H7xpbzxZOFGm/yVQkpo+ffv5gdh
# p+hv1GDsvJOtJinJmgGbBFZIThbqI+MHvAmMmkfb3fTxmSkop2mSJL1Y2x/955S2
# 9Gu0gSJIkc3z30vU/iXrMpWx2tS7UVfVP+5tKuzGtgkP7d/doqDrLF1u6Ci3TpjA
# ZdeLLlRQZm867eVeXED58LXd1Dk6UvaAhvmWYXoiLz4JA5gPBcz7J311uahxCweN
# xE+xxxR3kT0WKzASo5G/PyDez6NHdIUKBeE3jDPs2ACc6CkJ1Sji4PKWVT0/MYIG
# QDCCBjwCAQEwaDBUMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1p
# dGVkMSswKQYDVQQDEyJTZWN0aWdvIFB1YmxpYyBDb2RlIFNpZ25pbmcgQ0EgUjM2
# AhA2nIGBal4DSIUR8kAAOO1LMA0GCWCGSAFlAwQCAQUAoIGEMBgGCisGAQQBgjcC
# AQwxCjAIoAKAAKECgAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYB
# BAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIP64SgKqijW5
# HraTDVvT0RJHF3X9oX4OTsycbkW4Mz2pMA0GCSqGSIb3DQEBAQUABIICAIlf01wK
# YGpYoZlccdSnJ3hEQHMdiHVG4l4pRoObA3gUJXo6qPxLSuUQ4gSVf3Rx2W8BFNtt
# /ejuBb7z1ff6h6h4L6c19PAJ2SqeJpjk1f5XdaG9B/3UTyPy3zVKD8wYcO7yTtxO
# /24NZHLNZnS4yaTdodZBb8ycTW1QxlF9FS3xOlT0fv0BpuHh4BU0wHw9y9emW9T0
# 9ZSq6zWuk1Rs8pQu2nz8GePw2yfTLUbbvrkMBxhwr3pVOhxYemtiBHAWBlU/1mnq
# Xy+5BY06mnd3DDgQIu87EjwmXgfE0zpp33CvRnbcy2GLQ+vrgpfqXSaPMmz7Y3Bh
# +hbfpq4j9ZvBpTK7HzTf0sy5B2StnnYPQIwUwpqzUU30vKyb4SGNTV8cSw0XalQL
# vhmUb6xYv+m6MeYYnG18swQH1v4JEGRIsyHjKuxxS3Q5GeYmmThERIsc6cG69uCR
# HY7SC6AGvLWL6qco3e/PMn8e6aaS06MzbMHN3V+YTkkHqHViO4wC4y45aJ8++O4/
# kCZw/WZSjY5+AgE7XR8H/6x1HpqGv50MClgltKcu2gPreDg0X/HxwFZA7e2rncck
# Kf61uVs9BDK8PPlFptcKFDkBk5RAQWVGi3LS4Bt+s6cjtLjo7kV+sgflfTQgLwDf
# 9fTpb207VnnwInNViRAM0K6uBXpdYuO323GGoYIDIjCCAx4GCSqGSIb3DQEJBjGC
# Aw8wggMLAgEBMGkwVTELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEsMCoGA1UEAxMjU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBDQSBS
# MzYCEDpSaiyEzlXmHWX8zBLY6YkwDQYJYIZIAWUDBAICBQCgeTAYBgkqhkiG9w0B
# CQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0yNDEwMjIxNTMzMjJaMD8G
# CSqGSIb3DQEJBDEyBDCTlS+plv1zBfeHyvWx6ImN6nMf0uyaEK5c+TD6dETSl5Zq
# YiMplTkcSaJroK9sdN4wDQYJKoZIhvcNAQEBBQAEggIAOEGBtHtyt8DgL/3Fei9I
# SsIy3QttdpPZaK1EWtzT+nWdRfDko0vJ23zbw4iq11rw7aS6X27BmfhcEcIhuvTU
# DeiT8uxvC5pjy0lTk6FeexuwbOnVZ5+sZRXehoHdTJy/0PTWEVdgd9D1HYvVTvt7
# qDh3kniOC63Xi2g9S/EnjF4HaoKoHoUL2NGVzLy+8y1n9MSEprYdiBN/LsyaO8kt
# 4m20obJtA9hagHLW2rhv8oKHkerMSFy2+cig4vmF2bNuxQwevd7pAzbMhJF4bzCS
# CKL2zWbdAdrURmx2FdzhEXRmLIkngvpfmWE/JQc+PBCjxh3+cDoBEl98RYE9+Egq
# 3WgZgZWwujkTGWX4VzD45vCcvnlr5vWr81yErK0pr/ixAWL3WoPwi4Ygnf7rdyBk
# 693MSYIWe7FwvaHcq7gx7rcdM/tznBWKLbMBXToN0ZJZ7GHyv5BHUtT8czl4riiJ
# bO1j2plbZwZZgwQklgBEIEJerHus9dlLB8zTF/hjd+ydlFJdcCUEgGE+EIkX2L3B
# U16UHePG1g8sD1tBVAa3VnWec/MNLvF+43UzAI0mtHpxAW78f9hqnQ9oCDtMhAZJ
# rJsMlnTR4pmgH0To4c8VZd0WdzPPy84epd7wnAyEAmQo1filZCONw17SKNuk2zms
# QSqO80GcOw1O2dMrJGp9PaA=
# SIG # End signature block
